#-----------------------------------------------------
#
# Copyright (c) 2017 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------


__author__ = "vdattatr"

import logging
import libvirt
import sys
from xml.dom import minidom
log = logging.getLogger("runtime.hosting")
from resourcemanager import ResourceManager
from platformcapabilities import PlatformCapabilities


class PolarisResourceManager(ResourceManager):
    __singleton = None  # the one, true Singleton
    LXC_DRIVER = 'lxc:///'
    QEMU_DRIVER = 'qemu:///system'

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(PolarisResourceManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, cgroup_mount_path=None, cgroup_parent_name=None,
                 cgroup_create_tree = False, cgroup_constant_shares=100,
                 persistent_store=None, use_ext4=False):
        '''
        Donot sub-class this class. If you do, because of singleton code above, init
        would be called multiple times
        '''
        self._cgroup_mount_path = cgroup_mount_path
        self._cgroup_parent_name = cgroup_parent_name
        self._cgroup_create_tree = cgroup_create_tree
        self._cgroup_constant_shares = cgroup_constant_shares
        self._pc = PlatformCapabilities.getInstance()
        self._resource_profiles = None
        self._app_map = {}
        self._app_disks = {}
        self._app_rfsdisks = {}
        self._load_resource_profiles()
        if self._cgroup_create_tree:
            self._initialize_cgroup_partition()
        self._persistent_store = persistent_store
        self._use_ext4 = use_ext4
        self._pc.set_persistent_dir(self._persistent_store)
        #self._init_existing_disks(self._persistent_store)

    def check_resource_availability(self, app_type, metadata, app_resources={},override_host_mode=False, appid="", supported_features=None):
        self.update_platform_resource()
        extended_resources = False
        if app_resources.get("profile") == "exclusive" :
            if self._pc.total_cpu_units != self._pc.available_cpu_units or self._pc.total_memory != self._pc.available_memory:
                log.error("Not enough resources for exclusive profile")
                raise ResourceLimitError("Not enough resources for exclusive profile. Deactivate all other apps")
            extended_resources = app_resources.get("extended_resources", False)
            if extended_resources:
                if not self._pc.extended_resources:
                    raise ResourceLimitError("Extended resources not supported on this device")
                if not self._pc.extended_resources_available:
                    raise ResourceLimitError("Extended resources already being used.")

        self.check_runtime_resource_availability(app_resources.get('cpu'), app_resources.get('memory'), extended_resources=extended_resources)
        self._check_resource_availability(app_type,app_resources,override_host_mode, appid, supported_features)
        self._check_capability_availability(appid, metadata, app_resources)

    def update_platform_resource(self):
        total_cpu_units_libvirt = self._get_libvirt_domains_cpu(self.LXC_DRIVER) + self._get_libvirt_domains_cpu(self.QEMU_DRIVER)
        total_memory_libvirt_mb = self._get_libvirt_domains_memory(self.LXC_DRIVER) + self._get_libvirt_domains_memory(self.QEMU_DRIVER)
        self._pc.available_memory=self._pc.total_memory - total_memory_libvirt_mb
        self._pc.available_cpu_units=self._pc.total_cpu_units - total_cpu_units_libvirt

    def get_resource_allocations(self):
        self.update_platform_resource()
        return self._get_resource_allocations()

    def _get_libvirt_domains_memory(self, domain_str):
        sum = 0
        try:
            conn = libvirt.open(domain_str)
            if conn == None:
                raise ValueError('Failed to open connection', file=sys.stderr)
        except ValueError as e:
            log.error(e)
        except Exception:
            log.error('Opening Libvirt Connection failed')
        try:
            domains = conn.listAllDomains(0)
            if domains == None:
                conn.close()
                raise ValueError('Failed to get a list of domain IDs', file=sys.stderr)
        except ValueError as e:
            log.error(e)
        except Exception:
            log.error('Failed to get Libvirt domain details')
        if len(domains) != 0:
            for domain in domains:
                sum += domain.maxMemory()
        conn.close()
        log.debug('The sumtotal of memory for all %s is %s', domain_str, sum)
        return sum / 1024

    def _get_libvirt_domains_cpu(self, domain_str):
        sum = 0
        try:
            conn = libvirt.open(domain_str)
            if conn == None:
                raise ValueError('Failed to open connection', file=sys.stderr)
        except ValueError as e:
            log.error(e)
        except Exception:
            log.error('Opening Libvirt Connection failed')
        try:
            domains = conn.listAllDomains(0)
            if domains == None:
                conn.close()
                raise ValueError('Failed to get a list of domain IDs', file=sys.stderr)
        except ValueError as e:
            log.error(e)
        except Exception:
            log.error('Failed to get Libvirt domain  details')
        if len(domains) != 0:
            for domain in domains:
                domain_xml = domain.XMLDesc(0)
                domain_xml = minidom.parseString(domain_xml)
                node = domain_xml.getElementsByTagName('shares')
                sum += int(node[0].firstChild.nodeValue)
        sum = self.construct_cpu_units_from_shares(sum)
        log.debug('The sumtotal of cpu units for all %s is %s', domain_str, sum)
        conn.close()
        return sum

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = PolarisResourceManager(*args)
        return cls.__singleton
