import os
import logging
import json
from ..utils.utils import Utils

from copy import copy, deepcopy

from collections import OrderedDict
from   ..utils.infraexceptions import *

log = logging.getLogger("runtime.hosting")

class ConnectorDependencies(object):
    """ Directed acyclic graph implementation. """

    def __init__(self):
        """ Construct a new DAG with no nodes or edges. """
        self.reset_graph()

    def add_node(self, node_name, graph=None):
        """ Add a node if it does not exist yet, or error out. """
        if not graph:
            graph = self._graph
        if node_name in graph:
            raise KeyError('node %s already exists' % node_name)
        graph[node_name] = set()

    def add_node_if_not_exists(self, node_name, graph=None):
        try:
            self.add_node(node_name, graph=graph)
        except KeyError:
            pass

    def delete_node(self, node_name, graph=None):
        """ Deletes this node and all edges referencing it. """
        if not graph:
            graph = self._graph
        if node_name not in graph:
            raise KeyError('node %s does not exist' % node_name)
        graph.pop(node_name)

        for node, edges in graph.iteritems():
            if node_name in edges:
                edges.remove(node_name)

    def delete_node_if_exists(self, node_name, graph=None):
        try:
            self.delete_node(node_name, graph=graph)
        except KeyError:
            pass

    def add_edge(self, ind_node, dep_node, graph=None):
        """ Add an edge (dependency) between the specified nodes. """
        if not graph:
            graph = self._graph
        if ind_node not in graph or dep_node not in graph:
            raise KeyError('one or more nodes do not exist in graph')
        test_graph = deepcopy(graph)
        test_graph[ind_node].add(dep_node)
        is_valid, message = self.validate(test_graph)
        if is_valid:
            graph[ind_node].add(dep_node)
        else:
            raise DAGValidationError()

    def delete_edge(self, ind_node, dep_node, graph=None):
        """ Delete an edge from the graph. """
        if not graph:
            graph = self._graph
        if dep_node not in graph.get(ind_node, []):
            raise KeyError('this edge does not exist in graph')
        graph[ind_node].remove(dep_node)


    def from_dict(self, graph_dict):
        """ Reset the graph and build it from the passed dictionary.

        The dictionary takes the form of {node_name: [directed edges]}
        """

        self.reset_graph()
        for new_node in graph_dict.iterkeys():
            self.add_node(new_node)
        for ind_node, dep_nodes in graph_dict.iteritems():
            if not isinstance(dep_nodes, list):
                raise TypeError('dict values must be lists')
            for dep_node in dep_nodes:
                self.add_edge(ind_node, dep_node)

    def reset_graph(self):
        """ Restore the graph to an empty state. """
        self._graph = OrderedDict()

    def ind_nodes(self, graph=None):
        """ Returns a list of all nodes in the graph with no dependencies. """
        if graph is None:
            graph = self._graph

        dependent_nodes = set(node for dependents in graph.itervalues() for node in dependents)
        return [node for node in graph.keys() if node not in dependent_nodes]

    def validate(self, graph=None):
        """ Returns (Boolean, message) of whether DAG is valid. """
        graph = graph if graph is not None else self._graph
        if len(self.ind_nodes(graph)) == 0:
            return (False, 'no independent nodes detected')
        try:
            self.topological_sort(graph)
        except RuntimeError:
            return (False, 'failed topological sort')
        return (True, 'valid')


    def predecessors(self, node, graph=None):
            """ Returns a list of all predecessors of the given node """
            if graph is None:
                graph = self._graph
            return [key for key in graph if node in graph[key]]

    def resolve_node_dependents(self, node, li):
        if node:
            pred_list = self.predecessors(node)
            for dep in pred_list:
                if dep not in li:
                    li.append(dep)
                    log.debug("resolving node dependents for %s" % (dep,))
                    self.resolve_node_dependents(dep, li)

        return filter(lambda node: node in li, self.get_nodes_from_sorted_list())


    def topological_sort(self, graph_unsorted=None):
        """
        Repeatedly go through all of the nodes in the graph, moving each of
        the nodes that has all its edges resolved, onto a sequence that
        forms our sorted graph. A node has all of its edges resolved and
        can be moved once all the nodes its edges point to, have been moved
        from the unsorted graph onto the sorted one.
        """

        # This is the list we'll return, that stores each node/edges pair
        # in topological order.
        graph_sorted = []

        # Convert the unsorted graph into a hash table. This gives us
        # constant-time lookup for checking if edges are unresolved, and
        # for removing nodes from the unsorted graph.
        if not graph_unsorted:
            graph_unsorted = self._graph

        graph_unsorted = dict(graph_unsorted)
        for k, v in graph_unsorted.items():
            if k in v:
                v.remove(k)


        # Run until the unsorted graph is empty.
        while graph_unsorted:

            # Go through each of the node/edges pairs in the unsorted
            # graph. If a set of edges doesn't contain any nodes that
            # haven't been resolved, that is, that are still in the
            # unsorted graph, remove the pair from the unsorted graph,
            # and append it to the sorted graph. Note here that by using
            # using the items() method for iterating, a copy of the
            # unsorted graph is used, allowing us to modify the unsorted
            # graph as we move through it. We also keep a flag for
            # checking that that graph is acyclic, which is true if any
            # nodes are resolved during each pass through the graph. If
            # not, we need to bail out as the graph therefore can't be
            # sorted.
            acyclic = False
            for node, edges in list(graph_unsorted.items()):
                for edge in edges:
                    if edge in graph_unsorted:
                        break
                else:
                    acyclic = True
                    del graph_unsorted[node]
                    graph_sorted.append((node, edges))

            if not acyclic:
                # Uh oh, we've passed through all the unsorted nodes and
                # weren't able to resolve any of them, which means there
                # are nodes with cyclic edges that will never be resolved,
                # so we bail out with an error.
                raise RuntimeError("A cyclic dependency occurred")

        return graph_sorted

    def get_nodes_from_sorted_list(self, sorted_list=None):
        li = []
        if not sorted_list:
            sorted_list = self.topological_sort()
        for node, edges in sorted_list:
            li.append(node)
        return li

    def sorted_list_find_index_of_node(self, id, sorted_list=list()):
        index = 0
        for node, egdes in sorted_list:
            if node == id:
                break
            index += 1
        return index

    @property
    def graph(self):
        return self._graph

    def get_node_dependencies(self, node, graph=None):
        if not graph:
            graph = self._graph
        if node in graph:
            return graph[node]
        return None


    def downstream(self, node, graph=None):
        """ Returns a list of all nodes this node has edges towards. """
        if graph is None:
            graph = self._graph
        if node not in graph:
            raise KeyError('node %s is not in graph' % node)
        return list(graph[node])

    def get_node_downstream_elements(self, node, graph=None):
        """Returns a list of all nodes ultimately downstream
        of the given node in the dependency graph, in
        topological order."""
        if graph is None:
            graph = self._graph
        nodes = [node]
        nodes_seen = set()
        i = 0
        while i < len(nodes):
            downstreams = self.downstream(nodes[i], graph)
            for downstream_node in downstreams:
                if downstream_node not in nodes_seen:
                    nodes_seen.add(downstream_node)
                    nodes.append(downstream_node)
            i += 1
        return filter(lambda node: node in nodes_seen, self.get_nodes_from_sorted_list())


    def all_leaves(self, graph=None):
        """ Return a list of all leaves (nodes with no downstreams) """
        if graph is None:
            graph = self._graph
        return [key for key in graph if not graph[key]]

    def size(self):
        return len(self._graph)

    def node_exists(self, node):
        if self._graph and node in self._graph:
                return True
        return False
