# ----------------------------------------
# Author: Sandeep Puddupakkam (spuddupa)|
# Polaris IoX Sept 2016                 |
# ----------------------------------------
# __author__ = 'spuddupa'

import os
import logging
import os.path
import json
import yaml
import pickle

from appfw.utils.utils import Utils
from appfw.utils.cafevent import CAFEvent
from appfw.taskmgmt.taskmanager import TaskManager

from autoconfig import ios_cli_config

log = logging.getLogger("pdservices")


class AutoconfigcliService(object):
    """
    TODO: Add description here.
    """
    __singleton = None  # the one, true Singleton

    ENABLED = "enabled"
    IOS_HOST = "ios_host"
    IOS_PORT = "ios_telnet_port"
    EXEC_PWD = "ios_cli_exec_pwd"
    EXTERNAL_INTF = "external_interface"
    EXTERNAL_INTF_TYPE = "external_interface_type"

    _valid_rest_api_keys = [ENABLED,
                            IOS_HOST,
                            IOS_PORT,
                            EXEC_PWD,
                            EXTERNAL_INTF,
                            EXTERNAL_INTF_TYPE
                            ]
    _repo_folder = ""

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        # if not cls.__singleton:
        if cls != type(cls.__singleton):
            cls.__singleton = \
                super(AutoconfigcliService, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, repo_folder):

        AutoconfigcliService._repo_folder = repo_folder
        self._runtime_config_file = os.path.join(repo_folder, ".config")
        self._config = self._load()

        log.debug("Autoconfigcli Service initialized..")
        if self._config and self._config.get(AutoconfigcliService.ENABLED) == True:
            log.debug("Autoconfigcli Service Enabled.")
        else:
            log.debug("Autoconfigcli Service Disabled.")
        log.debug(yaml.dump(self._config))

    def _load(self):
        config = None
        startup_config_file = Utils.getAutoConfigCLIConfigFile()
        # First load the startup config.
        # If file does not exist, return None.
        # None indicates feature not available on this platform
        if os.path.isfile(startup_config_file):
            config = yaml.safe_load(file(startup_config_file))
            # Now over-write startup config with running config.
            # if key exists in startup_config, it will be over-written.
            if os.path.isfile(self._runtime_config_file):
                running_config = yaml.safe_load(file(self._runtime_config_file))
                for key, value in running_config.iteritems():
                    config[key] = value
        return config

    def _save(self):
        if self._config:
            o = yaml.safe_dump(self._config)
            file(self._runtime_config_file, "w").write(o)

    def cb(self, event):

        """This method will be called by the worker thread. """
        action = event.get("action")
        log.info("AutoconfigcliService CB event : %s" % action)
        if action == "RECONCILE":
            self._reconcile(event)
        elif action == "ADD":
            self._add(event)
        elif action == "DEL":
            self._delete(event)
        elif action == "DELETE_ALL":
            self._delete_all(event)
        else:
            log.info("AutoconfigcliService CB event not handled ")

    @staticmethod
    def quote_str(input_str):
        if input_str and ' ' in input_str:
            return '"' + input_str + '"'
        else:
            return input_str

    def _get_autoconfig_env(self, action, app_id=None, intf_name=None,
                            intf=None):
        env = {}
        # Do not proceed fwd if service is not configured or disabled.
        if not self.is_service_enabled():
            return env

        # Begin common values for all actions.
        if app_id:  # Only passed in in case of add and delete
            env["app_id"] = app_id
        # db dir
        env["db_dir"] = AutoconfigcliService._repo_folder

        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")
        # To see what these values are and how we get them, run
        # ./ioxclient platform network bridge list
        networkInfoList = nc.list_hosting_bridges()
        # Always take index 0 as it is bridge between ios and gos.
        networkInfo = networkInfoList[0]
        # To debug further, Uncomment the debugs below
        # log.info("Global Network info")
        # log.info(networkInfo)
        # log.info('Interface info:')
        # log.info(intf)

        if networkInfo:
            host = self._config.get(self.IOS_HOST, "")
            if host:
                env["ios_ip"] = host
            else:
                # Get ios ip first
                li = networkInfo.get("lease_info")
                if li:
                    ios_ip = li.get("routers")
                    if ios_ip:
                        env["ios_ip"] = ios_ip

            # Get iox ip
            ii = networkInfo.get("interface_info")
            if ii:
                iox_ip = ii.get("ipv4_address")
                if iox_ip:
                    env["iox_ip"] = iox_ip

        # port
        port = self._config.get(self.IOS_PORT, "")
        if port:
            env["port"] = port
        # ios exec password
        pwd = self._config.get(self.EXEC_PWD, "")
        if pwd:

            env["pwd"] = pwd

        # End common values for all actions.

        if action == "reconcile":
            env["action"] = "reconcile"
        elif action == "delete":
            env["action"] = "delete"

        elif action == "delete_all":
            env["action"] = "delete_all"

        elif action == "add":
            env["action"] = "add"

            # interface
            if intf_name:
                env["interface"] = intf_name

            # output interface type
            oivtt = self._config.get(self.EXTERNAL_INTF_TYPE, "")
            if oivtt:

                env["out_ivt_type"] = oivtt

            # output interface
            oivt = self._config.get(self.EXTERNAL_INTF, "")
            if oivt:
                env["out_ivt"] = AutoconfigcliService.quote_str(oivt)

            if intf:
                log.debug(str(intf))
                network_name = intf.get("network_name")
                if network_name.startswith("iox-nat"):
                    # This is a natted ip. Replace with svcbr0 ip
                    intf_info = networkInfo.get("interface_info")
                    if intf_info:
                        ipv4 = intf_info.get("ipv4_address")
                else:  # Bridge mode
                        ipv4 = intf.get('ipv4')
                if ipv4:
                    env["in_ip"] = ipv4
                else:
                    log.error("Input interface not set.")

                port_mappings = intf.get('port_mappings')
                if port_mappings:
                    udplist = port_mappings.get('udp')
                    tcplist = port_mappings.get('tcp')
                    if udplist:
                        tmp = udplist[0]
                        if tmp and len(tmp) >= 2:
                            # Using [1] as it would be the natted port
                            # in case of nat
                            env["uip"] = tmp[1]
                            env["uop"] = tmp[1]

                    if tcplist:
                        tmp = tcplist[0]
                        if tmp and len(tmp) >= 2:
                            # Using [1] as it would be the natted port
                            # in case of nat
                            env["tip"] = tmp[1]
                            env["top"] = tmp[1]

        log.debug("Env is:%s" % str(env))
        return(env)

    def _reconcile(self, event=None):
        pwd = self._config.get(self.EXEC_PWD, None)
        env = event.get("env", {})
        if env and pwd and pwd != "":
            ios_cli_config("reconcile", pwd, ios_ip=env.get("ios_ip"),
                    port=env.get("port"), db_dir=env.get("db_dir"))
        else:
            log.error("Autoconfig reconcile not done. env/pwd not set")

    def _delete_all(self, event=None):
        pwd = self._config.get(self.EXEC_PWD, None)
        env = event.get("env", {})

        if env and pwd and pwd != "":
            ios_cli_config("deleteall", pwd, ios_ip=env.get("ios_ip"),
                    port=env.get("port"), db_dir=env.get("db_dir"))
        else:
            log.error("Autoconfig reconcile not done. env/pwd not set")

    def _add(self, event):
        pwd = self._config.get(self.EXEC_PWD, None)
        app_id = event.get("app_name", "")
        env = event.get("env", {})
        if env and pwd and pwd != "":
            ios_cli_config("add", pwd,
                    ios_ip=env.get("ios_ip"),
                    port=env.get("port"),
                    app_name=app_id,
                    in_ip=env.get("in_ip"),
                    out_ivt_type=env.get("out_ivt_type"),
                    out_ivt=env.get("out_ivt"),
                    interface=env.get("interface"),
                    tip=env.get("tip", 0),
                    top=env.get("top", 0),
                    uip=env.get("uip", 0),
                    uop=env.get("uop", 0),
                    db_dir=env.get("db_dir"))
        else:
            log.error("Autoconfig reconcile not done. env/pwd not set")

    def _delete(self, event):
        pwd = self._config.get(self.EXEC_PWD, None)
        app_id = event.get("app_name", "")
        env = event.get("env", {})

        if env and pwd and pwd != "":
            ios_cli_config("delete", pwd,
                    ios_ip=env.get("ios_ip"),
                    port=env.get("port"),
                    app_name=app_id,
                    db_dir=env.get("db_dir"))
        else:
            log.error("Autoconfig reconcile not done. env/pwd not set")

    def start(self):
        from appfw.runtime.runtime import RuntimeService

        log.info("Starting AutoconfigcliService service")

        runtime_service = RuntimeService.getInstance()

        runtime_service._controller.subscribe(container_event_cb)
        if self.is_service_enabled():
            # We  have to do the reconcile here
            ac_event = {}
            ac_event["action"] = "RECONCILE"
            env = self._get_autoconfig_env("reconcile")
            ac_event["env"] = env
            log.info("AutoconfigcliService service. start(). Adding to tmq")
            taskmgr = TaskManager.getInstance()
            if taskmgr:
                taskmgr.queue_task(self.cb, args=(ac_event,))
            else:
                log.error("Unable to get TaskManager instance")
        else:
            log.info("Autoconfig reconcile not done. Service not enabled")

    def stop(self):
        log.info("Stopping AutoconfigcliService service")
        if self.is_service_enabled():
            ac_event = {}
            ac_event["action"] = "DELETE_ALL"
            env = self._get_autoconfig_env("delete_all")
            ac_event["env"] = env

            log.info("AutoconfigcliService service. stop(). Adding to tmq")
            taskmgr = TaskManager.getInstance()
            if taskmgr:
                taskmgr.queue_task(self.cb, args=[ac_event])
            else:
                log.error("Unable to get TaskManager instance")
        else:
            log.info("Autoconfig delete_all not done. Service not enabled")

    def get_config(self):
        conf = {}
        for key in AutoconfigcliService._valid_rest_api_keys:
            conf[key] = self._config.get(key, "")
        return conf

    def set_config(self, input):
        log.debug("Request:[%s]" % (input))

        ret_str = "The following key:values were saved. "
        for key, value in input.iteritems():
            log.info("Checking for key:[%s][%s]" % (key, value))
            if key in AutoconfigcliService._valid_rest_api_keys:
                if isinstance(value, str) and len(value) == 0:
                    value = None
                if key == self.ENABLED and value is False and \
                                self._config.get(self.ENABLED) is True:
                    # Service was toggled from enabled to disabled.
                    ac_event = {}
                    ac_event["action"] = "DELETE_ALL"
                    env = self._get_autoconfig_env("delete_all")
                    ac_event["env"] = env

                    log.info("AutoconfigcliService service. delete_all(). "
                             "Adding to tmq")
                    taskmgr = TaskManager.getInstance()
                    if taskmgr:
                        taskmgr.queue_task(self.cb, args=[ac_event])
                    else:
                        log.error("Unable to get TaskManager instance")
                self._config[key] = value
                ret_str += "%s:%s " % (key, value)
        self._save()
        return ret_str

    def get_db(self):
        log.debug("Request AC DB")
        db = {}
        db_file = os.sep.join([AutoconfigcliService._repo_folder, "autoconfig.db"])
        log.debug("DB file:[%s]" % db_file)
        if os.path.isfile(db_file):
            with open(db_file, 'rb') as f:
                db = pickle.load(f)
        log.debug(db)
        return db

    def is_service_enabled(self):
        if not self._config or self._config.get(AutoconfigcliService.ENABLED) is False:
            return False
        else:
            return True

    def add(self, app_id, intf_name, intf):
        log.info("AutoconfigcliService service. add()")
        if self.is_service_enabled():
            env = self._get_autoconfig_env("add", app_id, intf_name, intf)
            ac_event = {}
            ac_event["action"] = "ADD"
            ac_event["app_name"] = app_id
            ac_event["env"] = env

            self.cb(ac_event)
        else:
            log.info("Autoconfig add not done. Service not enabled")

    def delete(self, app_id, intf_name, intf):
        log.info("AutoconfigcliService service. delete()")
        if self.is_service_enabled():
            env = self._get_autoconfig_env("delete", app_id)

            ac_event = {}
            ac_event["action"] = "DEL"
            ac_event["app_name"] = app_id
            ac_event["env"] = env
            self.cb(ac_event)
        else:
            log.info("Autoconfig delete not done. Service not enabled")

    @classmethod
    def getInstance(cls, repo_folder):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton is None:
            cls.__singleton = AutoconfigcliService(repo_folder)

        return cls.__singleton


def container_event_cb(cafevent):  # The event passed here is a CAFEvent
    from appfw.runtime.hostingmgmt import HostingManager
    if not cafevent:
        log.info("Autoconfig network_event_cb. Event is null")
        return
    event_type = cafevent.event_type
    log.debug("Autoconfig network_event_cb. Event-TYPE:[%s]" % event_type)
    p = cafevent.payload
    payload = None
    network_info = None
    if p:
        payload = json.loads(p)
    if payload:
        # log.debug("Payload")
        # log.debug(payload)
        network_info = payload.get("networkInfo")
        # log.debug("Network info")
        # log.debug(network_info)
    if event_type == CAFEvent.TYPE_NETWORK_CHANGED and network_info:
        app_id = cafevent.app_id
        hm = HostingManager.get_instance()
        if hm:
            ac = hm.get_service("autoconfigcli-management")
            if ac:
                for intf, values in network_info.iteritems():
                    if values.get('mode') != 'static':
                        ac.add(app_id, intf, values)

    elif event_type == CAFEvent.TYPE_STOPPED:
        app_id = cafevent.app_id
        log.info("Autoconfig network_event_cb. app_id:[%s] "
                 "Event container stopped "
                 "Payload:[%s]" % (app_id, p))
        hm = HostingManager.get_instance()
        if hm:
            ac = hm.get_service("autoconfigcli-management")
            if ac:
                ac.delete(app_id, None, None)

    else:
        log.info("Autoconfig network_event_cb. Event [%s] ignored." %
                 event_type)
    # add code here to call the add/delete methods.

