'''
Created on Sep 29, 2012

@author: rnethi

Copyright (c) 2012-2013 by cisco Systems, Inc.
All rights reserved.
'''
import os
import shutil
import ConfigParser
import logging
import subprocess
import time
import threading
import datetime
import psutil
import glob
import tempfile
import zipfile
import tarfile
import pwd
import grp
import platform
import stat
import re

from threading import Thread
from container import AbstractContainer, AbstractContainerManager
from apptypes import AppType
from ..utils.utils import Utils, CGROUP_CONTROLLERS, APP_CONFIG_NAME_2
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *
from ..utils import subprocessutils
from ..runtime import bootstrap
from ..overrides import event
from appfw.cartridge.cartridge import *
from ..utils.commandwrappers import *
from appfw.utils.filegen import filegen
from appfw.hosting.filemgmt import FileMgmt
from appfw.api.systeminfo import SystemInfo
from container_utils import ContainerUtils

log = logging.getLogger("runtime.hosting")


class ProcessCommands(object):
    def __init__(self):
        self._start_command = None
        self._stop_command = None
        self._custom_scripts = {}
        self._allowed_hooks = ['notification', 'config', 'custom-status', 'statistics']
        self.updatedProcs = False

    @property
    def start_command(self):
        return self._start_command

    @start_command.setter
    def start_command(self, cmd):
        self._start_command = cmd

    @property
    def stop_command(self):
        return self._stop_command

    @stop_command.setter
    def stop_command(self, cmd):
        self._stop_command = cmd

    def get_custom_scripts(self, k):
        """Return custom command if it exists. Else None"""
        if k not in self._allowed_hooks:
            raise ValueError("Custom hook %s not allowed" % k)
        return self._custom_scripts.get(k, None)

    def add_custom_scripts(self, k, v):
        """Add a custom command of a type"""
        if k not in self._allowed_hooks:
            raise ValueError("Custom hook %s not allowed" % k)
        self._custom_scripts[k] = v


class ProcessContainer(AbstractContainer):
    """
    Abstract container implementation for a container that uses a process to host a connector
    """

    def __init__(self, containerId, containerDir, processContainerManager, resources={}, appconfigfile="", app_env={}):
        self._containerId = containerId
        self._containerDir = containerDir
        self._cartridge_list = []
        systemConfigPath = Utils.getSystemConfigPath()
        systemConfig = ConfigParser.SafeConfigParser()
        systemConfig.read(systemConfigPath)
        self._uid = None
        self._gid = None
        self.mem_limit_mb = 0
        self.cpu_shares = 0
        self.disk = 0
        self._resources = resources
        self._has_failures = False
        self._get_resource_requirements()
        self._app_env = app_env
        self._cgroup_name = None
        self._logDirName = systemConfig.get("app-settings", "appLogsDir")
        self._appdata_dir = "appdata"
        if systemConfig.has_option("app-settings", "appDataDir"):
            self._appdata_dir = systemConfig.get("app-settings", "appDataDir")
        if systemConfig.has_option("process-container", "root"):
            self.rootPath = systemConfig.get("process-container", "root")

        self.disk_type = Utils.getSystemConfigValue("controller",
                                                "disk_type", "restricted")

        self.disk_type = Utils.getSystemConfigValue("controller",
                                                "disk_type", "restricted")
        self.chroot_require = processContainerManager.chroot_require
        self.chroot_path = processContainerManager.chroot_path
        self.chroot_mount_paths = processContainerManager.chroot_mount_paths
        self.data_volume = processContainerManager.data_volume
        self.data_mount_point = processContainerManager.data_mount_point
        self.cartridge_mount_point = processContainerManager.cartridge_mount_point
        self.create_user = processContainerManager.create_user
        self.enable_cgroup = processContainerManager.enable_cgroup
        self.cgroup_controllers = processContainerManager.cgroup_controllers
        if self.create_user is True:
            userId = self._containerId.encode('base64', 'strict').strip().translate(None,'+/=').lower()
            self._userId = userId[:32]
            out, rc = adduser('--gecos', '', '--no-create-home', '--disabled-password',  self._userId)
            if rc != 0:
                log.error("adduser %s failed Error:%s" % (self._userId, out))

            self._uid = pwd.getpwnam(self._userId).pw_uid
            self._gid = grp.getgrnam(self._userId).gr_gid
        else:
            self._userId = systemConfig.get("app-settings", "appUserName")

        self.data_volume_path = self.data_volume
        if self.data_volume is not None:
            self.data_volume_path = os.path.join(self.data_volume, containerId)
            if not os.path.isdir(self.data_volume_path):
                os.makedirs(self.data_volume_path)
                if self.create_user:
                   os.chown(self.data_volume_path, self._uid, self._gid)
            #Provision app config
            self._app_config_file = ""
            if appconfigfile:
                existing_cfg = os.path.join(self.data_volume_path, os.path.basename(appconfigfile))
                if not os.path.isfile(existing_cfg):
                    shutil.copy(appconfigfile, self.data_volume_path)
                self._app_config_file = os.path.basename(appconfigfile)
                log.debug("Provisioned the appconfig file at : %s", self.data_volume_path)
            #Create logs dir
            self._logsDir = os.path.join(self.data_volume_path, self._logDirName)
            if not os.path.isdir(self._logsDir):
                os.makedirs(self._logsDir)
                if self.create_user:
                   os.chown(self._logsDir, self._uid, self._gid)

            #Provision appdata directory 
            self._appdata_dirpath = os.path.join(self.data_volume_path, self._appdata_dir)
            if not os.path.exists(self._appdata_dirpath) :
                os.makedirs(self._appdata_dirpath)
                if self.create_user:
                   os.chown(self._appdata_dirpath, self._uid, self._gid)
            log.debug("Provisioned appdata dir at %s" % self._appdata_dirpath)

        self._connectorDirName = systemConfig.get("app-settings", "appDirName")

        #self._logsDir = os.path.join(self._containerDir, self._connectorDirName, self._logDirName)
        self._processCommands = ProcessCommands()
        self._customLogFiles = None
        self._monitoredLogFiles = []
        self._proc = None

    def __str__(self):
        return "PROCESS"

    @property
    def app_resources(self):
        return self._resources

    @property
    def cartridge_list(self):
        # As we are not supporting any cartridges in process container
        # so returning empty list
        #return self._cartridge_list
        return []

    @property
    def app_env(self):
        return self._app_env

    @app_env.setter
    def app_env(self, val):
        self._app_env = val

    @property
    def cgroup_name(self):
        return self._cgroup_name

    @cgroup_name.setter
    def cgroup_name(self, val):
        self._cgroup_name = val

    def getId(self):
        """
        Returns container id
        """
        return self._containerId
    
    def start(self):
        """
        Starts a container
        """
        if not self.isRunning():
            log.info("Launching connector start command with args: %s" % self._processCommands.start_command)
            try:
                env = os.environ.copy()
                self._proc = subprocess.Popen(args=self._processCommands.start_command, cwd=self._containerDir, env=env)
                pid_path = self.rootPath + "/" + self._containerId + "/.pid"
                with open(pid_path, "w") as f:
                    f.write(str(self._proc.pid))
                log.info("started container id = %s with pid = %d",
                                     self._containerId, self._proc.pid)
            except Exception as exception:
                log.exception("exception while starting container id = %s: %s",
                                          self._containerId, str(exception))
        else:
            log.info("Connector:%s is already running", self.getId())
        return

    def stop(self, graceful=True):
        """
        Stops a container
        """
        if self._proc:
            if self.isRunning():
                log.debug("Stopping Connector: %s ", self._containerId)
                if self._processCommands.stop_command:
                    children = Utils.get_child_processes(self._proc.pid)
                    env = os.environ.copy()
                    env["CAF_APP_PID"] = str(self._proc.pid)
                    env["CAF_CHILDREN_PIDS"] = " ".join([str(c) for c in children])
                    p = subprocess.Popen(args=self._processCommands.stop_command, cwd=self._containerDir, env=env)
                    # Wait for 5 seconds
                    count = 0
                    while (p.poll() is None) and count < 5:
                        time.sleep(1)
                        count += 1
                    #Now kill the process
                    try:
                        p.kill()
                    except:
                        pass
                    p.wait()
                self._stop_process()
        pid_path = self.rootPath + "/" + self._containerId + "/.pid"
        if (os.path.exists(pid_path)):
            os.remove(pid_path)
        return

    def _stop_process(self):
        try :
            pid = self._proc.pid
            children = Utils.get_child_processes(pid)
            log.debug("Process : %d, Children : %s" % (pid, children))
            if (self._proc.poll() is None) :
                #Step 1 stop the connector proc itself
                log.info("sending SIGTERM to process pid = %d", self._proc.pid)
                self._proc.terminate()
                time.sleep(1)
                # Wait for process to exit.
                count = 0
                while (self._proc.poll() is None) and count < 10:
                    time.sleep(1)
                    count += 1
                # Timed out - that's it, we have to kill it
                if (count >= 10) :
                    log.warning(" sending SIGKILL to process pid = %d", self._proc.pid)
                    try:
                        self._proc.kill()
                    except OSError:
                        # Ignore if process already terminated
                        pass
            else :
                log.warning("process with pid = %d was already dead", self._proc.pid)
            # Finally, wait() to avoid zombie
            self._proc.wait()

            # Kill all the children of the process. This could happen if the process didn't terminate
            # all its children by itself.
            for pid in children:
                try:
                    os.kill(pid, 9) # Send SIGKILL
                    log.warning("Killed child process %d with SIGKILL" % pid)
                except OSError:
                    # Ignore if process was already terminated
                    pass

        except Exception, exception :
            log.error("exception while trying to stop process with pid = %d, exception = %s",
                      self._proc.pid, str(exception))
            raise

    def isRunning(self):
        """
        Tests if the connector is running or not
        """
        if self._proc is not None:
            return self._proc.poll() is None
        return False

    def runCommand(self, cmd):
        """
        Runs a command inside a container
        """
        pass


    def getProcessInfo(self):
        """
        Gets process information from inside a container
        """
        pass

    def get_app_ipaddress_info(self):
        """
        Returns the ipaddress  for the container
        """
        network_info = {}
        if self.isRunning():
            try:
                data, ret = ifconfig()
                if ret != 0:
                    log.error("Error occurred while executing the ifconfig command, cause %s"%data)
                    raise Exception("Error occurred while executing the ifconfig command, cause %s"%data)
                log.debug("Container %s ifconfig: %s" % (self.getId(), data))
                for paragraph in data.split("\n\n"):
                    if paragraph is None:
                        continue
                    try:
                        interface_name = paragraph.split()[0]
                    except IndexError:
                        continue

                    re_ipv4 = re.search(r'inet addr:(\S+)', paragraph)
                    re_ipv6 = re.findall(r'inet6 addr: (\S+)', paragraph)
                    re_mac = re.search(r'HWaddr (\S+)', paragraph)
                    ipv4 = None
                    ipv6 = None
                    mac = None
                    if re_ipv4:
                        ipv4 = re_ipv4.group(1)

                    if re_ipv6:
                        ipv6 = re_ipv6

                    if re_mac:
                        mac = re_mac.group(1)

                    d = network_info.get(interface_name)
                    if d is None:
                        network_info[interface_name] = {}
                        d = network_info.get(interface_name)

                    d["ipv4"] = ipv4
                    d["ipv6"] = ipv6
                    d["mac"] = mac

                return network_info
            except Exception as ex:
                log.error("Error while getting the network info for the container %s, cause %s"%(self.getId(), ex.message))
        return network_info

    def _get_resource_requirements(self):
        self.mem_limit_mb = self._resources.get("memory", 10)
        self.cpu_shares = self._resources.get("cpu", 10)
        self.disk = self._resources.get("disk", 10)

        log.debug("Requested resources : Memory : %s, CPU Shares : %s disk space : %s" % (
            self.mem_limit_mb, self.cpu_shares, self.disk))

    def get_app_memory_usage(self):
        """
        Returns the Memory occupied by Container in KB
        """
        log.debug("get_app_memory_usage called for container %s" % self._containerId)
        proc_attr = self._proc
        if (proc_attr is not None) :
            # return the memory usage in KB
            process_info = psutil.Process(proc_attr.pid)
            if not process_info.is_running() :
                return None
            mem = process_info.get_memory_info()[0] / float(2 ** 10)
            return mem
        return None


    def get_app_cpu_usage(self):
        """
        Returns the CPU usage by Container in percentage
        """
        log.debug("get_app_cpu_usage called for container %s" % self._containerId)
        proc_attr = self._proc
        if (proc_attr is not None) :
                # return the memory usage in KB
            process_info = psutil.Process(proc_attr.pid)
            if not process_info.is_running() :
                return None
            cpu_usage = process_info.get_cpu_percent(0.2)
            return cpu_usage
        return None


    def get_app_network_usage(self):
        """
        Gets Network information from inside a container
        """
        # Not available for process containers
        return None

    def get_app_disk_usage(self):
        """
        Gets Disk information from inside a container in MBs
        """
        appdir = os.path.join(self._containerDir, self._connectorDirName)
        return Utils.get_dir_size(appdir) / float(1024 * 1024)

    def get_app_cpu_allocated(self):
        """Return allocated cpu for the app in percentage"""
        return self.cpu_shares

    def get_app_memory_allocated(self):
        """Return allocated memory for the app in MB"""
        return self.mem_limit_mb

    def get_app_disk_allocated(self):
        """Return allocated disk for the app in MB"""
        return self.disk

    def get_app_network_allocated(self):
        """Return allocated network resources for the app"""
        pass

    def _get_logfile_path(self, logfilename):
        """Get absolute path for a given log file name"""
        if os.path.isabs(logfilename):
            return logfilename
        else:
            return os.path.join(self._logsDir, logfilename)

    def getLogsList(self):
        """
        Gets list of logs from inside a container
        """
        logName = []
        logSize = []
        logTimestamp = []

        log.debug("Logs Dir: %s, Container Dir: %s" % (self._logsDir, self._containerDir))
        logDirList = [os.path.join(self.data_volume_path, self._logDirName, f)
                      for f in os.listdir(self._logsDir)]

        # Show if there are any specified custom logfiles
        if self._customLogFiles:
            log.debug("Extending with custom log files : %s" % self._customLogFiles)
            logDirList.extend(self._customLogFiles)

        for fileName in logDirList:
            filePath = self._get_logfile_path(fileName)

            if os.path.isfile(filePath):
                self._monitoredLogFiles.append(filePath)
                logName.append(fileName)
                logSize.append(os.path.getsize(filePath))
                logTimestamp.append(time.ctime(os.path.getmtime(filePath)))
            else:
                log.warning("%s is not a valid file" % filePath)

        logsList = zip(logName, logSize, logTimestamp)
        log.debug("List of process container '%s' log files : %s" % (str(self.getId()), str(logsList)))

        return logsList

    def getLogTail(self, filename, lines=10):
        """
        Gets last 'lines' from connector log specified by filename
        """
        logLines = []
        filePath = self._get_logfile_path(filename)

        log.debug("Requested log file : %s" % filePath)
        if os.path.isfile(filePath) and filePath in self._monitoredLogFiles:
            logLines = Utils.tail(file(filePath), lines)
            log.debug("Logtail for %s, %d lines : %s" % (filename, lines, str(logLines)))
        else:
            log.error("Requested file %s is not a valid log file" % filename)

        return logLines

    def getLogPath(self, filename):
        """If the passed filename is a valid log file (i,e monitored)
        then return full path. Else return None"""

        filePath = self._get_logfile_path(filename)
        if os.path.isfile(filePath) and filePath in self._monitoredLogFiles:
            log.debug("Log path for %s is %s" % (filename, filePath))
            return filePath

        log.error("Requested file %s is not a valid log file" % filename)
        return None

    def getCoreList(self):
        """
        Gets list of core from inside a container
        """
        # Support for core files yet to get
        return []

    def getCoreFile(self, corefilename):
        return ""

    def executeCustomScript(self, scriptName, args=None):
        """
        Execute a custom script as required by <scriptName>. Pass <args>
        If successful, return {'out': out, 'err': err, 'exitcode': exitcode}
        Else, return None
        """
        customScript = self._processCommands.get_custom_scripts(scriptName)
        if customScript:
            log.debug("Executing Custom Script : %s. Args: %s. CWD: %s" % (customScript, args, self._containerDir))
            p = subprocess.Popen(args=[customScript, args],
                                 cwd=self._containerDir, stdout=subprocess.PIPE, stderr=subprocess.PIPE)

            (out, err) = subprocessutils.custom_communicate(p)
            # (out, err) = p.communicate()
            exitcode = p.returncode
            rd = {'out': out, 'err': err, 'exitcode': exitcode}
            log.debug("Returning : %s" % str(rd))
            return rd

        log.warning("%s script is not supported by %s app" % (scriptName, str(self.getId())))
        return None

    def getLogContents(self, fileName):
        """
        Gets logs contents for a given file from inside a container
        """
        dbugString = "Getting log contents from '%s' for container '%s'." \
                       % (fileName, self.getId())

        log.debug(dbugString)
        logContent = ''
        if os.path.isabs(fileName):
            logFile = fileName
        else:
            logFile = os.path.join(self._containerDir, fileName)

        if os.path.isfile(logFile) and logFile in self._monitoredLogFiles:
            try:
                with open(logFile) as f:
                    logContent = f.read()
                return logContent
            except IOError:
                log.error("Exception while " + dbugString)
                raise IOError
        else:
            log.error("Requested file %s is not a valid log file" % fileName)
            return None



    def _copyArchive(self, archivePath, reconcile):
        """
        Copies a specified archive to container and explodes the archive
        """
        #extractall doesn't seem to preserve permissions
        #http://bugs.python.org/issue15795
        #Using system() as a work-around for now

        #with zipfile.ZipFile(archivePath, 'r') as archiveZip:
        #    archiveZip.extractall(self._containerDir)

        #TODO: handle spaces in the file arguments
        # User id for the container is first 3 char of containerId + base64
        # of container id

        if reconcile is True:
            if (os.path.exists(self._containerDir)):
                return
        app_mount  = os.path.join(self._containerDir, "app_mount")
        if not os.path.isdir(app_mount):
            os.makedirs(app_mount)

        try:
            #Mount app 
            out, rc = mountext2(archivePath, app_mount)
            if rc != 0:
                log.error("Error in mounting staged  app ext2 image: %s", str(out))
                raise AppInstallationError("Error in creating app ext2 image: %s", str(out))
            Utils.copytree_contents(app_mount, self._containerDir)
        except Exception as ex:
            log.error("Error while copying staged app dir: %s" % str(ex))
            raise AppInstallationError("Error while copying staged app dir: %s" % str(ex))
        finally:
            if Utils.ismount_exists(app_mount) :
                #Remove app mount and ext2 app image
                out, rc = umount("-l",  app_mount)
                if rc != 0:
                    log.error("Unmounting failed for an app. ret code: %s error: %s"
                      % (rc, str(out)))
            log.debug("Removing the app mount dir: %s" % app_mount)
            shutil.rmtree(app_mount, ignore_errors=True)
            
        for (path, dirs, files) in os.walk(self._containerDir):
            if self.create_user is True:
                os.chown(path, self._uid, self._gid)
            for file in files:
                if self.create_user is True:
                    os.chown(os.path.join(path,file), self._uid, self._gid)
        if self.chroot_require is True:
            self._create_chroot_env()

    def _create_chroot_env(self):
        """
        Creates chroot env by creating and mount required system dirs
        """
        usrdir = os.path.join(self._containerDir, "usr")
        if not os.path.exists(usrdir) :
            os.mkdir(usrdir)

        if not self.chroot_mount_paths is None:
            for mntdir in self.chroot_mount_paths.split(",") :
                tgmntpath = self._containerDir + mntdir
                if os.path.exists(mntdir):
                    if not os.path.exists(tgmntpath) :
                        os.mkdir(tgmntpath)
                    if not Utils.ismount_exists(tgmntpath) :
                        out, rc = mount(arglist=["--bind" ,  mntdir , tgmntpath])
                        if rc != 0:
                            log.error("mount failed:  code: %s error: %s" %
                                                            ( rc, str(out)))

                        out, rc =mount(arglist=["-o", "remount,ro,bind",  mntdir, tgmntpath])
                        if rc != 0:
                            log.error("mount failed:  code: %s error: %s" %
                                                            ( rc, str(out)))


        if not self.data_volume_path is None:
            tgmntpath = self._containerDir + self.data_mount_point
            if not os.path.exists(tgmntpath) :
                os.mkdir(tgmntpath)
            if not Utils.ismount_exists(tgmntpath) :
                out, rc = mount(arglist=["--bind", self.data_volume_path, tgmntpath])
                if rc != 0:
                    log.error("mount failed:  code: %s error: %s" % 
                                                        ( rc, str(out)))
                    raise Exception("mount failed:  code: %s error: %s" % 
                                                        ( rc, str(out)))
            if self.create_user is True:
                os.chown(tgmntpath, self._uid, self._gid)

    def _addCartridge(self, cartridge_list):
        # Setup mount directory within rootfs for mounting cartridge
        cartridge_list = None
        if cartridge_list is None:
            return
        self._cartridge_list = cartridge_list
        if self.chroot_require is True:
            for cartridge in cartridge_list:
                if cartridge.type == "baserootfs":
                    continue
                if not "mountable" in cartridge.handleas:
                    continue
                cartridge_src_mount = cartridge.get_location()
                tgmntpath = self._containerDir + self.cartridge_mount_point + os.sep + cartridge.id 
                if not os.path.exists(tgmntpath) :
                    os.makedirs(tgmntpath)
                if not Utils.ismount_exists(tgmntpath) :
                    log.debug("Mounting cartridge root src %s dst %s" % (cartridge_src_mount, tgmntpath))
                    out, rc = mount(arglist=["--bind", cartridge_src_mount, tgmntpath])
                    if rc != 0:
                        log.error("mount failed:  code: %s error: %s" % 
                                                        ( rc, str(out)))
                        raise Exception("mount failed:  code: %s error: %s" % 
                                                        ( rc, str(out)))
                    out, rc =mount(arglist=["-o", "remount,ro,bind", cartridge_src_mount, tgmntpath]) 
                    if rc != 0:
                        log.error("mount failed:  code: %s error: %s" % 
                                                        ( rc, str(out)))
                        raise Exception("mount failed:  code: %s error: %s" % 
                                                        ( rc, str(out)))

    def _getWrapperContainerScript(self, env={}):
        """
        Returns a common wrapper script; script which exports environment variables.
        """
        dataroot =  self.getContainerRoot()
        script = ""
        script += "#!/bin/sh \n"

        appContentDir = self._connectorDirName + "/app"
        if self.chroot_require is True:
            app_data_dir = self.data_mount_point
            #cfgfile = os.path.join("/", appContentDir, Utils.find_app_config_filename(os.path.join(self.getContainerRoot(),appContentDir)))
        else :
            app_data_dir = self.data_volume_path
            #cfgfile = os.path.join(dataroot, appContentDir, Utils.find_app_config_filename(os.path.join(self.getContainerRoot(),appContentDir)))
        cfgfile = os.path.join(app_data_dir, self._app_config_file)
        app_logs_dir = os.path.join(app_data_dir, self._logDirName)
        app_appdata_dir = os.path.join(app_data_dir, self._appdata_dir)
        self.app_env.update(env)
        self.app_env["CAF_APP_PERSISTENT_DIR"] = app_data_dir
        self.app_env["CAF_APP_LOG_DIR"] = app_logs_dir
        if os.path.isfile(cfgfile):
            self.app_env["CAF_APP_CONFIG_FILE"] = cfgfile
        self.app_env["CAF_APP_CONFIG_DIR"] = app_logs_dir
        self.app_env["CAF_APP_APPDATA_DIR"] = app_appdata_dir
        self.app_env["CAF_APP_USERNAME"] = self._userId
        self.app_env["CAF_SYSTEM_UUID"] = SystemInfo.get_system_uuid()
        self.app_env["CAF_DISK_TYPE"] = self.disk_type
	if (SystemInfo.is_secure_storage_supported() == True):
	    self.app_env["CAF_SS_PORT"] = SystemInfo.get_secure_storage_port()
	    self.app_env["CAF_SS_IP_ADDR"] = SystemInfo.get_secure_storage_ip_addr()
        """
        script += "export CAF_APP_PERSISTENT_DIR="+app_data_dir+" \n"
        script += "export CAF_APP_LOG_DIR=%s\n" % app_logs_dir
        script += "export CAF_APP_CONFIG_FILE=%s\n" % cfgfile
        script += "export CAF_APP_CONFIG_DIR=%s\n" % app_logs_dir
        script += "export CAF_APP_APPDATA_DIR=%s\n" % app_appdata_dir
        script += "export CAF_APP_USERNAME=%s\n" % self._userId
        script += "export CAF_SYSTEM_UUID=%s\n" % SystemInfo.get_system_uuid()
	if (SystemInfo.is_secure_storage_supported() == True):
	    script += "export CAF_SS_PORT=%s\n" % SystemInfo.get_secure_storage_port()
	    script += "export CAF_SS_IP_ADDR=%s\n" % SystemInfo.get_secure_storage_ip_addr()
        """
        resources = self._resources
        if "device-info" in resources:
            device_info_list = resources["device-info"]
            log.debug("Device info list %s", device_info_list)
            self.app_env.update(Utils.create_device_info_env_var(device_info_list=device_info_list))
        if "oauth" in resources:
            oauth_info_list = resources["oauth"]
            app_network = resources.get('network', None)
            app_network_name = []
            oauth_ip_list = []
            if app_network is not None:
                for net_intf in app_network:
                    if "network-name" in net_intf:
                        app_network_name.append(net_intf['network-name'])
            if not app_network_name:
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                nc = hm.get_service("network-management")
                app_network_name.append(nc.get_default_network())

            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            network_manager = hm.get_service("network-management")
            network = network_manager.get_network(app_network_name[0])
            hb = network.get_container_source_bridge()
            server_ipv4 = hb.get_bridge_ipv4_address()
            if server_ipv4 is not None:
                oauth_ip_list.append(server_ipv4)
            server_ipv6 = hb.get_bridge_ipv6_address()
            if server_ipv6 is not None:
                oauth_ip_list.append(server_ipv6)

            server_port = Utils.getSystemConfigValue("api", "port", default=8443, parse_as="int")

            log.debug("Invoke api for setting oauth related env variables for %s", self._containerId)
            self.app_env.update(Utils.create_oauth_info_env_var(container_id=self._containerId, server_port=str(server_port), server_ip_list=oauth_ip_list, oauth_info_list=oauth_info_list))

        if self.app_env is not None:
            for key,value in self.app_env.items():
                script += "export "+key+"=\""+value+"\" \n"
        #Add cartridge commands
        if self._cartridge_list is not None:
            for cartridge in self._cartridge_list:
                if cartridge.type == "baserootfs":
                    continue
                if not "mountable" in cartridge.handleas:
                    continue
                if self.chroot_require is True:
                    ccmds = cartridge.get_commands(os.path.join(self.cartridge_mount_point, cartridge.id))
                else:
                    cartridge_src_dir =  cartridge.get_location()
                    ccmds = cartridge.get_commands(cartridge_src_dir)
                script +=  "\n" + "\n".join(ccmds)
        log.debug("Container Start Wrapper script : %s", script)
        return script

    def _addStartCommand(self, cmd, resources={}):
        log.debug("Adding Start Command : %s. Container : %s" % (cmd, self.getContainerRoot()))
        if cmd is None:
            self._processCommands.start_command = cmd
            return
        dataroot =  self.getContainerRoot()
        #Generate wrapper script which exports all the app env variables
        
        script = self._getWrapperContainerScript(self.app_env)
        script += "\nexec " + cmd
    
        start_container_path = os.path.join(self._connectorDirName, 
                                                    "start_container")
        filePath = os.path.join(self.getContainerRoot(), start_container_path)

        with open(filePath, "wb") as f:
            f.write (script)
        os.chmod(filePath, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
        chroot_cmd = []
        if self._cgroup_name:
            log.debug("Creating the start command to start the process in cgroup!")
            cgexec = which("cgexec")
            if cgexec:
                chroot_cmd = [cgexec, "-g", ",".join(self.cgroup_controllers) + ":"+ self._cgroup_name]
            else:
                log.error("Required binary '%s' is not found!"%"cgexec")
                raise Exception("Required binary '%s' is not found!"%"cgexec")
        if self.chroot_require is True:
            if self.create_user is True:
                chroot_cmd.extend([self.chroot_path, self.getContainerRoot(), "/bin/su",  self._userId,  "-c",  "/" + start_container_path])
            else:
                chroot_cmd.extend([self.chroot_path, self.getContainerRoot(), "/bin/sh",  "-c",  start_container_path])
        else:
            if self.create_user is True:
                chroot_cmd.extend(["/bin/su",  self._userId,  "-c",  start_container_path])
            else:
                if not chroot_cmd:
                    chroot_cmd = ["/bin/sh", "-c"]
                chroot_cmd.extend([start_container_path])
        self._processCommands.start_command = chroot_cmd

        
    def _addStopCommand(self, cmd):
        log.debug("Adding Stop Command : %s. Container : %s" % (cmd, self.getContainerRoot()))
        if cmd is None:
            self._processCommands.stop_command = cmd
            return
        #Generate wrapper script which exports all the app env variables

        script = self._getWrapperContainerScript()
        script += "\nexec " + cmd

        stop_container_path = os.path.join(self._connectorDirName,
                                                    "stop_container")
        filePath = os.path.join(self.getContainerRoot(), stop_container_path)

        with open(filePath, "wb") as f:
            f.write (script)
        os.chmod(filePath, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)

        chroot_cmd = []
        if self.chroot_require is True:
            dataroot =  self.getContainerRoot()
            if self.create_user is True:
                chroot_cmd = [self.chroot_path, self.getContainerRoot(),   
                        "/bin/su",  self._userId,  "-c",  "/" + stop_container_path]
            else :
                chroot_cmd = [self.chroot_path, self.getContainerRoot(),   
                        "/bin/sh",  "-c",  stop_container_path]
        else:
            if self.create_user is True:
                chroot_cmd = ["/bin/su",  self._userId,  "-c",  stop_container_path]
            else:
                if not chroot_cmd:
                    chroot_cmd = ["/bin/sh", "-c"]
                chroot_cmd.append(stop_container_path)
        self._processCommands.stop_command = chroot_cmd

    def _addCustomScripts(self, cs):
        if cs:
            for k, v in cs.iteritems():
                log.debug("Adding CustomScript. Name: %s. Script: %s. Container : %s" % (
                    k, v, self.getContainerRoot()))
                self._processCommands.add_custom_scripts(k, v)

    def _addCustomLogFiles(self, customLogFiles=[]):
        log.debug("Adding custom log files : %s" % str(customLogFiles))
        self._customLogFiles = customLogFiles

    def hasFailures(self):
        #Return False as FAILED state is no longer supported
        #This ends up in lockout from LM and FD
        return False
        # May be it can be use later
        if self._proc:
            exit_code = self._proc.poll()
            if exit_code:
                if exit_code != 0:
                    self._has_failures = True
        return self._has_failures

    def getContainerRoot(self):
        return self._containerDir

    def preserve_container_data(self, archive_path, preserve_file_list=[], preserve_app_config=False):
        # Currently we preserve entire container root while upgrade.
        data_root = self.getContainerRoot()
        tmpdir = ProcessContainerManager.getInstance().get_tmpdir()
        container_data_arch, preserved_files_records = Utils.preserve_container_data(data_root, self._containerId, archive_path,
                      preserve_file_list, preserve_app_config, tmpdir)
        return container_data_arch, preserved_files_records

    def restore_container_data(self, container_data_arch):
        try:
            data_root = self.getContainerRoot()
            Utils.restore_container_data(data_root, container_data_arch)
        except Exception as ex:
            log.error("Failed to restore container data: error - %s", str(ex))
            raise Exception("Failed to restore container data: error - %s", str(ex))

class ProcessContainerReconcile(ProcessContainer):
    def __init__(self, containerId, containerDir, processContainerManager, resources={}, appconfigfile="", app_env={}):
        super(ProcessContainerReconcile, self).__init__(containerId, containerDir, processContainerManager, resources, appconfigfile, app_env)
        self._procpid = None
        pid_path = self.rootPath + "/" + self._containerId + "/.pid"
        if os.path.exists(pid_path):
            with open(pid_path, "r") as f1:
                data = f1.read()
            self._procpid = int(data)

    def hasFailures(self):
        if self._proc is not None:
            exit_code = self._proc.poll()
            if exit_code:
                if exit_code != 0:
                    self._has_failures = True
        elif self._procpid is not None:
            if psutil.pid_exists(self._procpid) is False:
                self._has_failures = True
        return self._has_failures

    def get_app_memory_usage(self):
        """
        Returns the Memory occupied by Container in KB
        """
        if self._proc is not None:
            mem = super(ProcessContainerReconcile, self).get_app_memory_usage()
            return mem
        elif self._procpid is not None:
            log.debug("get_app_memory_usage called for container %s" % self._containerId)
            proc_attr = self._procpid
            if (proc_attr is not None) :
                # return the memory usage in KB
                process_info = psutil.Process(proc_attr)
                if not process_info.is_running() :
                    return None
                mem = process_info.get_memory_info()[0] / float(2 ** 10)
                return mem
        return None


    def get_app_cpu_usage(self):
        """
        Returns the CPU usage by Container in percentage
        """
        if self._proc is not None:
            cpu_usage = super(ProcessContainerReconcile, self).get_app_cpu_usage()
            return cpu_usage
        elif self._procpid is not None:
            log.debug("get_app_cpu_usage called for container %s" % self._containerId)
            proc_attr = self._procpid
            if (proc_attr is not None) :
                # return the memory usage in KB
                process_info = psutil.Process(proc_attr)
                if not process_info.is_running() :
                    return None
                cpu_usage = process_info.get_cpu_percent(0.2)
                return cpu_usage
        return None

    def start(self):
        """
        Starts a container
        """
        if not self.isRunning():
            """
            The reconcile is done at the init itself

            This code will be hit when a process container that was reconciled was moved to stop 
            state and then again to start
            In this scenario the process container has a reconcile nature but not actually
            having a container pid. create a new process container using base class
            start method
            """
            super(ProcessContainerReconcile, self).start()
            self._procpid = None
            return

    def stop(self, graceful=True):
        """
        Stops a container
        """
        if self.isRunning():
            log.debug("Stopping Connector: %s pid %d", self._containerId, self._procpid)
            if self._proc is not None:
                super(ProcessContainerReconcile, self).stop()
                self._procpid = None
            elif self._procpid is not None:
                pid = str(self._procpid)
                children = Utils.get_child_processes(self._procpid)
                kill(pid)

                for pid in children:
                    if os.path.isdir(os.path.join("/proc", str(pid))):
                        try:
                            os.kill(pid, 9) # Send SIGKILL
                            log.warning("Killed child process %d with SIGKILL" % pid)
                        except OSError:
                            # Ignore if process was already terminated
                            pass
                self._procpid = None
            pid_path = self.rootPath + "/" + self._containerId + "/.pid"
            if (os.path.exists(pid_path)):
                os.remove(pid_path)
            return

    def isRunning(self):
        """
        Tests if the connector is running or not
        """
        if self._proc is not None:
            return self._proc.poll() is None
        elif self._procpid is not None:
            if psutil.pid_exists(self._procpid) is True:
                return True
        return False

class ProcessContainerManager(AbstractContainerManager):
    '''
    Container Manager implementation that uses processes to host connectors. A container manager provides isolation, resource
    constraints(such as memory, cpu & storage limits) for connectors

    '''
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(ProcessContainerManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = ProcessContainerManager(*args)
        return cls.__singleton


    def __init__(self, config, languageRuntimes, rtx=None):
        '''
        Initialize a container
        '''
        self._config = config
        self._languageRuntimes = languageRuntimes
        self._rtx = rtx
        self.rootPath = config.get("process-container", "root")
        self.rootPath = os.path.abspath(self.rootPath)
        self.appDir = config.get("app-settings", 'appDirName')
        self._appdata_dir = "appdata"
        if config.has_option("app-settings", "appDataDir"):
            self._appdata_dir = config.get("app-settings", "appDataDir")
        self._create_user = True

        # Check for chroot requirements
        try:
            ch_req = 'auto_detect'
            self._data_volume = None
            self._data_mount_point = None
            self._cartridge_mount_point = None
            ch_req = config.get("process-container", "chroot_require")
            if ch_req == 'enabled':
                self._chroot_require = True
            elif ch_req == 'disabled':
                self._chroot_require = False
                self._chroot_path = None
                self._chroot_mount_paths = None
        except Exception:
            ch_req = 'auto_detect'

        if not ch_req == 'disabled' :
            # Get the configured chroot path
            if config.has_option("process-container", "chroot_path") :
               self._chroot_path = config.get("process-container", "chroot_path")
               self._chroot_require = True
            else:
                #autodetect chroot binary
                ch_path = Utils.which("chroot")
                if ch_path is not None:
                    self._chroot_path = ch_path
                    self._chroot_require = True
                else :
                #Raise Exception if chroot binary not found and chroot is configured
                    if self._chroot_require is True:
                        raise Exception("chroot configured but chroot binary not found")
                    self._chroot_require = False

        # Get the cgroup enabled or disabled
        self.enable_cgroup = Utils.getSystemConfigValue("process-container", "enable_cgroup", False, "bool")
        _cgroup_controllers = Utils.getSystemConfigValue("process-container", "cgroup_controllers", "memory,cpu")
        self.cgroup_controllers = _cgroup_controllers.strip().split(",")
        #Get the mount dir for the chroot_env
        if self._chroot_require is True:
            if config.has_option("process-container", "chroot_mount_paths"):
                self._chroot_mount_paths = config.get("process-container", 
                                                    "chroot_mount_paths")
            else:
                self._chroot_mount_paths = "/bin,/usr/bin,/lib,/usr/lib,/proc,/dev,/etc"
            if platform.machine() == 'x86_64':
                #mount /usr/lib64 and /lib64 for 64 bit machines
                if not '/lib64' in self._chroot_mount_paths:
                    self._chroot_mount_paths = self._chroot_mount_paths + ",/lib64"
                if not '/usr/lib64' in self._chroot_mount_paths:
                    self._chroot_mount_paths = self._chroot_mount_paths + ",/usr/lib64"
            if config.has_option("process-container", "data_container_mount_point"):
                self._data_mount_point = config.get("process-container", 
                                                "data_container_mount_point")
            else:
                self._data_mount_point = "/data" 

            if config.has_option("process-container", "cartridge_mount_point"):
                self._cartridge_mount_point = config.get("process-container", 
                                                "cartridge_mount_point")
            else:
                self._cartridge_mount_point = "/opt/cisco/runtimes" 

        if config.has_option("process-container", "create_user"):
            create_user = config.get("process-container", "create_user")
            if create_user == "disabled":
                self._create_user=False

        #Ensure that the root directory exists.
        if (os.path.exists(self.rootPath)):
            if (not os.path.isdir(self.rootPath)):
                raise Exception("Repository path exists and is not a directory")
        else:
            os.makedirs(self.rootPath)
        log.info("Repository initialized in the directory:%s", self.rootPath)

        if config.has_option("process-container", "data_volume_root"):
            self._data_volume = config.get("process-container", 
                                                "data_volume_root")
            self._data_volume = os.path.abspath(self._data_volume)
        else:
            self._data_volume = os.path.join(self.rootPath, "data")
        self.data_root = self._data_volume
        #Ensure that the data directory exists.
        if (os.path.exists(self._data_volume)):
            if (not os.path.isdir(self._data_volume)):
                raise Exception("Data path exists and is not a directory")
        else:
            os.makedirs(self._data_volume)

        # Read tmp location from system config, default to /tmp if not specified
        self.tmpUploadDir = '/tmp'
        if self._config.has_option("controller", "upload_dir"):
            self.tmpUploadDir = self._config.get("controller", "upload_dir")

        if not os.path.exists(self.tmpUploadDir):
            os.makedirs(self.tmpUploadDir)
        self._containers = {}

    @property
    def get_tmpdir(self):
        return self.tmpUploadDir

    @property
    def chroot_require(self):
        return self._chroot_require

    @property
    def chroot_path(self):
        return self._chroot_path

    @property
    def chroot_mount_paths(self):
        return self._chroot_mount_paths

    @property
    def data_volume(self):
        return self._data_volume

    @property
    def data_mount_point(self):
        return self._data_mount_point

    @property
    def cartridge_mount_point(self):
        return self._cartridge_mount_point

    @property
    def create_user(self):
        return self._create_user

    def _report_crash(self, container_id):
        """Called by the container monitor to report a container crash"""
        # Looks like a process container crashed. Post an event to notification service
        # if available.
        if self._rtx:
            ns = self._rtx.get_service("notification-service")
            if ns:
                log.debug("Posting a 'crashed' event on container %s" % str(container_id))
                event_message = "Container %s crashed  source : %s" % (str(container_id), CAFEvent.SOURCE_CONTAINER)
                event_severity =  logging.getLevelName(logging.ERROR)

                ns.post_event(CAFEvent(container_id,
                              CAFEvent.TYPE_CRASHED,
                              CAFEvent.SOURCE_CONTAINER,
                              event_severity=event_severity,
                              event_message=event_message))



    def inprocess(self):
        """    
        Indicates if the containers are managed in-process
        """
        return False

    def supportsAppType(self, apptype):
        """
        Indicates if this container supports hosting the specified application type.
        """
        if apptype == AppType.PAAS:
            return True
            
        return False

    def verifyReconcile(self, containerId):
        folderPath = os.path.join(self.rootPath, containerId)
        if not os.path.exists(folderPath):
            return False
        else:
            log.debug("A container already exists with the same id %s",containerId)
            #create the set of mountpoints to be verified.
            mntpaths = []
            if not self._chroot_mount_paths is None:
                for mntdir in self._chroot_mount_paths.split(",") :
                    if os.path.exists(mntdir):
                        tgmntpath = os.path.join(self.rootPath, (containerId + mntdir))
                        mntpaths.append(tgmntpath)
            appdir = os.path.join(self.rootPath, containerId)
            if self._data_mount_point is not None:
                tgmntpath = appdir + self._data_mount_point
                mntpaths.append(tgmntpath)
            datadir = None
            if self._data_volume is not None:
                datadir = os.path.join(self._data_volume, containerId)
                mntpaths.append(datadir)
            

            #verify the mounts
            mountexists = True
            for mntpnt in mntpaths:
                if not Utils.ismount_exists(mntpnt):
                    mountexists = False
                    break

            #verify the container pid
            pidexists = True
            pidpath = os.path.join(self.rootPath, containerId, ".pid")
            if os.path.exists(pidpath):
                with open(pidpath, "r") as f:
                    data = f.read()
                    proc = int(data)
                    if proc is not None:
                        if psutil.pid_exists(proc) is False:
                            pidexists = False
            else:
                pidexists = False

        # Reconcile only if the container pid and all mounts exists
        # Else cleanup and bring the new process container
        if pidexists and mountexists:
            return True
        elif pidexists and not mountexists:
            self.cleanupReconciler(appdir, datadir, str(proc), [])
        elif mountexists and not pidexists:
            self.cleanupReconciler(appdir, datadir, None, mntpaths)
        else:
            self.cleanupReconciler(appdir, datadir, None)
       
        return False

    # This is cleanup and need to remove mounts, container pid,
    # and necessary directories
    def cleanupReconciler(self, appdir, datadir, pid = None, mntpaths = []):
        if pid:
            kill(pid)
        for mnt in mntpaths:
            out, rc = umount("-l",  mnt)
            if rc != 0:
                log.error("umount failed, error %s", str(out))
        if appdir is not None:
            shutil.rmtree(appdir)
        if datadir is not None:
            shutil.rmtree(datadir)

    def create(self, containerRequest):
        """
        Creates a logical container
        """
        folderPath = os.path.join(self.rootPath, containerRequest.containerId)
        reconcile = False
        if self.verifyReconcile(containerRequest.containerId) is True:
            reconcile = True
        else:
            if not os.path.exists(folderPath):
                os.mkdir(folderPath)
        container = None
        
        try:
            if self._chroot_require:
                self._create_data_dir(containerRequest._disk_ext, containerRequest._containerId, containerRequest.use_ext4)
            container = self._createContainerInstance(containerRequest.containerId, folderPath, containerRequest.appmanifest.resources, containerRequest.appconfigname, containerRequest.appmanifest.app_env, reconcile)
            self._containers[containerRequest.containerId] = container
            if self.enable_cgroup:
                container.cgroup_name = self._create_container_cgroup(container)
            container._copyArchive(containerRequest.connectorArchivePath, reconcile)
            container._addCartridge(containerRequest.cartridge_list)
            container._addStartCommand(containerRequest.startCommand)
            container._addStopCommand(containerRequest.stopCommand)
            container._addCustomScripts(containerRequest.customScripts)
            container._addCustomLogFiles(containerRequest.customLogFiles)
            
        except:
            if container:
                self.destroy(containerRequest.containerId)
            raise
        return container

    def _create_container_cgroup(self, container):
        """
        Will create the cgroup and sets the limits given in descriptor file.
        """
        log.debug("Creating the cgroup for the container %s"%container.getId())
        user = None
        group_name = Utils.getSystemConfigValue("cgroup-settings", "cgroup_parent_name", default="apphosting.partition")
        group_name = os.path.join("/", group_name, container.getId())
        if self.create_user:
            user = container._userId
        if self.enable_cgroup:
            out, ret = create_cgroup(group_name, self.cgroup_controllers, user)
            if ret != 0:
                log.error("Error while creating cgroups for %s, cause: %s" % (container.getId(), out))
                raise Exception("Error while creating cgroups for %s, cause: %s" % (container.getId(), out))
            log.debug("Sucessfully created the cgroup for the container %s, with group name %s"%(container.getId(), group_name))
            resources = container.app_resources.copy()
            # Converting the memory in bytes.
            resources["memory"] = int(resources["memory"]) * 1024 * 1024
            control_key_map = {"memory": "limit_in_bytes",
                                "cpu": "shares"}
            for controller, attr in control_key_map.iteritems():
                out, ret = set_cgroup_control_limit(group_name, controller, attr, resources[controller])
                if ret != 0:
                    log.error("Error while setting the cgroup limit for the group %s, cause: %s" % (container.getId(), out))
                    raise Exception("Error while setting the cgroup limit for the group %s, cause: %s" % (container.getId(), out))
                log.debug("Successfully set the limit for cgroup controller %s value %s" % (controller, resources[controller]))
            return group_name

    def _create_data_dir(self, data_ext, containerId, use_ext4=False):
        """
        Will mount given ext disk to the containers data location
        """
        if use_ext4:
            log.debug("create data dir ext4, data_ext %s", data_ext)
            cmdargs = ["-t", "ext4", "-o"]
        else:
            log.debug("create data dir ext2, data_ext %s", data_ext)
            cmdargs = ["-t", "ext2", "-o"]
        mtpnt = os.path.join(self._data_volume, containerId)
        if not os.path.isdir(mtpnt):
            os.makedirs(mtpnt)
        rw = "loop,rw"
        cmdargs.append(rw)
        cmdargs.extend([data_ext, mtpnt])
        if not Utils.ismount_exists(mtpnt):
            out, rc = mount(cmdargs)
            if rc != 0:
                log.error("Error in creating data directory: %s", str(out))
                raise AppInstallationError("Error in creating data directory: %s", str(out))
        log.debug("Created data directory for container %s at %s" % (containerId, mtpnt))

    def get(self, containerId):
        """
        Returns a logical container given a container id
        """
        try:
            return self._containers[containerId]
        except KeyError:
            pass
        return None
    
    def list(self):
        """
        List known logical containers
        """
        return iter(self._containers)
    
    def stop(self, graceful=True):
        """
        Stop all the containers
        """
        for container_id, container in self._containers.iteritems():
            container.stop(graceful=graceful)
        self._containers.clear()

    def destroy(self, containerId, is_getting_shutdown=False):
        """
        Destroy a logical container
        Returns True if successful
        """
        container = self.get(containerId)
        # Unmount data directory created for this container
        log.debug("unmounting the persistent data dir for container %s , data root %s", containerId, self._data_volume)
        if Utils.ismount_exists(os.path.join(self._data_volume, containerId)):
            out, rc = umount("-l",  os.path.join(self._data_volume, containerId))
            if rc != 0:
                log.error("Unmounting failed for data dir. ret code: %s error: %s"
                                % (rc, str(out)))
        if container:
            # Delete the if cgroup is created for the container
            if self.enable_cgroup:
                if container.cgroup_name:
                    delete_cgroup(container.cgroup_name, self.cgroup_controllers)
            if container.chroot_require is True:
                mntpaths = []
                if not self._chroot_mount_paths is None:
                    for mntdir in self._chroot_mount_paths.split(",") :
                        if os.path.exists(mntdir):
                            tgmntpath = container._containerDir + mntdir
                            mntpaths.append(tgmntpath)
                            out, rc = umount("-l",  tgmntpath)
                            if rc != 0:
                                log.error("umount failed, error %s", str(out))

                if not self._data_mount_point is None:
                    tgmntpath = container._containerDir + self._data_mount_point
                    mntpaths.append(tgmntpath)
                    out, rc =umount("-l", tgmntpath)
                    if rc != 0:
                        log.error("umount failed, error %s", str(out))

                if len(container._cartridge_list) > 0 and self._cartridge_mount_point:
                    for cart in container._cartridge_list:
                        if cart.type == "baserootfs":
                            continue
                        if not "mountable" in cart.handleas:
                            continue

                        tgmntpath = container._containerDir + self._cartridge_mount_point + os.sep + cart.id
                        mntpaths.append(tgmntpath)
                        out, rc =umount("-l", tgmntpath)
                        if rc != 0:
                            log.error("umount failed, error %s", str(out))
                
                # Make sure all mount paths are deleted before proceeding to delete 
                # container otherwise container will not be deleted
                for mntpath in mntpaths:
                    cnt = 0
                    while cnt < 5:
                        #wait for 3 seconds to wait if path is umounted
                        if os.listdir(mntpath) != []:
                            time.sleep(3)
                            cnt += 1
                        else: 
                            break
                    if cnt >= 5 :
                        log.error("Cannot umount %s" % mntpath)
                        raise Exception("Cannot umount %s for container %s" % (mntpath, containerId)) 
            if not self._data_volume is None:
                data_volume_path = os.path.join(self._data_volume, containerId)
                log.debug("removing data for container %s" % data_volume_path)
                shutil.rmtree(data_volume_path)
            log.debug("Process container: '%s' will be destroyed." % containerId)
            shutil.rmtree(container._containerDir)
            self._containers.pop(containerId)
            
            if self._create_user: 
                # Delete the conatiner user
                userId = containerId.encode('base64', 'strict').strip().translate(None,'+/=').lower()
                userId = userId[:32]
                out, rc = deluser(userId)
                if rc != 0:
                    log.error("Deletion of user failed %s. Error:%s" 
                                                    % (userId, str(out)))
            #remove the settings file
            settingsFilePath = self._settingsFilePath(containerId)
            if os.path.exists(settingsFilePath):
                os.remove(settingsFilePath)

            return True

    def _createContainerInstance(self, containerId, containerPath, resources={}, appconfigfile="", app_env={}, reconcile=False):
        """
        creates an instance of container
        """
        if reconcile:
            return ProcessContainerReconcile(containerId, containerPath, self, resources, appconfigfile, app_env)
        else:
            return ProcessContainer(containerId, containerPath, self, resources, appconfigfile, app_env)
    
    def _settingsFilePath(self, containerId):
        return  os.path.join(self.rootPath, containerId+".ini")
    
    def get_app_config_path(self, containerId):
        folderPath = os.path.join(self._data_volume, containerId)
        cfgfile = os.path.join(folderPath, Utils.find_app_config_filename(folderPath))
        if os.path.isfile(cfgfile):
            return cfgfile
        return ""

    def get_app_config(self, containerId):
        #folderPath = os.path.join(self.rootPath, containerId)
        folderPath = os.path.join(self._data_volume, containerId)
        #appContentDir = self.appDir + "/app"
        #cfgfile = os.path.join(folderPath, appContentDir, Utils.find_app_config_filename(os.path.join(folderPath, appContentDir)))
        cfgfile = os.path.join(folderPath, Utils.find_app_config_filename(folderPath))
        return ContainerUtils.get_app_config(cfgfile)


    def set_app_config(self, containerId, content):
        # folderPath = os.path.join(self.rootPath, containerId)
        folderPath = os.path.join(self._data_volume, containerId)
        #appContentDir = self.appDir + "/app"
        #cfgfile = os.path.join(folderPath, appContentDir, Utils.find_app_config_filename(os.path.join(folderPath, appContentDir)))
        cfgfile = os.path.join(folderPath, Utils.find_app_config_filename(folderPath))
        ContainerUtils.set_app_config(folderPath, cfgfile, content)

    def add_data_file(self, containerId, data_file_name, data_file):
        data_volume_path = os.path.join(self._data_volume, containerId)
        if data_file_name is None or data_file_name == "":
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        #Append appdata
        cshared = os.path.join(data_volume_path, self._appdata_dir)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))
        FileMgmt.add_data_file(data_file, dstfilepath, data_file_name)

    def get_data_file(self, containerId, data_file_name):
        """
        Reads the content of data_file_name
        if it is directory:
            Returns dictionary with key as dirlist and Value is the list of file dict 
            e.g data["dirlist] = [ 
                name: "a.txt", type: "file", path: "appdata/a.txt", size: 100, last_modified_time: 1-Jan-15),
                name: "dir1/", type: "dir", path: "appdata/dir1", size: 0, last_modified_time: 2-Feb-15
            ]
        if data_file_name is regular file:
            Returns the generator function that reads data  contents of data_file_name 
        """

        data_volume_path = os.path.join(self._data_volume, containerId)

        if data_file_name is None :
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        cshared = os.path.join(data_volume_path, self._appdata_dir)
        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))
        return FileMgmt.get_data_file(dstfilepath, data_file_name)

    def delete_data_file(self, containerId, data_file_name):
        """
        Delete the data_file_name from app appdata directory
        """
        data_volume_path = os.path.join(self._data_volume, containerId)

        if data_file_name is None or data_file_name == "" :
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        #Append appdata
        cshared = os.path.join(data_volume_path, self._appdata_dir)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))

        if dstfilepath == cshared:
            #Deleting the root appdata not allowed
            log.error("Cannot delete the root %s" % data_file_name)
            raise ValueError("Cannot delete the root %s" % data_file_name)
            
        FileMgmt.delete_data_file(dstfilepath, data_file_name)

