__author__ = 'suressan'

import logging
import json
import os
import falcon
from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.infraexceptions import MandatoryDataMissingError
from jsonencoder import JSONEncoder
from ..sla.agent_internals import SmartAgentReturnCodes as RetCode

log = logging.getLogger("runtime.api.resources")
jsonencoder = JSONEncoder()

@ResourceRoute("/platform/license/config", endpoint="license_config")
class SmartLicenseConfigResource(AuthenticatedResource):
    def on_put(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_400)
            return
        try:
            license_config = json.load(request.stream)
            smart_lic_service.set_config(license_config)
            response = make_response(response, 'Successfully set license configuration', falcon.HTTP_200)
        except Exception as ex:
            response = make_error_response(response,
                                           ex.message,
                                           "Error while updating IOx license configuration",
                                           http_status_code=falcon.HTTP_500)
            log.exception("Error while updating IOx license configuration")

    def on_get(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_400)
            return
        try:
            out = jsonencoder.encode(smart_lic_service.get_config())
            response.status = falcon.HTTP_200
            response.body = out
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        except Exception as ex:
            response = make_error_response(response, ex.message,
                                            "Error while retrieving IOx license configuration",
                                            http_status_code=falcon.HTTP_500)
            log.exception("Error occured while retrieving IOx license configuration")

@ResourceRoute("/platform/license/dev_cert", endpoint="license_devcert")
class SmartLicenseDevCertResource(AuthenticatedResource):
    def on_put(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_500)
            return
        try:
            payload = json.load(request.stream)
            dev_cert_enable = payload["enabled"]
            ret_code = smart_lic_service.sla_setup_testrootca(int(dev_cert_enable))
            ret_code_str = smart_lic_service.sla_return_code_to_str(ret_code)
            if ret_code == RetCode.SmartAgentSuccess:
                # change the response messge to include either enabled/disabled
                log.info("Successfully %s smart license development certificate",
                         ("enabled" if dev_cert_enable else "disabled"))
                response = make_response(response,
                                         "Successfully %s smart license development certificate" %
                                         ("enabled" if dev_cert_enable else "disabled"),
                                         falcon.HTTP_200)
            else:
                log.error("Failed to %s smart license development certificate %s" %
                          (("enable" if dev_cert_enable else "disable"), ret_code_str))
                response = make_error_response(response,
                                         "Failed to %s smart license development certificate, error = %s"
                                         % (("enable" if dev_cert_enable else "disable"), ret_code_str),
                                         "IOx license dev cert management failed",
                                         falcon.HTTP_500)
        except Exception as ex:
            response = make_error_response(response,
                      "Failed to %s IOx license dev certificate: %s" %
                      (("enable" if dev_cert_enable else "disable"), str(ex)),
                      "Failed to %s IOx license dev certificate" % ("enable" if dev_cert_enable else "disable"),
                      falcon.HTTP_500)
            # Here is the link for all error code descriptions
            # http://elo-repo:8080/sla-apis-latest/SmartAgentCommonAPI_8h_source.html
            log.exception("Exception occured while managing IOx license dev certificate")

@ResourceRoute("/platform/license/register", endpoint="license_register")
class SmartLicenseRegisterResource(AuthenticatedResource):
    def on_post(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_500)
            return
        try:
            # continue with calling regisration
            alpha_backend = False
            payload = json.load(request.stream)
            reg_id_token = payload["reg_id_token"]
            force_flag = False
            if "force" in payload:
                force_flag = payload["force"]
            ret_code = smart_lic_service.sla_register(reg_id_token, force_flag)
            ret_code_str = smart_lic_service.sla_return_code_to_str(ret_code)
            if ret_code == RetCode.SmartAgentRegistrationInProgress:
                log.info("Smart license registration is initiated and in progress.")
                response = make_response(response,
                                         "Initiated IOx license registration and currently in progress.",
                                         falcon.HTTP_200)
            elif ret_code == RetCode.SmartAgentErrorAlreadyRegistered:
                log.info("Smart license capability is already registered")
                response = make_response(response,
                                         "Already registered IOx with license manager portal",
                                         falcon.HTTP_200)
            else:
                log.error("Smart license registration failed with error - %s", ret_code_str)
                response = make_error_response(response,
                                         "IOx license registration failed with error - %s" % ret_code_str,
                                         "IOx license registration failed",
                                         falcon.HTTP_500)

        except Exception as ex:
            response = make_error_response(response,
                      "IOx license registration failed: %s" % str(ex),
                      "IOx license registration failed",
                      falcon.HTTP_500)
            # Here is the link for all error code descriptions
            # http://elo-repo:8080/sla-apis-latest/SmartAgentCommonAPI_8h_source.html
            log.exception("Exception occured while registring the device with cloud license manager")

    def on_delete(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_400)
            return
        try:
            ret_code = smart_lic_service.sla_deregister()
            ret_code_str = smart_lic_service.sla_return_code_to_str(ret_code)
            if ret_code == RetCode.SmartAgentSuccess:
                log.info("Successfully deregistered smart licensing agent")
                response = make_response(response,
                                         "Successfully deregistered smart licensing agent",
                                         falcon.HTTP_200)
            elif ret_code == RetCode.SmartAgentErrorNotRegistered:
                log.info("Already deregistered, return code = %s", ret_code)
                response = make_response(response,
                                         "IOx device is already deregistered.",
                                         falcon.HTTP_200)
            else:
                log.error("Failed to deregister smart licensing agent, ret code = %s", ret_code)
                response = make_error_response(response,
                                         "Failed to deregister smart licensing agent with error - %s" % ret_code_str,
                                         "IOx license deregistration failed",
                                         falcon.HTTP_500)
        except Exception as ex:
            response = make_error_response(response,
                      "Failed to deregister IOx with license manager portal: %s" % str(ex),
                      "Failed to deregister IOx with license manager portal",
                      falcon.HTTP_500)
            log.exception("Exception occured while retrieving trying to deregister IOx with license manager portal")

@ResourceRoute("/platform/license/renew/id", endpoint="license_renew_idcert")
class SmartLicenseRenewIdResource(AuthenticatedResource):
    def on_post(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_500)
            return
        try:
            ret_code = smart_lic_service.sla_renew_idcert()
            ret_code_str = smart_lic_service.sla_return_code_to_str(ret_code)
            if ret_code == RetCode.SmartAgentSuccess or ret_code == RetCode.SmartAgentIdCertRenewInProgress:
                log.info("Successfully initiated the renewal of IOx license ID certificate")
                response = make_response(response,
                                         "Successfully initiated the renewal of IOx license ID certificate",
                                         falcon.HTTP_200)
            else:
                log.error("Failed to initiate renewal of IOx license ID certificate %s",
                          ret_code_str)
                response = make_error_response(response,
                                         "Failed to initiate renewal of IOx license ID certificate, error description = %s" %
                                         ret_code_str,
                                         "IOx license ID cert renewal failed",
                                         falcon.HTTP_500)
        except Exception as ex:
            response = make_error_response(response,
                      "Failed to initiate renewal of IOx license ID certificate: %s" % str(ex),
                      "Failed to initiate renewal of IOx license ID certificate",
                      falcon.HTTP_500)
            log.exception("Exception occured while renewing IOx license ID certificate")

@ResourceRoute("/platform/license/renew/auth", endpoint="license_renew_auth")
class SmartLicenseRenewAuthResource(AuthenticatedResource):
    def on_post(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_500)
            return
        try:
            ret_code = smart_lic_service.sla_renew_auth()
            ret_code_str = smart_lic_service.sla_return_code_to_str(ret_code)
            if ret_code == RetCode.SmartAgentSuccess or ret_code == RetCode.SmartAgentAuthRenewInProgress:
                log.info("Successfully initiated the renewal of IOx license authorization")
                response = make_response(response,
                                         "Successfully initiated the renewal of IOx license authorization",
                                         falcon.HTTP_200)
            else:
                log.error("Failed to initiate renewal of IOx license authorization %s",
                          ret_code_str)
                response = make_error_response(response,
                                         "Failed to initiate renewal of IOx license authorization, error code = %s" %
                                         ret_code_str,
                                         "IOx license authorization renewal failed",
                                         falcon.HTTP_500)
        except Exception as ex:
            response = make_error_response(response,
                      "Failed to initiate renewal of IOx license authorization: %s" % str(ex),
                      "Failed to initiate renewal of IOx license authorization",
                      falcon.HTTP_500)
            log.exception("Exception occured while renewing IOx license authorization")

@ResourceRoute("/platform/license/log/level", endpoint="license_loglevel")
class SmartLicenseLogResource(AuthenticatedResource):
    def on_put(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_400)
            return
        try:
            payload = json.load(request.stream)
            level = payload["level"]
            enabled = payload["enabled"]
            if enabled:
                ret_code = smart_lic_service.sla_debug_set(level)
                smart_lic_service.set_config({"loglevel": level})
            else:
                ret_code = smart_lic_service.sla_debug_clear(level)
                # While clearing loglevel, by default reset the running config to ERROR log level
                smart_lic_service.set_config({"loglevel": "error"})

            ret_code_str = smart_lic_service.sla_return_code_to_str(ret_code)
            if ret_code == RetCode.SmartAgentSuccess:
                # return msg with enabled or disabled
                log.info("Successfully %s smart license debug flag - %s", (("turned ON" if enabled else "turned OFF"), level))
                response = make_response(response,
                                         "Successfully %s debug flag" % ("turned ON" if enabled else "turned OFF"),
                                         falcon.HTTP_200)
            else:
                log.error("Failed to %s debug flag - %s", ("turn ON" if enabled else "turn OFF",
                                                           ret_code_str))
                response = make_error_response(response,
                                         "Failed to %s debug flag - %s" % ("turn ON" if enabled else "turn OFF",
                                                    ret_code_str),
                                         "Failed to configure debug flag",
                                         falcon.HTTP_500)
        except Exception as ex:
            response = make_error_response(response,
                      "Failed to %s debug flag - %s" % ("turn ON" if enabled else "turn OFF", level),
                      "Failed to configure debug flag",
                      falcon.HTTP_500)
            log.exception("Exception occured while configuring smart license debug flag")

@ResourceRoute("/platform/license/test/expire", endpoint="license_testexpire")
class SmartLicenseTestExpireResource(AuthenticatedResource):
    def on_post(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_400)
            return
        try:
            payload = json.load(request.stream)
            jobname = payload["name"]
            expire_in = int(payload["expire-in"])
            ret_code = smart_lic_service.sla_test_expire_job(jobname, expire_in)
            ret_code_str = smart_lic_service.sla_return_code_to_str(ret_code)
            if ret_code == RetCode.SmartAgentSuccess or ret_code == RetCode.SmartAgentAuthRenewInProgress or ret_code == RetCode.SmartAgentIdCertRenewInProgress:
                log.info("Successfully set test expiry for job - %s", jobname)
                response = make_response(response,
                                         "Successfully set test expiry for job - %s" % jobname,
                                         falcon.HTTP_200)
            elif ret_code == RetCode.SmartAgentErrorNotSupported:
                log.error("Failed to set test expiry for job - %s. Expiry is not supported with current state of license", jobname)
                response = make_error_response(response,
                                         "Expiry is not supported with current state of license",
                                         "Failed to set test expiry for job - %s" % jobname,
                                         falcon.HTTP_500)
            else:
                log.error("Failed to set test expiry for job - %s", jobname)
                response = make_error_response(response,
                                         "Failed to set test expiry for job - %s, return code - %s" % (jobname, ret_code_str),
                                         "Failed to set test expiry for job",
                                         falcon.HTTP_500)
        except Exception as ex:
            response = make_error_response(response,
                      "Failed to set test expiry for job - %s" % str(ex),
                      "Failed to set test expiry for job",
                      falcon.HTTP_500)
            log.exception("Exception occured while setting test expiry for job - %s" % (ret_code_str))

@ResourceRoute("/platform/license/test/eval_reset", endpoint="license_evalreset")
class SmartLicenseEvalResetResource(AuthenticatedResource):
    def on_post(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_400)
            return
        try:
            ret_code = smart_lic_service.sla_test_reset_eval_period()
            ret_code_str = smart_lic_service.sla_return_code_to_str(ret_code)
            if ret_code == RetCode.SmartAgentSuccess:
                log.info("Successfully reset evaluation period")
                response = make_response(response,
                                         "Successfully reset evaluation period",
                                         falcon.HTTP_200)
            else:
                log.error("Failed to reset evaluation period - %s", ret_code_str)
                response = make_error_response(response,
                                         "Failed to reset evaluation period - %s" % (ret_code_str),
                                         "Failed to reset evaluation period",
                                         falcon.HTTP_500)
        except Exception as ex:
            response = make_error_response(response,
                      "Failed to reset evaluation period - %s" % str(ex),
                      "Failed to reset evaluation period",
                      falcon.HTTP_500)
            log.exception("Exception occured while resetting evaluation period")

@ResourceRoute("/platform/license", endpoint="license")
class SmartLicenseShowResource(AuthenticatedResource):
    def on_get(self, request, response):
        smart_lic_service = APIService.instance.smart_license
        if smart_lic_service is None:
            response = make_error_response(response,
                      "License management is not enabled on this device",
                      "License management is not enabled on this device",
                      falcon.HTTP_400)
            return
        try:
            show_type = request.get_header('X-Show-Type')
            if show_type == "summary":
                output = smart_lic_service.show_license_summary()
            elif show_type == "all":
                output = smart_lic_service.show_license_all()
            elif show_type == "udi" or show_type == "UDI":
                output = smart_lic_service.show_license_udi()
            elif show_type == "usage":
                output = smart_lic_service.show_license_usage()
            elif show_type == "techsupport":
                output = smart_lic_service.show_license_techsupport()
            else:
                output = smart_lic_service.show_license_status()
            log.debug("sending output = %s", output)
            response.set_header("Content-Type", "text/plain")
            response = make_response(response, output, falcon.HTTP_200)
            response.set_header("Cache-Control", "no-cache")
        except Exception as ex:
            response = make_error_response(response,
                      "Failed to get license status: %s" % str(ex),
                      "Failed to get license status",
                      falcon.HTTP_500)
            log.exception("Exception occured while retrieving license status")
