'''
Created on Sep 17, 2014

@author: yaulee

Copyright (c) 2014-2015 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import os
import sys
import time
import subprocess
import ConfigParser
from ..utils.infraexceptions import C3Exception
 
log = logging.getLogger("runtime")

PERS_STATUS_PARAM = 'status'

class MWStatus(object):
    """Middleware persistent status can take one of these values"""
    RUNNING = "RUNNING"
    STOPPED = "STOPPED"

class MiddlewareServices(object):

    __singleton = None

    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super(MiddlewareServices, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, configfile=None):
        self._servicesList = {}
        if configfile is None:
            from apiservice import APIService
            if APIService.instance.config.has_section("middleware"):
                configfile = APIService.instance.config.get("middleware", "config")
                statusfile = APIService.instance.config.get("middleware", "mwstate")
            else:
                raise C3Exception("Middleware services are not configured.")
                
        self._mwPersist = MWPersistentStatus(statusfile)
            
        config = ConfigParser.SafeConfigParser()
        config.read(configfile)
        for sec_name in config.sections():
            service = {}
            service["id"] = sec_name
            for item_name, data in config.items(sec_name):
                service[item_name] = data
            self._servicesList[sec_name] = service
            
            # Start Middleware services that are in running state
            if self._mwPersist.getStatus(sec_name) == MWStatus.RUNNING:
                self.start(sec_name)
            
            

    @classmethod
    def getInstance(cls):
        '''
        Returns the singleton instance of MiddlewareServices.
        '''
        if cls.__singleton is None:
            cls.__singleton = MiddlewareServices()

        return cls.__singleton

    def getServiceInfo(self, service_id):
        if self._servicesList.has_key(service_id):
            return self._servicesList[service_id]
        else:
            return None

    def start(self, service_id):
        if self.getServiceInfo(service_id) == None:
            log.error("Service not found: %s", service_id)
            return False

        ret = subprocess.call(["monit", "start", service_id])
        if ret != 0:
            log.error("Error return from monit: %s" % ret)
        else:
            # try to get status and determine if the process is up and running
            # since monit will retry 5 times, we do the same thing here in 5 seconds interval.
            for number in range(5):
                result = subprocess.check_output("monit summary | awk '/%s/ {$1=$2=\"\"; sub(/^ */,\"\",$0); print $0}'" % service_id, shell=True)
                if result.startswith('Running'):
                    #Set status in statusfile to RUNNING (if not already)
                    self._mwPersist.setStatus(service_id, MWStatus.RUNNING)
                    return True
                time.sleep(5)
            log.error("Start failed in 5 trys: %s" % service_id)
 
        return False

    def stop(self, service_id):
        if self.getServiceInfo(service_id) == None:
            log.error("Service not found: %s", service_id)
            return False

        ret = subprocess.call(["monit", "stop", service_id])
        if ret != 0:
            log.error("Error return from monit: %s" % ret)
        else:
            # try to get status and determine if the process is up and running
            # since monit will retry 5 times, we do the same thing here in 5 seconds interval.
            for number in range(5):
                result = subprocess.check_output("monit summary | awk '/%s/ {$1=$2=\"\"; sub(/^ */,\"\",$0); print $0}'" % service_id, shell=True)
                if result.startswith('Not monitored'):
                    #Set status in statusfile to STOPPED (if not already)
                    self._mwPersist.setStatus(service_id, MWStatus.STOPPED)
                    return True
                time.sleep(5)
            log.error("Stop failed in 5 trys: %s" % service_id)
 
        return False

    def status(self, service_id):
        if self.getServiceInfo(service_id) == None:
            log.error("Service not found: %s", service_id)
            return False

        result = subprocess.check_output("monit summary | awk '/%s/ {$1=$2=\"\"; sub(/^ */,\"\",$0); print $0}'" % service_id, shell=True)
        serviceInfo = self._servicesList[service_id]
        # map monit status to be more user friendly
        if result.startswith('Not monitored'):
            serviceInfo['status'] = "Stopped"
        else:
            serviceInfo['status'] = result.rstrip()

        return serviceInfo

    def get_all_status(self):
        serviceInfos = []
        for service_id in self._servicesList.keys():
            serviceInfos.append(self.status(service_id))

        return serviceInfos
        
        
class MWPersistentStatus(object):

    def __init__(self, statusfile):
    
        self._statusfile = statusfile
        self._createStatusFile(statusfile)
        
    
    def _createStatusFile(self, statusfile):
        if not os.path.isfile(statusfile):
            dname = os.path.dirname(statusfile)
            if not os.path.exists(dname):
                os.makedirs(dname)
                
            with open(statusfile, 'w', 0) as f:
                pass  
            
    def getStatus(self, service):
        
        # statusfile should be there, but in case someone deleted it. Fresh state. 
        self._createStatusFile(self._statusfile)
    
        statusParser = ConfigParser.SafeConfigParser()
        statusParser.read(self._statusfile)
        
        if not statusParser.has_section(service):
            return MWStatus.STOPPED
            
        return statusParser.get(service, PERS_STATUS_PARAM)
         
    def setStatus(self, service, status):
    
        # statusfile should be there, but in case someone deleted it. Fresh state.   
        self._createStatusFile(self._statusfile)
    
        if self.getStatus(service) == status:
            # No change
            return
    
        statusParser = ConfigParser.SafeConfigParser()
        statusParser.read(self._statusfile)
        
        if not statusParser.has_section(service):
            statusParser.add_section(service)
            
        statusParser.set(service, PERS_STATUS_PARAM, status)
        
        with open(self._statusfile, 'w', 0) as f:
            statusParser.write(f)
    

if __name__ == '__main__':

    service = MiddlewareServices("config/middleware-config.ini")
    print service.status('dmo')
    print service.get_all_status()
    print service.start('dmo')


