#-----------------------------------------------------
# Common resource utils
# Created on Dec 3rd, 2012
#
# @author: rnethi
#
# Copyright (c) 2012-2013 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

'''
 Common resource utils
'''

import falcon
import os
import mimetypes
from jsonencoder import JSONEncoder
jsonencoder = JSONEncoder()

def get_falcon_status_code_map():
    """ Return a map of HTTP_CODE-> HTTP_STRING"""
    falcon_map = {}
    import re
    expr = re.compile('^HTTP_[0-9][0-9][0-9]')
    for httpstrings in dir(falcon):
        if expr.match(httpstrings):
            httpvalue = getattr(falcon, httpstrings)
            falcon_map[int(httpvalue[0:3])] = httpvalue

    return falcon_map

def render_template(template_path, **context):
    from jinja2 import Template
    template_str = open(template_path, 'r').read()
    return Template(template_str).render(**context)


def flush_request(request):
    """
    In falcon Frame work if response is being sent without reading complete request then,
    falcon automatically reads request in to memory and then response back.
    To over come this adding this wrapper to flush the request stream before responding.
    """
    if not isinstance(request, falcon.Request):
        raise ValueError("%s is not of type falcon.Response" % str(request))
    if request.stream:
        with open(os.devnull, "wb") as f:
            while True:
                chunk = request.stream.read(4096)
                if not chunk:
                    break
                f.write(chunk)


def make_error_response(response, description = '', message = '', http_status_code=falcon.HTTP_500, errorcode = -1, headers={}):

    if not isinstance(response, falcon.Response):
        raise ValueError("%s is not of type falcon.Response" % str(response))

    response.status = http_status_code
    resp_body = {
                'errorcode' : errorcode,
                'description' : description,
                'message' : message
                }

    response.body = jsonencoder.encode(resp_body)

    for k, v in headers.items():
        response.set_header(k, v)

    response.set_header('Content-Type', "application/json")

    return response
def make_response(response, body='', code=falcon.HTTP_200, headers={}):
    """Update response object with supplied parameters
    and return the updated object"""
    if not isinstance(response, falcon.Response):
        raise ValueError("%s is not of type falcon.Response" % str(response))

    response.status = code
    response.body = body
    for k, v in headers.items():
        response.set_header(k, v)

    return response

class FileWrapper(object):
    """Wrap a file object with an iterator"""
    def __init__(self, file, buffer_size=8192):
        self.file = file
        self.buffer_size = buffer_size

    def close(self):
        if hasattr(self.file, 'close'):
            self.file.close()

    def __iter__(self):
        return self

    def next(self):
        data = self.file.read(self.buffer_size)
        if data:
            return data
        raise StopIteration()

def wrap_file(environ, file, buffer_size=8192):
    """Wrap a file object with wsgi.file_wrapper if avaialble,
    else use FileWrapper class"""
    return environ.get('wsgi.file_wrapper', FileWrapper)(file, buffer_size)


def send_file(response, filepath, send_as_attachment=False,
              attachment_filename=None,
              wsgi_environ={}):
    """
    Send a file in response.
    Automatically guess the mimetype.
    If send_as_attachment is True, set Content-Disposition header
    """
    if not os.path.isfile(filepath):
        raise falcon.HTTPNotFound()

    mimetype = mimetypes.guess_type(filepath)[0]
    if mimetype is None:
        mimetype = 'application/octet-stream'

    headers = {'Content-Type': mimetype}
    file = open(filepath, 'rb')
    headers['Content-Length'] = str(os.path.getsize(filepath))
    data = wrap_file(wsgi_environ, file)

    if send_as_attachment:
        if attachment_filename:
            attachment_name = attachment_filename
        else:
            attachment_name = os.path.basename(filepath)

        headers["Content-Disposition"] =  "attachment;filename="+attachment_name

    response.set_headers(headers)
    response.status = falcon.HTTP_200
    response.stream = data


class ResourceBase(object):
    """
    Base class for all resources
    """
    def getRootPathPrefix(self):
        import apiservice
        return apiservice.APIService.API_PATH_PREFIX


class AuthenticatedResource(ResourceBase):
    """
    Base class for all resources that need authenticated access
    Automatically enforces authentication using supported authentication
    mechanisms
    """
    # apiservice.ResourceRoute adds appropriate hooks to check for tokens
    pass

class OauthResourceValidator(ResourceBase):
    pass

class IOXResourceValidator(ResourceBase):
    pass
