__author__ = 'madawood'

import json
import urlparse
import logging
import falcon

from jsonencoder import JSONEncoder

from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_error_response
from batch_utils import simulate_request
from ..utils.infraexceptions import *
from ..utils.utils import Utils

log = logging.getLogger("runtime.api.resources")
jsonencoder = JSONEncoder()


def response_generator(falcon_app, headers, req_batch, resp_as, modus):
    for req in req_batch:
        log.debug("Request for processing is:%s" % req)
        url = str(req.get("relative_url"))
        method = str(req.get("method"))
        resp_body = {}
        try:
            if url and method:
                log.debug("Going to process the URL:%s, METHOD:%s" % (url, method))
                req_headers = req.get("headers", {})
                req_headers["X-Token-Id"] = str(headers.get("X-TOKEN-ID"))
                req_headers["X-Connector-Id"] = str(req_headers.get("X-CONNECTOR-ID"))
                # Popping the accept header, because compressed data will be binary data,
                #   and binary data can't be encode in to json object
                popped_key = [req_headers.pop(key) for key in req_headers.keys() if key.lower() == "accept-encoding"]
                url = urlparse.urlparse(url)
                result = simulate_request(app=falcon_app, path=str(url.path), headers=req_headers,
                                          body=json.dumps(req.get("body", {})), method=str(method).upper(),
                                          query_string=str(url.query))
                resp_body["body"] = result.text
                resp_body["status_code"] = result.status_code
                resp_body["headers"] = (dict(result.headers))
            else:
                log.error("Mandatory fields url/method : %s/%s are invalid" % (url, method))
                raise MandatoryDataMissingError("Mandatory fields url/method : %s/%s are invalid" % (url, method))
        except Exception as ex:
            log.exception("Error while processing the url/method %s/%s, cause: %s" % (url, method, str(ex)))
            resp_body["status_code"] = 500
            resp_body["headers"] = {}
            resp_body["body"] = ex.message
        if resp_as == "immediate":
            data = Utils.prepareDataForChunkedEncoding(json.dumps(resp_body))
            yield data
        else:
            yield resp_body
        if resp_body["status_code"] != 200 and resp_body["status_code"] != 201 and modus == "stop-on-failure":
            log.error("Error occurred for the request: %s, so stopping the batch process!" % str(url))
            return


@ResourceRoute("/batch", endpoint="batch")
class BatchRequestResource(AuthenticatedResource):
    def on_post(self, request, response):
        content_length = request.get_header('Content-Length')
        if content_length:
            content_length = int(content_length)
            content_limit = 1 * 1024 * 1024  # Setting the content limit to 1MB to prevent user from uploading huge files while activation
            if content_length > content_limit:
                response = make_error_response(response,
                                               "Exceeds the maximum content-limit %s bytes" % content_limit,
                                               "Exceeds the maximum content-limit %s bytes" % content_limit,
                                               falcon.HTTP_500)
                return
        try:
            rbody = json.loads(request.get_param('batch'))
            if isinstance(rbody, list):
                app = APIService.instance.application
                headers = request.headers
                respond_as = "immediate" if str(headers.get("X-RESPOND")).lower() == "immediate" else "collate"
                modus = "continue-on-failure" if str(
                        headers.get("X-MODUS")).lower() == "continue-on-failure" else "stop-on-failure"
                response.status = falcon.HTTP_200
                if respond_as.lower() == "immediate":
                    log.debug("Response will be given as chunks for each request from batch")
                    response.stream = response_generator(app, headers, rbody, respond_as, modus)
                else:
                    log.debug("Response will given as list once all requests been processed from batch!")
                    collate_resp = []
                    gen = response_generator(app, headers, rbody, respond_as, modus)
                    if gen:
                        for data in gen:
                            collate_resp.append(data)
                    response.body = json.dumps(collate_resp)
            else:
                raise MandatoryDataMissingError("Mandatory field multiget: %s are missing or is not a list" % rbody)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error while performing the multiget: %s"% str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                               "Error while performing the multiget: %s" % str(ex),
                                               "Error while performing the multiget",
                                               falcon.HTTP_500)
            return
