#!/bin/bash
#!/bin/bash

if [ -f ../scripts/startup.py ] ; then
    STARTUP=../scripts/startup.py
else
    STARTUP=../scripts/startup.pyc
fi

IOX_TMPFS_MOUNT_LOCATION="/iox"
IOX_CAF_MOUNT_LOCATION="/iox/image"
IOX_STARTUP_CONFIG_FILE="/sw/opt/cisco/caf/IOxStartUp.cfg"
IOX_ROLLBACK_CONFIG_FILE="/sw/opt/cisco/caf/IOxRollback.cfg"

clean_up_swupdate_mounts()
{
    if grep -qs $IOX_CAF_MOUNT_LOCATION /proc/mounts; then
        logger -t CAF -p user.info "Image mounted, location : $IOX_CAF_MOUNT_LOCATION"
        loop_device=$(df $IOX_CAF_MOUNT_LOCATION | tail -n 1 | awk 'echo $1')
        umount -l $IOX_CAF_MOUNT_LOCATION
        lo_image=$(echo "$loop_device" | cut -d ' ' -f1)
        losetup -d "$lo_image" >/dev/null
    else
        logger -t CAF -p user.info "CAF update image not mounted"
    fi
    if grep -qs $IOX_TMPFS_MOUNT_LOCATION /proc/mounts; then
        logger -t CAF -p user.info "tmpfs mounted, location: $IOX_TMPFS_MOUNT_LOCATION"
        if ! umount -l $IOX_TMPFS_MOUNT_LOCATION
        then
            logger -t CAF -p user.err "Cannot umount tmpfs: $IOX_TMPFS_MOUNT_LOCATION"
        else
            rm -r $IOX_TMPFS_MOUNT_LOCATION
        fi
    else
        logger -t CAF -p user.info "tmpfs not mounted"
    fi
}

do_fsck_image()
{
    image=$1
    logger -t CAF -p user.info "Running fsck on the image: $image"
    out=$(e2fsck -f -p  "$image")
    logger -t CAF -p user.info "e2fsck output: $out"
    if ! $out
    then
        out=$(e2fsck -y  "$image")
        if ! $out
        then
            logger -t CAF -p user.err "fsck with y option failed on the image with output: $out"
            return 1
        fi
    else
        return 0
    fi
}


DEFAULT_SCRIPT_PATH="../scripts"
DEFAULT_CONFIG_PATH="../config"

SCRIPT_PATH="../scripts"
CONFIG_PATH="../config"


verify_platform_version_compatibility()
{
    host_platform_version=$1
    update_host_platform_version=$2
    update_target_platform_version=$3
    do_version_check "$host_platform_version" "$update_host_platform_version"
    if [ "$?" -eq "10" ]; then
        return 0
    else
        do_version_check "$host_platform_version" "$update_target_platform_version"
        if [ "$?" -eq "9" ]; then
            #This means host platform version is less than target platform version of update image
            return 1
        elif [ "$?" -eq "11" ]; then
        #comment , should not support this
        #This means host platform version is ge than target update version.This should work ideally
        #platform version represents software utilities that CAF depends on
        #first major , second minor, third bug fix, last build number
        #compare first two and should be same for first release
            return 0
        else
            return 1
        fi
    fi
}

set_iox_image_location()
{
#Contract is that size is in bytes
#format can be ext2/ext3/ext4 or tar.gz
#This function is called incase image update is required and all the compatibility checks and sanity has been passed
    logger -t CAF -p user.info "Set the new IOx image location"
    iox_update_image=$1
    format=$2
    size=$3
    if [[ $size -lt 1048576 ]]; then
        size_in_mb="$(( (size + 1023)/1024 ))m"
    else
        size_in_mb="$(( (size + 1048575)/1048576 ))m"
    fi
    if echo "$format" | grep -q "ext"; then
        command="mount"
    elif echo "$format" | grep -q "tar"; then
        command="tar"
    else
        return 1
    fi

    if grep -qs "$IOX_TMPFS_MOUNT_LOCATION" /proc/mounts; then
        logger -t CAF -p user.info "tmpfs mount exists"
    else
        mkdir $IOX_TMPFS_MOUNT_LOCATION >/dev/null
        logger -t CAF -p user.info "Creating tmpfs mount"
        logger -t CAF -p user.info "tmpfs size: $size_in_mb"
        if ! mount -t tmpfs -o size="$size_in_mb" tmpfs $IOX_TMPFS_MOUNT_LOCATION
        then
            logger -t CAF -p user.err "tmpfs creation failed"
            return 1
        fi
    fi
    mkdir -p $IOX_CAF_MOUNT_LOCATION
    if [ "$command" == "mount" ]; then
        if grep -qs "$IOX_CAF_MOUNT_LOCATION" /proc/mounts; then
            logger -t CAF -p user.info "CAF sw update mount exists: $IOX_CAF_MOUNT_LOCATION"
            return 1
        else
            #Mount as ro
            if do_fsck_image "$iox_update_image"
            then
                if ! $command -o ro,noatime,loop -t "$format" "$iox_update_image" "$IOX_CAF_MOUNT_LOCATION"
                then
                    logger -t CAF -p user.err "Mount command failed for sw update image"
                    return 1
                fi
            else
                return 1
            fi
        fi
    else
        if ! $command -zxvf "$iox_update_image" -C "$IOX_CAF_MOUNT_LOCATION"
        then
            logger -t CAF -p user.err "Extracting tar archive failed to tmpfs location"
            return 1
        fi
    fi
    caf_script_path="/caf/scripts"
    caf_config_path="/caf/config"
    #Overwrite the script and config variables
    SCRIPT_PATH=$IOX_CAF_MOUNT_LOCATION$caf_script_path
    CONFIG_PATH=$IOX_CAF_MOUNT_LOCATION$caf_config_path
    echo $SCRIPT_PATH
    echo $CONFIG_PATH
    return 0
}

#This func compares two version in x.x.x.x format
#Returns 9 when second input param is greater than first one
#Returns 11 when first is greater than second one
#Returns 10 when both are equal
EQUAL="10"
GREATER_THAN="11"
LESS_THAN="9"
do_version_check()
{
   echo "version check"
   echo "$1"
   echo "$2"
   [ "$1" == "$2" ] && return 10

   ver1front=`echo $1 | cut -d "." -f -1`
   ver1back=`echo $1 | cut -d "." -f 2-`
   echo $ver1front
   echo $ver1back
   ver2front=`echo $2 | cut -d "." -f -1`
   ver2back=`echo $2 | cut -d "." -f 2-`
   echo $ver2front
   echo $ver2back

   if [ "$ver1front" != "$1" ] || [ "$ver2front" != "$2" ]; then
       [ "$ver1front" -gt "$ver2front" ] && return 11
       [ "$ver1front" -lt "$ver2front" ] && return 9

       [ "$ver1front" == "$1" ] || [ -z "$ver1back" ] && ver1back=0
       [ "$ver2front" == "$2" ] || [ -z "$ver2back" ] && ver2back=0
       do_version_check "$ver1back" "$ver2back"
       return $?
   else
           [ "$1" -gt "$2" ] && return 11 || return 9
   fi
}

verify_host_update_caf_versions()
{
    echo "$1"
    echo "$2"
    do_version_check "$1" "$2"
    retval="$?"
    if [ "$retval" -eq "10" ]; then
        echo "CAF versions on host and update image are equal"
        return 0
    elif [ "$retval" -eq "9" ]; then
        echo "CAF version on host is less than the one in update image"
        return 0
    elif [ "$retval" -eq "11" ]; then
        logger -t CAF -p user.err "CAF version on host is greater than one in update image, go ahead with host image itself"
        return 1
    else
        logger -t CAF -p user.err "Error while comparing the IOx versions"
        return 1
    fi
}

reset_image_if_updated()
{
    clean_up_swupdate_mounts
    logger -t CAF -p user.info "Cleaning up mount points, startup config file so that default image will be loaded"
    #What happens to rollback file, TODO
    if [ -f "$IOX_STARTUP_CONFIG_FILE" ]; then
        if [ -f "$IOX_ROLLBACK_CONFIG_FILE" ]; then
            mv "$IOX_ROLLBACK_CONFIG_FILE" "$IOX_STARTUP_CONFIG_FILE"
        else
            if [ -e "$IOX_STARTUP_CONFIG_FILE" ]; then
                rm "$IOX_STARTUP_CONFIG_FILE"
            fi
        fi
    fi
    return 0
}
do_image_rollback()
{
    #Clean up all the mount points created earlier
    clean_up_swupdate_mounts
    logger -t CAF -p user.info "Do CAF sw rollback"

    if [ -f "$IOX_ROLLBACK_CONFIG_FILE" ]; then
        # shellcheck source=/dev/null
        . "$IOX_ROLLBACK_CONFIG_FILE"
        if [ -z "$IMAGE_PAYLOAD" ] || [ "$IMAGE_PAYLOAD" == "NONE" ] || [ "$FORMAT" == "NONE" ] || [ "$IMAGE_ID" == "NONE" ] || [ -z "$IOX_CAF_VERSION" ] || [ -z "$IOX_TARGET_PLATFORM_VERSION" ] || [ -z "$FORMAT" ]; then
            #Default platform image
            logger -t CAF -p user.info "Setting the CAF image to default platform image"
            SCRIPT_PATH="$DEFAULT_SCRIPT_PATH"
            CONFIG_PATH="$DEFAULT_CONFIG_PATH"
            if [ -f "$IOX_STARTUP_CONFIG_FILE" ]; then
                rm -f "$IOX_STARTUP_CONFIG_FILE"
            fi
        else
            verify_integrity_image "$IMAGE_PAYLOAD" "$MD5SUM"
            if [ "$?" -eq "0" ]; then
                set_iox_image_location "$IMAGE_PAYLOAD"  "$FORMAT" "$SIZE"
                if [ "$?" -eq "1" ]; then
                    logger -t CAF -p user.err "Rollback, setting of IOX image failed.Move to the default one"
                    SCRIPT_PATH="$DEFAULT_SCRIPT_PATH"
                    CONFIG_PATH="$DEFAULT_CONFIG_PATH"
                    #Since platform image will be booted, remove the startup config file for the updated image
                    if [ -f "$IOX_STARTUP_CONFIG_FILE" ]; then
                        mv "$IOX_ROLLBACK_CONFIG_FILE" "$IOX_STARTUP_CONFIG_FILE"
                    fi
                else
                    #Move the rollback config to startup config, so that it gets reflected in CAF
                    #IMAGE ID will be problem
                    #Test this
                    logger -t CAF -p user.info "IOxRollback, Image set to the previous one.Move the startup config file"
                    if [ -f "$IOX_STARTUP_CONFIG_FILE" ]; then
                        mv "$IOX_ROLLBACK_CONFIG_FILE" "$IOX_STARTUP_CONFIG_FILE"
                    fi
                fi
            else
                logger -t CAF -p user.err "Image md5sum verification failed"
                SCRIPT_PATH="$DEFAULT_SCRIPT_PATH"
                CONFIG_PATH="$DEFAULT_CONFIG_PATH"
                #Since platform image will be booted, remove the startup config file for the updated image
                if [ -f "$IOX_STARTUP_CONFIG_FILE" ]; then
                    mv "$IOX_ROLLBACK_CONFIG_FILE" "$IOX_STARTUP_CONFIG_FILE"
                fi
            fi
        fi
        #Finally remove the roll back config file
        rm -f "$IOX_ROLLBACK_CONFIG_FILE"

    else
        #Default platform image
        logger -t CAF -p user.info "IOx Rollback: Moving back to default platform image"
        SCRIPT_PATH="$DEFAULT_SCRIPT_PATH"
        CONFIG_PATH="$DEFAULT_CONFIG_PATH"
        if [ -f "$IOX_STARTUP_CONFIG_FILE" ]; then
            rm -f "$IOX_STARTUP_CONFIG_FILE"
        fi
    fi

    echo "$SCRIPT_PATH"
    echo "$CONFIG_PATH"
    export CAF_HOME
    CAF_HOME=$(dirname $SCRIPT_PATH)
    if [ -f $SCRIPT_PATH/startup.py ] ; then
        STARTUP="$SCRIPT_PATH"/startup.py
    else
        STARTUP="$SCRIPT_PATH"/startup.pyc
    fi
    logger -t CAF -p user.info "IOx Image rollback, Starting IOx: $STARTUP"
    python $STARTUP $CONFIG_PATH/system-config.ini $CONFIG_PATH/log-config.ini 2>&1 | logger -t CAF &
    pids=($(jobs -l % | sed 's/^[^ ]* \+//' | cut -d\  -f1))
    #Check if the pid is running
    if ps -p ${pids[0]} > /dev/null
    then
        echo ${pids[0]} > /var/run/caf.pid
    fi
    exit 0
}

remove_startup_config()
{
    if [ -f  "$IOX_STARTUP_CONFIG_FILE" ]; then
        rm -f "$IOX_STARTUP_CONFIG_FILE"
    fi
}

verify_integrity_image()
{
    image_path="$1"
    image_checksum="$2"
    b=$(md5sum $image_path)
    calculated_checksum=$(echo  $b | cut -d ' ' -f1)
    calculated_checksum=$(echo "$calculated_checksum" | sed -e 's/[\r\n]//g')
    image_checksum=$(echo "$image_checksum" | xargs)
    if [ "$image_checksum" == "$calculated_checksum" ]; then
        return 0
    else
        logger -t CAF -p user.err "md5 checksum verification failed for the image"
        return 1
    fi
}

update_image_if_required()
{
    clean_up_swupdate_mounts
    echo "Inside check if update required shell func"
    if [ -f  "$IOX_STARTUP_CONFIG_FILE" ]; then
        echo "Startup file present, sourcing the file"
        # shellcheck source=/software/caf
        . "$IOX_STARTUP_CONFIG_FILE"
        echo "$IMAGE_ID" "$FORMAT" "$SIZE" "$IMAGE_PAYLOAD" "$IOX_CAF_VERSION" "$IOX_HOST_PLATFORM_VERSION" "$IOX_TARGET_PLATFORM_VERSION" "$IOX_REPO_VERSION"
        if [ -z "$IMAGE_ID" ] || [ "$IMAGE_ID" == "NONE" ]; then
            logger -t CAF -p user.info "IMAGE ID not set"
            remove_startup_config
            return 1
        fi
        if [ -z "$FORMAT" ]; then
            logger -t CAF -p user.info "Format variable not set"
            remove_startup_config
            return 1
        fi
        if [ -z "$IMAGE_PAYLOAD" ] || [ "$IMAGE_PAYLOAD" == "NONE" ]; then
            logger -t CAF -p user.info "Image payload not set"
            remove_startup_config
            return 1
        fi
        if [ -z "$IOX_CAF_VERSION" ]; then
            logger -t CAF -p user.info "SW Update , IOx CAF version var not set"
            remove_startup_config
            return 1
        fi
        if [ -z "$IOX_HOST_PLATFORM_VERSION" ]; then
            logger -t CAF -p user.info "CAF Update image, IOx host platform version var not set"
            remove_startup_config
            return 1
        fi
        if [ -z "$MD5SUM" ] || [ "$MD5SUM" == "NONE" ]; then
            remove_startup_config
            return 1
        else
            verify_integrity_image "$IMAGE_PAYLOAD" "$MD5SUM"
            if [ "$?" -eq "1" ]; then
                logger -t CAF -p user.err "CAF md5sum verification failed"
                remove_startup_config
                return 1
            fi
        fi

        if [ -z "$IMAGE_VERIFY_SCRIPT" ]; then
            logger -t CAF -p user.info "CAF image verify script not provided"
        fi

        #cafevents.log, CAF should not write anything locally should always be in repo folder
        #cleanup all the mountpoints at the start
        #fsck on the ext2 fs, automatic (-f , -p), -y
        #then mount, check for first two octets , both should be equal , no plan to support backward compatibility
        #like caf target 1.3 will work with 1.4, ideally should work
        #same check in caf code while installing image
        #Cannot downgrade CAF below the platform image, since if host caf > update image we will always prefer first
        #Assumption is platform image will have always latest CAF version , always ge than update/patch version
        #Better to deactivate/uninstall the image before platform upgrade/Downgrade
        if [ -z "$IOX_TARGET_PLATFORM_VERSION" ]; then
            logger -t CAF -p user.info "SW Update image, IOx target platform version not set"
            return 1
        fi
        if [ -z "$SIZE" ]; then
            SIZE=$(du -b "$IMAGE_PAYLOAD" | cut -f1)
        fi
        if [ -f /etc/platform/version ];then
            host_platform_version=$(grep '^version_number:' /etc/platform/version | cut -d ":" -f2)
            #host_caf_version=$(grep '^version_number:' /home/root/iox/caf/config/version | cut -d ":" -f2)
            verify_platform_version_compatibility "$host_platform_version" "$IOX_HOST_PLATFORM_VERSION" "$IOX_TARGET_PLATFORM_VERSION"
            if [ "$?" -eq "1" ]; then
                logger -t CAF -p user.info "platform version compatibility failed"
                return 1
            fi
        else
            logger -t CAF -p user.err "platform versions cannot be compared, version not set in the platform"
            #TODO
            #return 1
        fi
        host_caf_version=$(grep '^version_number:' $CONFIG_PATH/version | cut -d ":" -f2)
        echo $host_caf_version $IOX_CAF_VERSION
        host_caf_version=$(echo "$host_caf_version" | xargs)
        verify_host_update_caf_versions "$host_caf_version" "$IOX_CAF_VERSION"
        if [ "$?" -eq "1" ]; then
            logger -t CAF -p user.err "Comparison of CAF version between platform provided and SW update failed"
            remove_startup_config
            return 1
        fi
        set_iox_image_location "$IMAGE_PAYLOAD" "$FORMAT" "$SIZE"
        return $?
    else
        logger -t CAF -p user.info "Startup config not available , continue with default CAF sw"
        return 1
    fi
    return 1
}

run_image_upgrade_verify_script()
{
    #By default hit /iox API
    if [ -z "$IMAGE_VERIFY_SCRIPT" ]; then
        return 0
    else
        shopt -s nocasematch >/dev/null
        if [ "$IMAGE_VERIFY_SCRIPT_TYPE" == "python" ]; then
            #Run python script, python path should be right and is platform specific
            if ! python "$IMAGE_VERIFY_SCRIPT"
            then
                logger -t CAF -p user.err "CAF sw verify python script failed"
                return 1
            fi
        else
            #Run shell script by default
            if ! sh "$IMAGE_VERIFY_SCRIPT"
            then
                logger -t CAF -p user.err "CAF sw verify shell script failed"
                return 1
            fi
        fi
    fi
    return 0
}

