#!/bin/bash
# Reference - https://docs.docker.com/engine/security/https/
# Usage1: <scriptname> <tls cert dir>                -------> To generate and verify docker tls creds
# Usage2: <scriptname> <tls cert dir> "verify"       -------> To just verify docker ca and server tls creds
#####################################################
dockercertdir=$1
only_verify_tls=$2
CN="localhost"
#cakey_pass=$(head /dev/urandom | tr -dc A-Za-z0-9 | head -c 13 ; echo '')
cakey_pass="cisco"
cakey="ca-key.pem"
cacert="ca.pem"

serverkey="server-key.pem"
servercsr="server.csr"
servercert="server-cert.pem"

clientkey="key.pem"
clientcsr="client.csr"
clientcert="cert.pem"

if [ -z $dockercertdir ]; then
    echo "no input args specifying tls cert dir"
    exit -1
fi

cleanup_docker_tls_certs()
{
    if [ -d ${dockercertdir} ]; then
        rm -rf ${dockercertdir}/*
    fi
    logger -t docker_tls_script "Cleaned up docker tls cert/keys"
}


# Verify tls certs/keys
verify_generic_openssl_cert_key_pair()
{
    key=$1
    cert=$2
    if [ -f "$dockercertdir/$key" ] && [ -f "$dockercertdir/$cert" ]; then
        # Verify certificate
        cert_res=$(openssl x509 -in "$dockercertdir/$cert" -text -noout 2>&1)
        if [ $? != 0 ]
        then
            logger -t docker_tls_script -p user.err "TLS certificate ("$dockercertdir/$cert") validation failed: $cert_res"
            return 1
        fi

        # Verify private key
        key_res=$(openssl rsa -in "$dockercertdir/$key" -passin pass:$cakey_pass -check 2>&1)
        if [ $? != 0 ]
        then
            logger -t docker_tls_script -p user.err "TLS key ("$dockercertdir/$key") validation failed: $key_res"
            return 1
        fi

        # Verify cert and private key match
        cert_mod_res=$(openssl x509 -noout -modulus -in "$dockercertdir/$cert" 2>&1)
        if [ $? != 0 ]
        then
            logger -t docker_tls_script -p user.err "Modulus of TLS certificate ("$dockercertdir/$cert") failed: $cert_mod_res"
            return 1
        fi

        key_mod_res=$(openssl rsa -noout -modulus -in "$dockercertdir/$key" -passin pass:$cakey_pass 2>&1)
        if [ $? != 0 ]
        then
            logger -t docker_tls_script -p user.err "Modulus of TLS key ("$dockercertdir/$key") failed: $key_mod_res"
            return 1
        fi

        if [ "$cert_mod_res" != "$key_mod_res" ]
        then
            logger -t docker_tls_script -p user.err "TLS key ('$dockercertdir/$key') and  certificate ('$dockercertdir/$cert') dont match."
            return 1
        fi

        # All validations succeeded
        logger -t docker_tls_script -p user.info "TLS certificate/key validation succeeded"
        return 0
    else
        logger -t docker_tls_script -p user.info "TLS certificate/key doesn't exist."
        return 1
    fi
}

if [ "$only_verify_tls" = "verify" ]; then
    # Validate ca and server certs
    verify_generic_openssl_cert_key_pair $cakey $cacert
    if [ "$?" -eq "1" ]; then
        cleanup_docker_tls_certs
        logger -t docker_tls_script "Verification of CA TLS certs failed"
        exit 1
    fi

    verify_generic_openssl_cert_key_pair $serverkey $servercert
    if [ "$?" -eq "1" ]; then
        cleanup_docker_tls_certs
        logger -t docker_tls_script "Verification of Server TLS certs failed"
        exit 1
    fi
    exit 0
fi

verify_all_certs_keys()
{
    verify_generic_openssl_cert_key_pair $cakey $cacert
    if [ "$?" -eq "1" ]; then
        logger -t docker_tls_script "Verification of CA TLS certs failed"
        return 1
    fi

    verify_generic_openssl_cert_key_pair $serverkey $servercert
    if [ "$?" -eq "1" ]; then
        logger -t docker_tls_script "Verification of Server TLS certs failed"
        return 1
    fi

    verify_generic_openssl_cert_key_pair $clientkey $clientcert
    if [ "$?" -eq "1" ]; then
        logger -t docker_tls_script "Verification of Server TLS certs failed"
        return 1
    fi
}

cleanup_docker_tls_certs
mkdir -p $dockercertdir

generate_ca_tls_creds()
{
    openssl genrsa -aes256 -passout pass:$cakey_pass -out $dockercertdir/$cakey
    if [ $? -ne 0 ]; then
        logger -t docker_tls_script "Failed to generate ca TLS key"
        return 1
    fi
    logger -t docker_tls_script "Successfully generated ca key"

    openssl req -new -x509 -days 3650 -key $dockercertdir/$cakey -passin pass:$cakey_pass -sha256 -out $dockercertdir/$cacert -subj "/C=US/ST=CA/L=Milpitas/O=Widgets Inc/CN=$CN"
    if [ $? -ne 0 ]; then
        logger -t docker_tls_script "Failed to generate ca cert"
        return 1
    fi
    logger -t docker_tls_script "Successfully generated ca cert"
}

generate_server_tls_creds()
{
    openssl genrsa -out $dockercertdir/$serverkey
    if [ $? -ne 0 ]; then
        logger -t docker_tls_script "Failed to generate docker server TLS key"
        return 1
    fi
    logger -t docker_tls_script "Successfully generated server key"

    openssl req -subj "/CN=$CN" -sha256 -new -key $dockercertdir/$serverkey -out $dockercertdir/$servercsr
    if [ $? -ne 0 ]; then
        logger -t docker_tls_script "Failed to generate docker server CSR"
        return 1
    fi
    logger -t docker_tls_script "Successfully generated server csr"

    echo subjectAltName = DNS:$HOSTNAME,IP:127.0.0.1 >> $dockercertdir/extfile.cnf
    echo extendedKeyUsage = serverAuth >> $dockercertdir/extfile.cnf
    openssl x509 -req -days 3650 -sha256 -in $dockercertdir/$servercsr -CA $dockercertdir/$cacert -CAkey $dockercertdir/$cakey \
      -passin pass:$cakey_pass -CAcreateserial -out $dockercertdir/$servercert -extfile $dockercertdir/extfile.cnf
    if [ $? -ne 0 ]; then
        logger -t docker_tls_script "Failed to generate docker server cert"
        return 1
    fi
    logger -t docker_tls_script "Successfully generated server cert"
}

generate_client_tls_creds()
{
    openssl genrsa -out  $dockercertdir/$clientkey
    if [ $? -ne 0 ]; then
        logger -t docker_tls_script "Failed to generate docker client key"
        return 1
    fi
    logger -t docker_tls_script "Successfully generated client key"

    openssl req -subj '/CN=client' -new -key  $dockercertdir/$clientkey -out  $dockercertdir/$clientcsr
    if [ $? -ne 0 ]; then
        logger -t docker_tls_script "Failed to generate docker client CSR"
        return 1
    fi
    logger -t docker_tls_script "Successfully generated client csr"

    echo extendedKeyUsage = clientAuth > $dockercertdir/extfile-client.cnf
    openssl x509 -req -days 365 -sha256 -in $dockercertdir/$clientcsr -CA $dockercertdir/$cacert -CAkey $dockercertdir/$cakey \
        -passin pass:$cakey_pass  -CAcreateserial -out $dockercertdir/$clientcert -extfile $dockercertdir/extfile-client.cnf
    if [ $? -ne 0 ]; then
        logger -t docker_tls_script "Failed to generate docker client cert"
        return 1
    fi
    logger -t docker_tls_script "Successfully generated client cert"
}

#CA cert/key generation
generate_ca_tls_creds
if [ "$?" -eq "1" ]; then
    cleanup_docker_tls_certs
    logger -t docker_tls_script "Failed to generate docker server tls creds"
    exit 1
fi

#####################

#Server cert/key generation
generate_server_tls_creds
if [ "$?" -eq "1" ]; then
    cleanup_docker_tls_certs
    logger -t docker_tls_script "Failed to generate docker server tls creds"
    exit 1
fi
#####################

# Client cert/key generation
generate_client_tls_creds
if [ "$?" -eq "1" ]; then
    cleanup_docker_tls_certs
    logger -t docker_tls_script "Failed to generate docker client tls creds"
    exit 1
fi

# Validate the certs
verify_all_certs_keys
if [ "$?" -eq "1" ]; then
    cleanup_docker_tls_certs
    logger -t docker_tls_script "Docker TLS certs verification failed"
    exit 1
fi

############## 
# File permissions
chmod -v 0400 $dockercertdir/$cakey $dockercertdir/$clientkey $dockercertdir/$serverkey
if [ $? -ne 0 ]; then
    # Log error and let the remote docker access request proceed.
    logger -t docker_tls_script "Failed to change permission for docker tls keys"
fi
chmod -v 0444 $dockercertdir/$cacert $dockercertdir/$servercert $dockercertdir/$clientcert
if [ $? -ne 0 ]; then
    # Log error and let the remote docker access request proceed.
    logger -t docker_tls_script "Failed to change permission for docker tls certs"
fi
logger -t docker_tls_script "Successfully set permissions for certs/keys"
############
# Create a tarfile out of files needed by remote docker client
cd $dockercertdir/
tar -cvf $dockercertdir/tlscerts.tar $cacert $clientcert $clientkey
if [ $? -ne 0 ]; then
    cleanup_docker_tls_certs
    logger -t docker_tls_script "Failed to create tar archive of docker client TLS creds"
    exit 1
fi
logger -t docker_tls_script "successfully generated client tls tar archive"
exit 0