__author__ = 'hvishwanath'

import sys
import tempfile
import datetime
import glob
from buildhandlers import *
from buildutils import *
import traceback

class CAFBuilder(object):
    # core/caf/scripts/build
    p = os.path.dirname(os.path.realpath(__file__))
    # core/caf/scripts
    p = os.path.split(p)[0]    #get scripts
    # core/caf
    caf_home = os.path.split(p)[0]    #get caf directory
    core_home = os.path.split(caf_home)[0]
    WS_TOPDIR = os.path.split(core_home)[0]
    # Create a build directory. core/caf/builds. This is where the builds will go.
    BUILD_BASE = os.path.join(caf_home, "builds")

    MODULE_DIR = os.path.dirname(os.path.realpath(__file__))
    ROOT_BASE = tempfile.mkdtemp()
    TIME_STAMP = datetime.datetime.strftime(datetime.datetime.now(), "%Y%m%d_%H%M%S")

    def __init__(self):
        pass

    @property
    def platform_id(self):
        return self._platformid


    def git_clone(self, git_remote_repo, git_branch, target_folder):
        print "Cloning %s branch from remote repo (%s) into %s" % (git_branch, git_remote_repo, target_folder)
        o = subprocess.check_output(['git', 'clone', '-b', git_branch, git_remote_repo, target_folder])
        return o

    def git_checkout_rev(self, git_dir, git_rev):
        print "Checking out revision : %s. Git Directory: %s" % (git_rev, git_dir)
        cwd = os.getcwd()
        os.chdir(git_dir)
        o = subprocess.check_output(['git', 'checkout', git_rev])
        os.chdir(cwd)
        return o

    def get_pip_source_cache(self, ROOT_FOLDER):
        source_cache = []
        source_cache.append(os.path.join(ROOT_FOLDER, "third-party/python27/caf"))
        source_cache.append(os.path.join(ROOT_FOLDER, "third-party/python27/ut"))

    def build(self,
              git_remote_repo,
              git_branch,
              git_rev,
              target_platform,
              exclude_source,
              with_rest_api,
              with_local_manager,
              xwt_home,
              use_pip,
              image_format,
              delete_previous=True,
              no_tar=False,
              python_path="python",
              skip_python_modules=False,
              create_image=False,
              buffer_size='2000'):

        if not target_platform in BuildHandlerMap.keys():
            print "No handler found for platform %s" % target_platform
            sys.exit(0)


        # Create / find target build directory
        build_dir = os.path.join(self.BUILD_BASE, target_platform)
        if not os.path.isdir(build_dir):
            os.makedirs(build_dir)
        print "Created target build directory : %s" % build_dir

        # Create temporary build directory
        build_root = os.path.join(self.ROOT_BASE, target_platform)
        if not os.path.isdir(build_root):
            os.makedirs(build_root)
        print "Created temporary build directory: %s" % build_root

        if git_remote_repo:
            # Clone from requested target
            self.git_clone(git_remote_repo, git_branch, build_root)
            assert(os.path.exists(os.path.join(build_root, "core/caf")))
            print "Repo clone successful."

            if git_rev:
                self.git_checkout_rev(build_root, git_rev)
                print "Checked out revision %s" % git_rev
        else:
            '''
            Use local workspace state to build caf and lm. During development
            this type of caf-build makes it easy to test the code changes on a platform,
            instead of copying individual files that has been touched to the platform.

            To persist the local workspace state before/after build, the workspace is copied
            to temp directory and being built there.
            '''
            os.chdir(self.WS_TOPDIR)
            cp_command = ['cp', '-r', ".", build_root]
            subprocess.check_output(cp_command)

        # Now formulate a version file.
        # 1) Retrieve the git commit hash
        cwd = os.getcwd()
        os.chdir(build_root)
        cur_rev = subprocess.check_output(['git', 'log' ,'--format=%H', '-n', '1'])
        os.chdir(cwd)
        cur_rev = cur_rev.lstrip().rstrip()

        # 2) Either create a new version file or update the existing one
        version_file = os.path.join(build_root, "core/caf/config/version")
        if os.path.isfile(version_file):
            # file exists, attempt to update markers in it with git information
            print "Version file already found at %s. Skipping creating auto version" % version_file
            modified = False
            newlines = []
            with open(version_file,'r') as f:
                for line in f.readlines():
                    newline = line.replace("@revision@", cur_rev)
                    modified |= (newline != line)
                    line = newline
                    newline = line.replace("@branch@", git_branch)
                    modified |= (newline != line)
                    newlines.append(newline)
            # write back modified content to the file
            if modified:
                with open(version_file, 'w') as f:
                    for line in newlines:
                        f.write(line)
        else:
            # file doesn't existing, store minimum information
            cur_version = "%s-%s" % (git_branch, cur_rev)
            file(version_file, "w").write(cur_version)
            print "Calculated version : %s. Wrote to %s" % (cur_version, version_file)


        # Find and instantiate the build handler
        hcls = BuildHandlerMap.get(target_platform)
        bh = hcls()

        # Figure out what the final build file should be named
        sanitized_git_branch = git_branch.replace("/","-")
        build_file = "iox.tar.gz"
        build_file = os.path.join(build_dir, build_file)
        # Change directory to build root
        curdir = os.getcwd()
        print "Changing directory to build root : %s" % build_root
        os.chdir(build_root)

        # Build source cache
        print "Determining source cache.."
        source_cache = []
        source_cache.append(os.path.join(build_root, "third-party/python27/caf"))
        source_cache.append(os.path.join(build_root, "third-party/python27/ut"))


        # Determine list of packages
        print "Determining list of packages to be installed for the target platform.."
        package_list = []
        if not skip_python_modules:
            print "Skipping inclusing of python modules"
            package_list = file("core/caf/config/caf-requirements.txt").read().split('\n')
            package_list = bh.get_platform_package_list(package_list, with_rest_api)


        # Get target dir to where the packages have to be installed
        print "Determining target directory for package installation.."
        relativedir = bh.get_package_target_dir()
        package_target_dir = os.path.join(build_root, relativedir)
        if not os.path.exists(package_target_dir):
            os.makedirs(package_target_dir)


        print "Source Cache: %s" % str(source_cache)
        print "Package List: %s" % str(package_list)
        print "Package target directory : %s" % str(package_target_dir)
        print "Installing CAF dependencies from local cache.."


        if use_pip:
            pip_install_local(source_cache=source_cache,
                              target_platform=target_platform,
                              target_dir=package_target_dir,
                              package_list=package_list,
                              python_path=python_path)

        else:

            install_packages_without_pip(source_cache=source_cache,
                                         target_platform=target_platform,
                                         target_dir=package_target_dir,
                                         package_list=package_list,
                                         python_path=python_path)

        print "Packaging missing python stdlib files.."
        x = bh.get_pystdlib_target_dir()
        pystdlib_target_dir = os.path.join(build_root, x)
        if not os.path.exists(pystdlib_target_dir):
            os.makedirs(pystdlib_target_dir)

        src_pystdlib = bh.get_pystdlib()
        get_missing_pystdlibs(source_files=src_pystdlib,
                              target_dir=pystdlib_target_dir)

        print "Placing ssl.crt and ssl.key files into caf/config"
        shutil.copy2("core/caf/tests/resources/ssl.crt", "core/caf/config")
        shutil.copy2("core/caf/tests/resources/ssl.key", "core/caf/config")


        if with_local_manager:
            print "Building local manager.."
            setup_local_manager(xwt_home=xwt_home)
            if target_platform != 'pi':
                print "Moving LM to build root"
                shutil.move("core/lm", os.path.join(build_root, "lm"))

        else:
            print "Local Manager will not be built.."


        print "Determining if there are any platform overrides.."
        platform_overrides = bh.get_platform_overrides()
        print "Platform overrides : %s" % str(platform_overrides)
        for t in platform_overrides:
            s, d = t
            if os.path.isfile(s):
                print "Overriding %s with %s" % (d, s)
                shutil.copyfile(s, d)


        print "Pruning unwanted files.."
        prune_files_list = bh.get_prune_files()
        for f in prune_files_list:
            print "\tRemoving %s" % f
            os.remove(os.path.join(build_root, f))


        print "Pruning unwanted directories.."
        prune_dirs_list = bh.get_prune_dirs()
        for d in prune_dirs_list:
            print "\tRemoving %s" % d
            shutil.rmtree(d)



        print "Invoking handler's hook for post processing if any.."
        bh.post_process(build_root)

        print "Deleting platform directory"
        shutil.rmtree("platform")

        print "Deleting toplevel core directory if exists"
        if os.path.isdir("core"):
            shutil.rmtree("core")

        if exclude_source:
            print "Removing source files (*.py) from the final build.."
            remove_src_files(build_root)

        if delete_previous:
            print "Deleting earlier builds"
            if no_tar:
                r = glob.glob(build_dir + "/caf")
                r += glob.glob(build_dir + "/local")
                r += glob.glob(build_dir + "/lm")
                for i in r:
                    print "Removing %s" % i
                    shutil.rmtree(i)
                r = glob.glob(build_dir + "/tpminterface.py")
                for i in r:
                    print "Removing %s" % i
                    os.remove(i)
            else:
                r = glob.glob(build_dir + "/%s-build-*.gz" % target_platform)
                for i in r:
                    print "Removing %s" % i
                    os.remove(i)

        # compress the entire build_root and write the tarball to build_file
        if no_tar:
            print "Copying package ready for rootfs of the platform : %s" % build_file
            print build_root + '/*'
            print build_dir
            files = glob.glob(build_root + '/*')
            print files
            for i in files:
                print i
                cp_command = ['cp', '-r', i, build_dir]
                subprocess.check_output(cp_command)

            sz = subprocess.check_output(['du', '-sh', build_dir])
            build_file = build_dir
        else:
            print "Creating tar.gz build for the platform : %s" % build_file

            tgz_command = ['tar', 'czf', build_file, '.']
            sz = subprocess.check_output(tgz_command)
            sz = subprocess.check_output(['du', '-sh', build_file])

        if create_image:

            '''
            If you need to create image disk, run this script as sudo
            There is no exception handling as of now
            '''
            #Get the size in KB
            size_in_kb = subprocess.check_output(['du', '-sk', '.'])
            size = size_in_kb.split("\t")
            #Extra 2 MB buffer, image_size is in KB
            image_size = int(size[0]) + int(buffer_size)
            countarg = "count=" + str(image_size)
            of_arg = "of=image" + "." + image_format
            finalarg = ['dd', 'if=/dev/zero', of_arg, 'bs=1024', countarg]
            o = subprocess.check_output(finalarg)
            image_name = "image" + "." + image_format
            command = "mkfs" + "." + image_format
            a = [command, '-E', 'nodiscard', '-b', '1024', '-F', image_name]
            o = subprocess.check_output(a)
            shutil.move(image_name, os.path.join(build_dir, image_name))


            import tempfile
            image_mount_path = "/tmp/image"
            if not os.path.exists(image_mount_path):
                os.mkdir(image_mount_path)
            tempdir = tempfile.mkdtemp("", "tmpExtract", image_mount_path)
            li = ['mount', '-t', image_format, '-o', 'loop', os.path.join(build_dir, image_name), tempdir]
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            from distutils.dir_util import copy_tree
            copy_tree(build_root, tempdir)
            li = ['umount', '-l', tempdir]
            rval = subprocess.check_output(li, stderr=subprocess.STDOUT)
            shutil.rmtree(tempdir, ignore_errors=True)
            shutil.rmtree(image_mount_path, ignore_errors=True)

            print "Image disk is available at %s" % (os.path.join(build_dir, image_name))

        print "Destroying build root : %s" % build_root
        rootdir = os.path.dirname(build_root)
        shutil.rmtree(build_root)
        shutil.rmtree(rootdir)


        print "Successfully generated build for platform : %s" % target_platform
        print "Build is at %s. Size : %s" % (build_file, sz)

        return




if "__main__" == __name__:
    import argparse
    parser = argparse.ArgumentParser(description='Build / Package CAF for different platforms')

    parser.add_argument('--git-repo', action="store",
                        dest="gitrepo", required=False,
                        help="Specify git repo to pull code from")
    parser.add_argument('--git-branch', action="store", dest="gitbranch",
                        default='develop', help="Specify git branch to pull code from. Default is 'develop' ")
    parser.add_argument('--git-rev', action="store",
                        dest="gitrev", default=None,
                        help="Specify git revision to pull. Default is None")
    parser.add_argument('--platform', action="store", dest="platform",
                        default=None, choices=('barbados', 'vega', 'axel-bcm','axel-qca', 'mallorca', 'apvirtual', 'ax-bcm32'),
                        help="Target platform for which CAF needs to be built")
    parser.add_argument('--exclude-source', dest='excludesource',
                        action="store_true", default=False,
                        help="Use this flag if you want to remove *.py files from the build")
    parser.add_argument('--use-pip', dest='usepip',
                        action="store_true", default=False,
                        help="Use this flag if you want to use pip for the build and install")
    parser.add_argument('--with-rest-api', dest='withrestapi',
                        action="store_true", default=False,
                        help="Use this flag to package libs required by REST APIs. Default is false.")

    parser.add_argument('--with-local-manager', dest='withlm',
                        action="store_true", default=False,
                        help="Set this flag to build LM with the target build. Default is false.")

    parser.add_argument('--xwt-home', action="store", dest="xwt_home",
                        default='/local/xwt', help="Specify the directory where xwt bundle(xwt-bundle-src-2.0.5.zip)\
                         can be found. Default is /local/xwt")

    parser.add_argument('--no-tar', action="store_true", dest="notar",
                        default=False, help="Do not tar the final sources to make it easy for copying to rootfs")

    parser.add_argument('--python-path', action="store", dest="python_path",
                        default='python', help="Override default python path to be used for library installs")

    parser.add_argument('--skip-python-modules', action="store_true", dest="skip_python_modules",
                        default=False, help="Skip the inclusion of 3rd party python modules")

    parser.add_argument('--create-image', action="store_true", dest="createImage",
                        default=False, help="Use this flag to create CAF disk image")

    parser.add_argument('--image-format', action="store", dest="imageFormat",
                        default='ext2', help="Specify the image format if it needs to be created.Default is 'ext2' ")

    parser.add_argument('--extra-buffer', action="store", dest="bufferSize",
                        default='2000', help="Additional buffer size that will be added to the image")

    args = parser.parse_args()

    print "Supplied options : %s" % args

    git_remote_repo = args.gitrepo
    git_branch = args.gitbranch
    git_rev = args.gitrev
    target_platform = args.platform
    exclude_source = args.excludesource
    with_rest_api = args.withrestapi
    with_local_manager = args.withlm
    xwt_home = args.xwt_home
    use_pip = args.usepip
    no_tar = args.notar
    python_path = args.python_path
    image_format = args.imageFormat
    create_image = args.createImage
    buffer_size = args.bufferSize
    if create_image:
        if os.getgid() != 0:
            exit("You need to have root privileges to be able to create image disk")

    cb = CAFBuilder()
    try:
        cb.build(git_remote_repo=git_remote_repo,
                 git_branch=git_branch,
                 git_rev=git_rev,
                 target_platform=target_platform,
                 exclude_source=exclude_source,
                 with_rest_api=with_rest_api,
                 with_local_manager=with_local_manager,
                 xwt_home=xwt_home,
                 use_pip=use_pip,
                 delete_previous=True,
                 no_tar=no_tar,
                 python_path=python_path,
                 skip_python_modules=args.skip_python_modules,
                 create_image=create_image,
                 image_format=image_format,
                 buffer_size=buffer_size)
    except Exception as ex:
        print "CAF builder script failed with an exception."
        print traceback.format_exc()
