__author__ = "madawood"

import logging
import json
import falcon
import os
import shutil
import tempfile
import time

from cafevent import CAFEvent
from ..hosting.state import State
from infraexceptions import *
from utils import Utils

log = logging.getLogger("runtime.hosting")

def ztr_install_response_generator(controller, ns, pkg, uniqueConnectorId, filePath, tempdir, is_cleanup, tmpUploadDir,
                                   uri_str, app_resources={}, app_config={}, headers={}):
    """
    This API will take the application package, installs it and get it to running state.
    :param controller: This is the controller service instance. With this we will be performing all app installation steps.
    :param ns: This is the notification service instance. By using this we can post events to CAF.
    :param pkg: This is a package instance, with which we can get all required info about the package needs to installed.
    :param uniqueConnectorId: Id with which app needs to get installed.
    :param filePath: Location of the package to install.
    :param tempdir: Temp file system location to used for all temporary file creation operations.
    :param is_cleanup: Once the operation is done, tells whether to remove the source package or not.
    :param tmpUploadDir: Dir location where package is downloaded from web.
    :param uri_str: Tells URI path of the application.
    :param app_resources: Payload to be used while activation the app.
    :param app_config: Boot strap config needs to be set for the application.
    :param headers: Headers provided by the user.
    :return: Generator object.
    """
    if controller is None:
        log.error("Controller object is None, so will be not be able to process with any app lifecycle operations!")
        raise ServiceDepenendencyError("Controller object is None, so will be not be able to process with any app lifecycle operations!")
    resp_body = {}
    log.debug("Activation payload %s ", app_resources)
    configvalue = None
    logLevel = None
    try:
        configvalue = app_config["config"]
        if app_config.has_key("log-level"):
            logLevel = app_config["log-level"]
        is_app_config_available = True
    except Exception as ex:
        is_app_config_available = False
        log.warning("App config file not available for update %s", str(ex))


    log.debug("ztr_install %s %s %s %s ", uniqueConnectorId, filePath, tempdir, is_cleanup)
    warn_messages = []
    enable_debug = headers.get('X-DebugMode'.upper())
    if (enable_debug == '0' or enable_debug == 'off' or enable_debug is None or enable_debug == ""):
        enable_debug = False
    else:
        enable_debug = True
    ignore_previous_mapping = headers.get('X-IgnorePreviousMapping'.upper())
    if (ignore_previous_mapping == '0' or ignore_previous_mapping == 'off' or ignore_previous_mapping is None or ignore_previous_mapping == ""):
        ignore_previous_mapping = False
    else:
        ignore_previous_mapping = True
    try:
        deploy_warn_messages = controller.install_app(uniqueConnectorId, filePath, tempdir, is_cleanup)
        warn_messages.extend(deploy_warn_messages)
        app_custom_options = headers.get('X-App-Custom-Options'.upper())
        if app_custom_options:
            log.debug('Setting application custom options')
            controller.set_app_custom_options(uniqueConnectorId, app_custom_options)
        connectorURL = uri_str + "/" + uniqueConnectorId
        resp_body["status_code"] = falcon.HTTP_201
        resp_body["body"] = "Successfully deployed"
        if len(warn_messages) > 0 :
            resp_body["body"] += "\n With warnings : \n %s" % warn_messages
            log.debug("response body %s" % resp_body["body"])
        resp_body["headers"] = {"Content-Location":connectorURL}
        data = json.dumps(resp_body)
        yield Utils.prepareDataForChunkedEncoding(data)
        if ns:
            event_message = "App: %s deployed using: %s" % (str(uniqueConnectorId), CAFEvent.SOURCE_RESTAPI)
            ns.post_event(CAFEvent(uniqueConnectorId,
                                   CAFEvent.TYPE_DEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message=event_message))


        log.info("App: %s deployed successfully" % str(uniqueConnectorId))
        #Activate the App with given activation payload
        activate_warn_messages = controller.activate_app(uniqueConnectorId, app_resources, enable_debug, ignore_previous_mapping)
        warn_messages.extend(activate_warn_messages)
        bodystr = "Activated"
        if len(warn_messages) > 0:
            bodystr += "\n With warnings : \n %s" % warn_messages
            log.debug("response body %s" % bodystr)
        resp_body["status_code"] = falcon.HTTP_200
        resp_body["body"] = bodystr
        resp_body["headers"] = {}
        data = json.dumps(resp_body)
        yield Utils.prepareDataForChunkedEncoding(data)
        if ns:
            event_message = "App: %s activated using: %s" % (str(uniqueConnectorId), CAFEvent.SOURCE_RESTAPI)

            ns.post_event(CAFEvent(uniqueConnectorId,
                                   CAFEvent.TYPE_ACTIVATED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message=event_message))


        log.info("App: %s activated successfully" % str(uniqueConnectorId))
        #Update the App config if available
        if is_app_config_available:
            log.debug("App Config available")
            log.debug("Updating App config file %s", configvalue)
            controller.set_app_config(uniqueConnectorId, configvalue)
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Updated App config file"
            resp_body["headers"] = {}
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            log.info("App: %s config updated successfully" % str(uniqueConnectorId))

        start_app = headers.get('X-App-Start'.upper())
        if (start_app == '0' or start_app == 'off' or start_app is None or start_app == ""):
            start_app = False
        else:
            start_app = True
        if start_app is True:
            #Start the App
            controller.start_app(uniqueConnectorId)
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["headers"] = {}
            resp_body["body"] = "Started"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                event_message = "App: %s started using: %s" % (str(uniqueConnectorId), CAFEvent.SOURCE_RESTAPI)
                ns.post_event(CAFEvent(uniqueConnectorId,
                                       CAFEvent.TYPE_STARTED,
                                       CAFEvent.SOURCE_RESTAPI,
                                       event_message=event_message))
            log.info("App: %s started" % str(uniqueConnectorId))
    except Exception as ex:
        log.exception("Exception while processing the ztr install request %s", str(ex))
        resp_body["status_code"] = 500
        resp_body["headers"] = {}
        resp_body["body"] = str(ex)
        data = json.dumps(resp_body)
        #In case any exception happens, clean up everything since this is atomic operation
        controller.remove_app(uniqueConnectorId)
        yield Utils.prepareDataForChunkedEncoding(data)
    finally:
        log.debug("ztr install , invoking finally block")
        shutil.rmtree(tempdir, ignore_errors=True)
        os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
        ovaPath = os.path.join(tmpUploadDir, uniqueConnectorId + '.ova')
        os.remove(ovaPath) if os.path.exists(ovaPath) else None
        if pkg:
            pkg.close()
    if resp_body["status_code"] != falcon.HTTP_200 and resp_body["status_code"] != falcon.HTTP_201:
        log.error("Error occurred for the ztr POST request,last response status code:%s, resp body:%s", resp_body.get('status_code'), resp_body.get('body'))
        return

def change_app_state(controller, ns, app_id, target_state):
        """
        This function only changes app state from running onwards to deploy state not vice versa
        """
        log.debug("change_app_state:%s" % target_state)
        #ns = APIService.instance.hosting_manager.get_service("notification-service")
        try:
            app_info = controller.get(app_id)
            app_current_state = app_info.get_internal_status()
            log.debug("App current state: %s" % app_current_state)
            if target_state == app_current_state:
                return
            if app_current_state == State.DEPLOYED:
                return
            if app_current_state == State.STOPPED and target_state == State.ACTIVATED:
                return
            if app_current_state == State.ACTIVATED and target_state == State.STOPPED:
                return
            if app_current_state == State.RUNNING:
                controller.stop_app(app_id)
                if ns:
                    event_message = "App: %s stopped using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
                    ns.post_event(CAFEvent(app_id,
                                           CAFEvent.TYPE_STOPPED,
                                           CAFEvent.SOURCE_RESTAPI,
                                           event_message=event_message))

                log.info("App: %s stopped" % (str(app_id)))
                app_current_state = State.STOPPED
                if target_state ==  State.STOPPED or target_state == State.ACTIVATED:
                    return

            if app_current_state == State.ACTIVATED or app_current_state == State.STOPPED:
                controller.deactivate_app(app_id)
                if ns:
                    event_message = "App: %s deactivated using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
                    ns.post_event(CAFEvent(app_id,
                                           CAFEvent.TYPE_DEACTIVATED,
                                           CAFEvent.SOURCE_RESTAPI,
                                           event_message=event_message))
                log.info("App: %s deactivated" % (str(app_id)))
                return
        except Exception as ex:
            log.exception("Error while changing the state while processing the ZTR upgrade request %s", str(ex))
            raise ex



def ztr_upgrade_response_generator(controller, ns, app_id, filePath, preserveData, is_cleanup, tmpUploadDir,
                                   uri_str, app_resources={}, app_config={}, headers={}, target_state=None, current_state=None, event_source=CAFEvent.SOURCE_RESTAPI, auto_upgrade=False):
    """
    This API will take the application package, upgrades it and get it to desired state defined or same as the old app state.
    :param controller: This is the controller service instance. With this we will be performing all app installation steps.
    :param ns: This is the notification service instance. By using this we can post events to CAF.
    :param pkg: This is a package instance, with which we can get all required info about the package needs to installed.
    :param uniqueConnectorId: Id with which app needs to get installed.
    :param filePath: Location of the package to install.
    :param is_cleanup: Once the operation is done, tells whether to remove the source package or not.
    :param tmpUploadDir: Temp file system location to used for all temporary file creation operations.
    :param uri_str: Tells URI path of the application.
    :param app_resources: Payload to be used while activation the app.
    :param app_config: Boot strap config needs to be set for the application.
    :param headers: Headers provided by the user.
    :param target_state: Once app is upgraded. This will define which state it should be there.
    :param current_state: If upgrade failed tells which state old app has to be brought.
    :param event_source: Tells source of the events get logged.
    :return: Generator object.
    """
    if controller is None:
        log.error("Controller object is None, so will be not be able to process with any app lifecycle operations!")
        raise ServiceDepenendencyError("Controller object is None, so will be not be able to process with any app lifecycle operations!")
    log.debug("ztr upgrade request , filepath: %s, clean up %s", filePath, is_cleanup)
    log.debug("Activation payload %s ", app_resources)
    resp_body = {}
    configvalue = None
    logLevel = None
    backup_app_package = None
    restore_previous_package=False
    is_app_config_available = False
    try:
        if app_config:
            log.debug("App Config available")
            configvalue = app_config["config"]
            if app_config.has_key("log-level"):
                logLevel = app_config["log-level"]
            is_app_config_available = True
    except Exception as ex:
        log.warning("App config file not available for update %s", str(ex))
        is_app_config_available = False

    try:
        tempdir = tempfile.mkdtemp("", "tmpExtract", tmpUploadDir)
        log.debug("Input extraction dir: %s"  % tempdir)
        app_info = controller.get(app_id)
        app_current_state = app_info.get_internal_status()
        # If there is no target state specified for upgraded app then its state will be old apps state
        new_app_status = app_current_state
        if auto_upgrade:
            controller.PDHook.call_app_lifecycle_hook(app_info.appType,
                                     controller.PDHook.PRE_AUTO_UPGRADE, app_info.connector.metadata.app_env,
                                     app_id, app_info.app_repo_path,
                                     filePath)
        else:
            controller.PDHook.call_app_lifecycle_hook(app_info.appType,
                                     controller.PDHook.PRE_ZTR_UPGRADE, app_info.connector.metadata.app_env,
                                     app_id, app_info.app_repo_path,
                                     filePath)
        if app_current_state == State.RUNNING:
            controller.stop_app(app_id)
            resp_body["headers"] = {}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Stopped"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                event_message = "App: %s stopped using: %s" % (str(app_id), event_source)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_STOPPED,
                                       event_source,
                                       event_message=event_message))

            log.info("App: %s stopped" % (str(app_id)))
            controller.deactivate_app(app_id)
            resp_body["headers"] = {}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Deactivated"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                event_message = "App: %s deactivated using: %s" % (str(app_id), event_source)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_DEACTIVATED,
                                       event_source,
                                       event_message=event_message))
            log.info("App: %s deactivated" % (str(app_id)))
        elif app_current_state == State.ACTIVATED or app_current_state == State.STOPPED:
            controller.deactivate_app(app_id)
            resp_body["headers"] = {}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Deactivated"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                event_message = "App: %s deactivated using: %s" % (str(app_id), event_source)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_DEACTIVATED,
                                       event_source,
                                       event_message=event_message))
            log.info("App: %s deactivated" % (str(app_id)))

        else:
            log.debug("App is already in deployed state %s ", app_id)
        if current_state:
            app_current_state = current_state
            new_app_status = current_state
        #Invoke the Upgrade API
        try:
            #Backup the old package
            backup_app = controller.backup_app(app_id)
            controller.upgrade_app(app_id, filePath, preserveData, tempdir, is_package_cleanup=is_cleanup, backup_app=backup_app)
            app_custom_options = headers.get('X-App-Custom-Options'.upper())
            if app_custom_options:
                log.debug('Setting application custom options')
                controller.set_app_custom_options(app_id, app_custom_options)
            connectorURL = uri_str + "/" + app_id
            resp_body["headers"] = {"Content-Location": connectorURL}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Upgrade successfull"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                log.debug("Posting 'upgraded' event on %s" % str(app_id))
                event_message = "App: %s upgraded using: %s" % (str(app_id), event_source)
                event_type = CAFEvent.TYPE_UPGRADED
                if auto_upgrade:
                    event_type = CAFEvent.TYPE_AUTO_UPGRADED
                ns.post_event(CAFEvent(app_id,
                                       event_type,
                                       event_source,
                                       event_message=event_message))
            log.info("App: %s upgraded" % (str(app_id)))
            restore_previous_package = True
            # If target state for upgraded app defined then define same to new_app_state var.
            if target_state:
                new_app_status = target_state
        except Exception as ex:
            #Old App package would be restored incase upgrade failed
            #App resources file will be restored back in that case
            #So do not pass the activation payload received as part of this request
            #The old resources file will be loaded
            app_resources = {}
            log.exception("Upgrade of the App %s failed ", app_id)
            #uri_str = request.uri.rsplit('/', 1)[0]
            connectorURL = uri_str + "/" + app_id
            resp_body["headers"] = {"Content-Location": connectorURL}
            resp_body["status_code"] = falcon.HTTP_500
            resp_body["body"] = "Upgrade failed, previous package will be restored"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                log.debug("Posting Upgrade of the App %s failed" % str(app_id))
                event_message = "App: %s Upgrade failed, previous package will be restored: %s" % (str(app_id), event_source)
                event_type = CAFEvent.TYPE_UPGRADED_FAILED
                if auto_upgrade:
                    event_type = CAFEvent.TYPE_AUTO_UPGRADED_FAILED
                ns.post_event(CAFEvent(app_id,
                                       event_type,
                                       event_source,
                                       event_message=event_message))
        if new_app_status == State.RUNNING:
            warn_messages = controller.activate_app(app_id, app_resources)
            bodystr = "Activated"
            if len(warn_messages) > 0:
                bodystr += "\n With warnings : \n %s" % warn_messages
                log.debug("response body %s" % bodystr)
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = bodystr
            resp_body["headers"] = {}
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                event_message = "App: %s activated using: %s" % (str(app_id), event_source)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_ACTIVATED,
                                       event_source,
                                       event_message=event_message))

            log.info("App: %s activated" % (str(app_id)))

            #Update the App config if available
            if is_app_config_available:
                #ConnectorConfig.updateConnectorConfiguration(app_id, configvalue, logLevel)
                controller.set_app_config(app_id, configvalue)
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "Updated App config file"
                resp_body["headers"] = {}
                data = json.dumps(resp_body)
                yield Utils.prepareDataForChunkedEncoding(data)

            controller.start_app(app_id)
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["headers"] = {}
            resp_body["body"] = "Started"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)

            if ns:
                event_message = "App: %s started using: %s" % (str(app_id), event_source)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_STARTED,
                                       event_source,
                                       event_message=event_message))
            log.info("App: %s started" % (str(app_id)))
            # If this flag is set means upgrade went through fine.
            if restore_previous_package:
                post_upgrade_config = True
                post_upgrade_config = headers.get('X-App-Execute-Post-Upgrade'.upper())
                if post_upgrade_config:
                    if (post_upgrade_config.strip().lower() == '0' or post_upgrade_config.strip().lower() == 'no' or post_upgrade_config.strip().lower()=='false'):
                        post_upgrade_config = False
                else:
                    post_upgrade_config = True

                log.debug("Post upgrade Execute script:%s" % post_upgrade_config)
                if post_upgrade_config:
                    post_upgrade_script_timeout = headers.get('X-App-Post-Upgrade-Timeout'.upper())
                    if not post_upgrade_script_timeout:
                        post_upgrade_script_timeout = 120
                    else:
                        post_upgrade_script_timeout = int(post_upgrade_script_timeout)
                        if post_upgrade_script_timeout <= 0:
                            post_upgrade_script_timeout = 120
                    log.debug("Post upgrade script timeout:%s" % post_upgrade_script_timeout)
                    app_info = controller.get(app_id)
                    post_upgrade_script_resp = run_post_upgrade_script(app_info, post_upgrade_script_timeout)
                    if post_upgrade_script_resp:
                        yield Utils.prepareDataForChunkedEncoding(post_upgrade_script_resp)

        elif new_app_status == State.ACTIVATED or new_app_status == State.STOPPED:
            warn_messages = controller.activate_app(app_id, app_resources)
            bodystr = "Activated"
            if len(warn_messages) > 0:
                bodystr += "\n With warnings : \n %s" % warn_messages
                log.debug("response body %s" % bodystr)
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = bodystr
            resp_body["headers"] = {}
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)

            if ns:
                event_message = "App: %s activated using: %s" % (str(app_id), event_source)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_ACTIVATED,
                                       event_source,
                                       event_message=event_message))
            log.info("App: %s activated" % (str(app_id)))

            #Update the App config if available
            if is_app_config_available:
                controller.set_app_config(app_id, configvalue)
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "Updated App config file"
                resp_body["headers"] = {}
                data = json.dumps(resp_body)
                yield Utils.prepareDataForChunkedEncoding(data)
        else:
            log.debug("App was in deployed state before upgrade %s", app_id)

        controller.cleanup_backup_app(backup_app)
        connectorURL = uri_str + "/" + app_id
        respHeaders = {"Content-Location": connectorURL}

        resp_body["status_code"] = falcon.HTTP_200
        resp_body["body"] = "Upgrade process successful and bought the the app back to previous state before upgrade"
        resp_body["headers"] = respHeaders
        data = json.dumps(resp_body)
        yield Utils.prepareDataForChunkedEncoding(data)
        post_app_info = controller.get(app_id)
        if auto_upgrade:
            if restore_previous_package:
                controller.PDHook.call_app_lifecycle_hook(post_app_info.appType,
                                         controller.PDHook.POST_AUTO_UPGRADE, post_app_info.connector.metadata.app_env,
                                         app_id, post_app_info.app_repo_path,
                                         filePath)
            else:
                controller.PDHook.call_app_lifecycle_hook(post_app_info.appType,
                                         controller.PDHook.AUTO_UPGRADE_FAILED, post_app_info.connector.metadata.app_env,
                                         app_id, post_app_info.app_repo_path,
                                         filePath)
        else:
            if restore_previous_package:
                controller.PDHook.call_app_lifecycle_hook(post_app_info.appType,
                                         controller.PDHook.POST_ZTR_UPGRADE, post_app_info.connector.metadata.app_env,
                                         app_id, post_app_info.app_repo_path,
                                         filePath)
            else:
                controller.PDHook.call_app_lifecycle_hook(post_app_info.appType,
                                         controller.PDHook.ZTR_UPGRADE_FAILED, post_app_info.connector.metadata.app_env,
                                         app_id, post_app_info.app_repo_path,
                                         filePath)
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        hasync_service = hm.get_service("hasync-service")
        if restore_previous_package and hasync_service:
            hasync_service.sync_caf_data("file", os.path.join(post_app_info.app_repo_path, app_id + ".tar"))
            hasync_service.sync_caf_data("op")
    except Exception as ex:
        log.exception("Error while processing the ZTR upgrade request %s", str(ex))
        resp_body["status_code"] = 500
        resp_body["headers"] = {}
        resp_body["body"] = str(ex)
        data = json.dumps(resp_body)
        yield Utils.prepareDataForChunkedEncoding(data)
        if ns:
            event_message = "App: %s, Error while upgrading the app: %s" % (str(app_id), event_source)
            event_type = CAFEvent.TYPE_UPGRADED_FAILED
            if auto_upgrade:
                event_type = CAFEvent.TYPE_AUTO_UPGRADED_FAILED
            ns.post_event(CAFEvent(app_id,
                                   event_type,
                                   event_source,
                                   event_message=event_message))
        # This condition is needed in cases of auto upgrade while CAF is starting.
        # When auto upgrade is triggered while starting CAF, that time app will be DEPLOYED state,
        # but 'current_state' will define the actual state. So while upgarading anything happens old app
        # has to get back to its actual state.
        if current_state:
            app_current_state = current_state
        recover_old_app = True
        recover_old_app = headers.get('X-App-Recover-App'.upper())
        if recover_old_app:
            if (recover_old_app.strip().lower() == '0' or recover_old_app.strip().lower() == 'no' or recover_old_app.strip().lower()=='false'):
                recover_old_app = False
        else:
            #default is recover old app is True
            recover_old_app = True
        log.debug("Recover old app:%s" % recover_old_app)
        if recover_old_app and restore_previous_package:
            try:
                log.debug("Going to restore the previous package")
                change_app_state(controller, ns, app_id, State.DEPLOYED)
                log.debug("After state change")

                controller.restore_app(app_id, backup_app)

                connectorURL = uri_str + "/" + app_id
                resp_body["headers"] = {"Content-Location": connectorURL}
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "Restore of previous version of app successfull"
                data = json.dumps(resp_body)
                yield Utils.prepareDataForChunkedEncoding(data)
                if ns:
                    event_message = "App: %s, Successfully deployed the previous app: %s" % (str(app_id), event_source)
                    ns.post_event(CAFEvent(app_id,
                                           CAFEvent.TYPE_DEPLOYED,
                                           event_source,
                                                   event_message=event_message))

                #App resources file will be restored back in that case
                #So do not pass the activation payload received as part of this request
                #The old resources file will be loaded
                app_resources = {}

                if app_current_state == State.RUNNING or app_current_state == State.ACTIVATED or app_current_state == State.STOPPED:
                    warn_messages = controller.activate_app(app_id, app_resources)
                    bodystr = "Activated"
                    if len(warn_messages) > 0:
                        bodystr += "\n With warnings : \n %s" % warn_messages
                        log.debug("response body %s" % bodystr)
                    resp_body["status_code"] = falcon.HTTP_200
                    resp_body["body"] = bodystr
                    resp_body["headers"] = {}
                    data = json.dumps(resp_body)
                    yield Utils.prepareDataForChunkedEncoding(data)
                    if ns:
                        event_message = "App: %s activated using: %s" % (str(app_id), event_source)
                        ns.post_event(CAFEvent(app_id,
                                               CAFEvent.TYPE_ACTIVATED,
                                               event_source,
                                               event_message=event_message))

                    log.info("App: %s activated" % (str(app_id)))

                    if app_current_state == State.RUNNING:
                        controller.start_app(app_id)
                        resp_body["status_code"] = falcon.HTTP_200
                        resp_body["headers"] = {}
                        resp_body["body"] = "Started"
                        data = json.dumps(resp_body)
                        yield Utils.prepareDataForChunkedEncoding(data)

                        if ns:
                            event_message = "App: %s started using: %s" % (str(app_id), event_source)
                            ns.post_event(CAFEvent(app_id,
                                                   CAFEvent.TYPE_STARTED,
                                                   event_source,
                                                   event_message=event_message))
                        log.info("App: %s started" % (str(app_id)))
            except Exception as ex:
                log.exception("Failed to restore the app to previous state:%s" % str(ex))
                resp_body["status_code"] = 500
                resp_body["headers"] = {}
                resp_body["body"] = str(ex)
                data = json.dumps(resp_body)
                yield Utils.prepareDataForChunkedEncoding(data)
                if ns:
                    event_message = "App: %s, Failed to restore the previous app: %s" % (str(app_id), event_source)
                    ns.post_event(CAFEvent(app_id,
                                           CAFEvent.TYPE_RESTORE_FAILED,
                                           event_source,
                                                   event_message=event_message))
        app_info_exception = controller.get(app_id)
        if auto_upgrade and app_info_exception:
            controller.PDHook.call_app_lifecycle_hook(app_info_exception.appType,
                                     controller.PDHook.AUTO_UPGRADE_FAILED, app_info_exception.connector.metadata.app_env,
                                     app_id, app_info_exception.app_repo_path,
                                     filePath)
        elif app_info_exception:
            controller.PDHook.call_app_lifecycle_hook(app_info_exception.appType,
                                     controller.PDHook.ZTR_UPGRADE_FAILED, app_info_exception.connector.metadata.app_env,
                                     app_id, app_info_exception.app_repo_path,
                                     filePath)
    finally:
        log.debug("ZTR Upgrade , invoking final block")
    os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
    ovaPath = os.path.join(tmpUploadDir, app_id + '.ova')
    os.remove(ovaPath) if os.path.exists(ovaPath) else None


def ztr_delete_response_generator(controller, ns, app_id):
    """
    This API will delete the apps regrdless of their current state.
    :param controller: This is the controller service instance. With this we will be performing all app installation steps.
    :param ns: This is the notification service instance. By using this we can post events to CAF.
    :param app_id: App id which needs to be deleted.
    :return: Generator object
    """
    if controller is None:
        log.error("Controller object is None, so will be not be able to process with any app lifecycle operations!")
        raise ServiceDepenendencyError("Controller object is None, so will be not be able to process with any app lifecycle operations!")
    resp_body = {}
    try:
        app_info = controller.get(app_id)
        app_current_state = app_info.get_internal_status()
        if app_current_state == State.RUNNING:
            controller.stop_app(app_id)
            resp_body["headers"] = {}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Stopped"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)

            if ns:
                event_message = "App: %s stopped using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_STOPPED,
                                       CAFEvent.SOURCE_RESTAPI,
                                       event_message=event_message))
            log.info("App: %s stopped" % (str(app_id)))

            controller.deactivate_app(app_id)
            resp_body["headers"] = {}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Deactivated"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                event_message = "App: %s deactivated using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_DEACTIVATED,
                                       CAFEvent.SOURCE_RESTAPI,
                                       event_message=event_message))
            log.info("App: %s deactivated" % (str(app_id)))

        elif app_current_state == State.ACTIVATED or app_current_state == State.STOPPED:
            controller.deactivate_app(app_id)
            resp_body["headers"] = {}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Deactivated\n"
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            if ns:
                event_message = "App: %s deactivated using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
                ns.post_event(CAFEvent(app_id,
                                       CAFEvent.TYPE_DEACTIVATED,
                                       CAFEvent.SOURCE_RESTAPI,
                                       event_message=event_message))
            log.info("App: %s deactivated" % (str(app_id)))

        else:
            log.debug("App %s is in deployed state", app_id)
        app_manager = controller
        app_manager.uninstall_app(app_id)
        resp_body["status_code"] = falcon.HTTP_200
        resp_body["body"] = "Uninstalled\n"
        resp_body["headers"] = {}
        data = json.dumps(resp_body)
        yield Utils.prepareDataForChunkedEncoding(data)
        if ns:
            log.debug("Posting 'undeployed' event on %s" % str(app_id))
            event_message = "App: %s undeployed using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
            ns.post_event(CAFEvent(app_id,
                                   CAFEvent.TYPE_UNDEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message = event_message))
    except Exception as ex:
        log.exception("Error while processing the ZTR delete method %s", str(ex))
        resp_body["status_code"] = 500
        resp_body["headers"] = {}
        resp_body["body"] = str(ex)
        data = json.dumps(resp_body)
        yield Utils.prepareDataForChunkedEncoding(data)

def run_post_upgrade_script(app_info, post_upgrade_script_timeout):
    """
    will run the post upgrade script and return the appropriate response
    """
    resp_body = {}
    if hasattr(app_info, "post_upgrade") :
        post_upgrade = app_info.post_upgrade
        if post_upgrade:
            log.debug("App post upgrade: %s" % str(post_upgrade))
            app_post_script = post_upgrade.get("post_script", None)
            if app_post_script:
                app_initial_wait_time = post_upgrade.get("initial_wait_time", 30)
                if app_initial_wait_time > 300:
                    #Do not block for more than 300 seconds
                    app_initial_wait_time = 300
                time.sleep(app_initial_wait_time)
                while (time.time() - app_info.last_started_time <
                                             app_initial_wait_time):
                    time.sleep(1)
                container = app_info._container
                if container and container.isRunning():
                    log.debug("Going to execute post upgrade script:%s" % app_post_script)
                    #Execute post upgrade script
                    post_upgrade_out, err, rv = container.execute_health_script(app_post_script, timeout=post_upgrade_script_timeout)

                    log.debug("Executed post upgrade script: %s rv:%s output:%s error:%s" %
                                 (app_post_script, rv, post_upgrade_out, err))
                    app_info.post_upgrade_status = rv
                    app_info.post_upgrade_output = post_upgrade_out
                    app_info.post_upgrade_err = err
                    if rv != 0:
                        log.error("App post upgrade script return failure %s out:%s error:%s" % (rv, post_upgrade_out, err))
                        raise Exception("Post upgrade script failure :%s Output: %s Error:%s" %(rv, post_upgrade_out, err))
                    else:
                        log.info("Post upgrade script successfully executed:%s Output:%s" % (rv, post_upgrade_out))
                        resp_body["status_code"] = falcon.HTTP_200
                        resp_body["headers"] = {}
                        resp_body["body"] = "Post upgrade script successful"
                        return json.dumps(resp_body)


def ztr_state_change_response_generator(controller, ns, app_id, target_state, resp_app_status=False):
    """
    This API will put the app from any state to desired state
    :param controller: This is the controller service instance. With this we will be performing all app state change steps.
    :param ns: This is the notification service instance. By using this we can post events to CAF.
    :param app_id: App for which state needs to be changed
    :param target_state: target of the app
    :return: Generator object
    """
    log.debug("Going to change the app: %s state to : %s "%(app_id, target_state))
    if controller is None:
        log.error("Controller object is None, so will be not be able to process with any app lifecycle operations!")
        raise ServiceDepenendencyError("Controller object is None, so will be not be able to process with any app lifecycle operations!")
    resp_body = {}
    try:
        app_info = controller.get(app_id)
        if target_state == State.RUNNING:
            if app_info.get_internal_status() == target_state:
                log.info("App is already in running state, so nothing to do!")
                resp_body["headers"] = {}
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "STARTED"
                if resp_app_status:
                    resp_body["app_checksums"] = controller.get_app_checksums(app_id)
                data = json.dumps(resp_body)
                yield Utils.prepareDataForChunkedEncoding(data)
            if app_info.get_internal_status() == State.DEPLOYED:
                controller.activate_app(app_id, {})
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["headers"] = {}
                resp_body["body"] = "ACTIVATED"
                if resp_app_status:
                    resp_body["app_checksums"] = controller.get_app_checksums(app_id)
                data = json.dumps(resp_body)
                yield Utils.prepareDataForChunkedEncoding(data)
                if ns:
                    event_message = "App: %s activated using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
                    ns.post_event(CAFEvent(app_id,
                                           CAFEvent.TYPE_ACTIVATED,
                                           CAFEvent.SOURCE_RESTAPI,
                                            event_message=event_message))
                log.info("App %s got activated!"%app_id)
            if app_info.get_internal_status() == State.ACTIVATED or app_info.get_internal_status() == State.STOPPED:
                controller.start_app(app_id)
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["headers"] = {}
                resp_body["body"] = "STARTED"
                if resp_app_status:
                    resp_body["app_checksums"] = controller.get_app_checksums(app_id)
                data = json.dumps(resp_body)
                yield Utils.prepareDataForChunkedEncoding(data)
                if ns:
                    event_message = "App: %s started using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
                    ns.post_event(CAFEvent(app_id,
                                           CAFEvent.TYPE_STARTED,
                                           CAFEvent.SOURCE_RESTAPI,
                                           event_message=event_message))
                log.info("App: %s started" % (str(app_id)))
        else:
            raise ValueError("Given target state: %s is invalid!"%target_state)
    except Exception as ex:
        log.exception("Error while processing the ZTR delete method %s", str(ex))
        resp_body["status_code"] = 500
        resp_body["headers"] = {}
        resp_body["body"] = str(ex)
        data = json.dumps(resp_body)
        yield Utils.prepareDataForChunkedEncoding(data)
