'''

@author: havishwa

Copyright (c) 2014-2015 by Cisco Systems, Inc.
All rights reserved.
'''

import tarfile
import shutil
import os
import sys
import logging
from appfw.utils.utils import Utils, CARTRIDGE_MANIFEST_NAME, CARTRIDGE_AUTOINSTALL_FILE_NAME, CARTRIDGE_AUTOINSTALL_ORDER_FILE_NAME, CARTRIDGE_LANG_PREFIX
from manifest import CartridgeManifest
from appfw.runtime.languages import LanguageRuntimes
import re
from appfw.runtime.error import MandatoryMetadataError
from ..utils.infraexceptions import CartridgeDependError, C3Exception, CartridgeDescriptorValidationError, AppTypeNotSupportedError
from ..utils.cafevent import CAFEvent
from appfw.runtime.caf_abstractservice import CAFAbstractService
import tempfile
import subprocess
from format_handler import FormatHandler
from ConfigParser import NoOptionError

log = logging.getLogger("cartridge")

class Cartridge(object):
    _to_serialize = ("id", "name", "description", "version", "author",
                     "authorLink", "runtime", "runtime_version", "cpuarch",
                     "payload", "type", "handleas", "provides_info", "dependson")


    def __init__(self, cartridge_id, cartridgedir, cartridge_mount_path):
        self._id = cartridge_id
        self._cartridgedir = cartridgedir
        self._cartridge_mount_path = cartridge_mount_path
        self._manifestfile = os.path.join(self._cartridgedir, CARTRIDGE_MANIFEST_NAME)
        self._manifest = CartridgeManifest(self._manifestfile)

        cartridge_payload = self._manifest.payload 
        if cartridge_payload is None:
            log.error("Mandatory key payload not found in cartridge descriptor file" )
            raise ValueError("Mandatory key payload not found in cartridge descriptor file")
        
        cartridge_payload_path =  os.path.join(self._cartridgedir, cartridge_payload)
        self._size = os.path.getsize(cartridge_payload_path)
        if not os.path.exists(cartridge_payload_path):
            log.error("Payload %s not found after extracting" % cartridge_payload)
            raise ValueError("Payload %s not found after extracting" % cartridge_payload)
            
        fmt_hdlr = FormatHandler.get_format_handler(self._manifest.format)
        #self._cart_mount = os.path.join(cartridgedir, "mntpoint")
        log.debug("cartridge target dir: %s", self._cartridge_mount_path)
        if not os.path.isdir(self._cartridge_mount_path):
            os.makedirs(self._cartridge_mount_path)

        fmt_hdlr.get_payload(cartridge_payload_path, self._cartridge_mount_path)


    @property
    def id(self):
        return self._id

    @property
    def cartridgedir(self):
        return self._cartridgedir

    @property
    def cartridge_mount_path(self):
        return self._cartridge_mount_path

    @property
    def manifest(self):
        return self._manifest

    @property
    def manifestfile(self):
        return self._manifestfile

    @property
    def manifest_version(self):
        return self.manifest.manifest_version.strip()

    @property
    def info(self):
        return self.manifest.info


    @property
    def name(self):
        return self.manifest.name.strip()

    @property
    def description(self):
        return self.manifest.description

    @property
    def author(self):
        return self.manifest.author

    @property
    def authorLink(self):
        return self.manifest.authorLink

    @property
    def version(self):
        return self.manifest.version

    #################################

    @property
    def runtime(self):
        return self.manifest.runtime.rstrip().lstrip()

    @property
    def runtime_version(self):
        return self.manifest.runtime_version.rstrip().lstrip()

    @property
    def cpuarch(self):
        return self.manifest.cpuarch

    @property
    def handleas(self):
        return self.manifest.handleas

    @property
    def provides(self):
        return self.manifest.provides

    @property
    def provides_info(self):
        cm =  CartridgeManager.getInstance()
        prov_list = []
        log.debug("Cartridge provides: %s" % str(self.provides))
        for cart in self.provides:
            prov = {}
            prov['id'] = cart["id"]
            prov['version'] = cart["version"]
            cart_detail = cm.get_cartridge_info(cart["id"], cart["version"])
            log.debug("Cart Detail : %s" , str(cart_detail))
            if cart_detail:
                prov["used_by"] = cart_detail['used_by'] 
            prov_list.append(prov)
        log.debug("Provided Detail list: %s" , str(prov_list))
        return prov_list 

    @property
    def dependson(self):
        return self.manifest.dependson

    @property
    def type(self):
        return self.manifest.type

    @property
    def format(self):
        return self.manifest.format

    @property
    def dirname(self):
        return self.manifest.dirname

    @property
    def payload(self):
        return self.manifest.payload

    @property
    def env(self):
        return self.manifest.env

    @property
    def cart_size(self):
        return self._size

    def get_location(self):
        """
        return filesystem location of the cartridge directory
        :return:
        """
        return self._cartridge_mount_path

    def get_commands(self, cartridge_root):
        """
        Return a list of commands needed by this cartridge.
        They may include setting up of env variables, bringing up necessary processes etc.,

        :param cartridge_root: the location where cartridge is made available to the app
        :return:
        """
        # For now it is just env commands
        return self.get_env_commands(cartridge_root)

    def get_env_commands(self, cartridge_root):
        """
        Returns a set of export commands based on cartridge's env section
        :param cartridge_root: the location where cartridge is made available to the app
        :return:
        """
        rval = []
        if not self.env:
            return rval
        import copy
        envc = copy.copy(self.env)
        r_str =  cartridge_root.rstrip("/")    
        for entry in envc:
            ec = copy.copy(entry)
            k, v = ec.popitem()
            v = v.replace("$CARTRIDGE_PREFIX", r_str)
            str = "export %s=%s:$%s" % (k, v, k)
            rval.append(str)
        return rval

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def __repr__(self):
        return self.__str__()

    def __str__(self):
        return "Cartridge ID: %s, Dir: %s" % (self.id, self.cartridgedir)

    def remove(self):
        fmt_hdler = FormatHandler.get_format_handler(self._manifest.format)
        fmt_hdler.remove(self.get_location())


class CartridgeManager(CAFAbstractService):
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(CartridgeManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = CartridgeManager(*args)
        return cls.__singleton

    def __init__(self, config):
        self._config = config
        self._cartridge_root = config.get("cartridge", "root")
        self._cartridge_auto_install = True
        self._corrupted_cartids = []
        try:
            self._autoinstall_path = config.get("cartridge", "auto_install")
        except NoOptionError:
            log.debug("Cartridge auto install is disabled on system")
            self._cartridge_auto_install = False

        if config.has_option("cartridge", "cartridge_mount_path"):
            self._cartridge_mount_path = config.get("cartridge", "cartridge_mount_path")
        else:
            self._cartridge_mount_path = os.path.join(self._cartridge_root, "mnt")

        # Read tmp location from system config, default to /tmp if not specified
        self.tmpUploadDir = '/tmp'
        if self._config.has_option("controller", "upload_dir"):
            self.tmpUploadDir = self._config.get("controller", "upload_dir")

        if not os.path.exists(self.tmpUploadDir):
            os.makedirs(self.tmpUploadDir)

        self.check_mandatory_files = True
        self.check_integrity = True
        self.check_signature = False
        self.languageruntimes = LanguageRuntimes.getInstance()
        if not os.path.isdir(self._cartridge_root):
            os.makedirs(self._cartridge_root)
        self._cartridges = dict()
        self.cartridgeMap = dict()
        self.loadOrder = []
        self._load()
        self.autoinstall_file_name = CARTRIDGE_AUTOINSTALL_FILE_NAME
        self.autoinstall_order_file_name = CARTRIDGE_AUTOINSTALL_ORDER_FILE_NAME
        if self._cartridge_auto_install:
            self.auto_install(self._autoinstall_path)

    def get_config(self):
        return dict(self._config.__dict__['_sections']['cartridge'])

    def get_cartridge_map(self):
        return self.cartridgeMap

    def get_cartridge_info(self, prov_id, prov_version):
        '''
        Returns the existing cartrdige info for depolyed cartridge
        '''
        try:
            log.debug("Looking for id: %s" , prov_id)
            log.debug("cartridge map: %s" , self.cartridgeMap)
            if not prov_id in self.cartridgeMap:
                log.debug("%s: not found in cartridge map", prov_id) 
                return None   
            inst_cart_list = self.cartridgeMap.get(prov_id)
            if inst_cart_list:
                idx = -1
                for inst_cart in inst_cart_list:
                    if prov_version == inst_cart['version']:
                        cart_details={}
                        cart_details['id'] = prov_id
                        cart_details['version']=prov_version
                        cart_details['cartridge_id'] =  inst_cart['cartridge_id']
                        cart_details['type'] =  inst_cart['type']
                        cart_details['used_by'] =  inst_cart['used_by']
                        return cart_details
        except Exception as e:
            log.error("Exception while retireving cartridge info %s %s" % (prov_id, str(e)))
            return None

    def remove(self, cartridge_id):
        if cartridge_id in self._cartridges:
            c = self._cartridges.get(cartridge_id)
            #update cartridgeMap with the new entries
            for cart_info in c.provides:
                inst_cart_list = self.cartridgeMap[cart_info['id']]
                if inst_cart_list:
                    idx = -1
                    for inst_cart in inst_cart_list:
                        if cart_info['version'] == inst_cart['version']:
                            idx =inst_cart_list.index(inst_cart)
                            if len(inst_cart['used_by']) > 0:
                                log.error("Cartridge %s is used by %s so cannot delete %s "
                                    % (cart_info['id'], inst_cart['used_by'], cartridge_id))
                                raise CartridgeDependError("Cartridge %s is used by %s so cannot delete %s" % (cart_info['id'], inst_cart['used_by'], cartridge_id))
                            break
                    if idx != -1:
                        log.debug("Deleting cartride entry %s from list" , inst_cart_list[idx])
                        del inst_cart_list[idx]
                if inst_cart_list is None or len(inst_cart_list) == 0:     
                    #Remove the cartinfo from internal map
                    log.info("Deleting cartridge %s from cartridge map" % cart_info['id'])
                    self.cartridgeMap.pop(cart_info['id'],None)
            #Remove the cartridge_id from used_by list in cartridgeMap
            self.remove_used_by(cartridge_id, "cartridge")

            c = self._cartridges.pop(cartridge_id)
            # remove the payload
            c.remove()
            shutil.rmtree(os.path.join(self._cartridge_root, cartridge_id),ignore_errors=True)
            shutil.rmtree(os.path.join(self._cartridge_mount_path, cartridge_id),ignore_errors=True)
            log.info("Removed cartridge entry. %s", c)
            if cartridge_id in self.loadOrder:
                self.loadOrder.remove(cartridge_id)
                self.setStartOrder(self.loadOrder)
    
    def remove_used_by(self, dep_id, res_type):
        #Remove the dep_id from used_by list in cartridgeMap
        res_rm = {"id": dep_id, "type": res_type}
        for cart_id, cart_list in self.cartridgeMap.iteritems():
            for cart in cart_list:
                if res_rm in cart['used_by']:
                    log.debug("Removing %s from cartridge %s" % 
                                            (dep_id, str(cart_id)))
                    cart['used_by'].remove(res_rm)
                else:
                    log.debug("%s not found in used_by list" % str(res_rm))
        log.debug("Cartridge Map after deleting %s" , str(self.cartridgeMap))


    def listStartOrder(self):
        return self._startConfig.loadOrder

    def setStartOrder(self, resStartList):
        resOrder = ",".join(resStartList)
        self._startConfig.startOrder = resOrder

    def _install_cartridge(self, pkg, cartridge_id):
        """
        Create cartridge with given package and cartridge_id provided.
        """
        cartridge_dir = os.path.join(self._cartridge_root, cartridge_id)
        if not os.path.isdir(cartridge_dir):
            os.makedirs(cartridge_dir)
        
        try:
            pkg.move_extracted_contents(cartridge_dir)
            log.debug("Created cartridge directory: %s", cartridge_dir)
        except Exception as ex:
            log.exception("Unable to extract archive: %s" , str(ex))
            if os.path.isdir(cartridge_dir):
                shutil.rmtree(cartridge_dir, ignore_errors=True)
            raise ex
        try:
            self._create(cartridge_id, cartridge_dir)
            log.info("Cartridge:%s installed successfully" % cartridge_id)
            return cartridge_id
        except Exception as ex:
            if os.path.isdir(cartridge_dir):
                shutil.rmtree(cartridge_dir, ignore_errors=True)
            raise ex

    def add(self, cartridge_archive):
        """
        Generate the cartridge id and check for the existance of it.
        Validate the cartridge archive given.
        Install the cartridge.
        """
        from appfw.app_package.packagemanager import PackageManager
        tempdir = tempfile.mkdtemp(dir=self.tmpUploadDir)
        pkg = None
        try:
            pkg = PackageManager.getPackage(cartridge_archive, tempdir, 
                    check_integrity=self.check_integrity, 
                    check_signature=self.check_signature,
                    check_mandatory_files=self.check_mandatory_files)

            if not pkg.exists_in_package(CARTRIDGE_MANIFEST_NAME):
                raise MandatoryMetadataError("Missing descriptor file %s in the archive" % CARTRIDGE_MANIFEST_NAME)
            cart_manifest = CartridgeManifest(os.path.join(tempdir, CARTRIDGE_MANIFEST_NAME))
            cartridge_id = self._generate_cart_id(cart_manifest)
            exists = self.exists(cartridge_id)
            if exists:
                log.error("Cartridge already exists with the specified id : %s" % cartridge_id)
                raise Exception("Cartridge already exists with the specified id: %s" % cartridge_id)
            #Validating the cartridge manifest
            self.validate_manifest(cart_manifest)
            #Install the cartridge
            self._install_cartridge(pkg, cartridge_id)
            return cartridge_id
        except Exception as e:
            log.error("Error in cartridge archive: %s", str(e))
            raise e
        finally:
            if os.path.exists(tempdir):
                shutil.rmtree(tempdir, ignore_errors=True)
            if pkg:
                pkg.close()

    def upgrade(self, cart_id, cart_archive, preserve_old=False):
        """
        Returns None if upgrade is successful,
        If it fails raise the exception accordingly
        """
        log.debug("Upgrading cartridge %s" % cart_id)
        from appfw.app_package.packagemanager import PackageManager
        tmpdir = None
        pkg = None
        try:
            tmpdir = tempfile.mkdtemp(dir=self.tmpUploadDir)
            pkg = PackageManager.getPackage(cart_archive, tmpdir,
                    check_integrity=self.check_integrity,
                    check_signature=self.check_signature,
                    check_mandatory_files=self.check_mandatory_files)
            if not pkg.exists_in_package(CARTRIDGE_MANIFEST_NAME):
                raise MandatoryMetadataError("Missing descriptor file %s in the archive" % CARTRIDGE_MANIFEST_NAME)
            cart_manifest = CartridgeManifest(os.path.join(tmpdir, CARTRIDGE_MANIFEST_NAME))
            #Validating package before it goes for upgrade
            log.debug("Validating the cart handlers provided")
            self._validate_cart_handlers(cart_manifest)
            log.debug("Validating the dependencies for the cartridge")
            self._validate_cart_dependency(cart_manifest)
            cartridge = self.get(cart_id)
            if cartridge is None:
                raise Exception("Cartridge doesn't exists")
            else:
                try:
                    #Removing cartridge
                    log.debug("Removing the cartridge %s"%cart_id)
                    self.remove(cart_id)
                    self._validate_cart_provides(cart_manifest)
                    #Deploying the cartridge archive
                    log.debug("Installing the cartridge %s"%cart_id)
                    self._install_cartridge(pkg, cart_id)
                except Exception as ex:
                    log.error("Error while upgrading cartridge %s : cause %s" % (cart_id, str(ex)))
                    raise ex
        finally:
            if tmpdir:
                shutil.rmtree(tmpdir, ignore_errors=True)
            if pkg:
                pkg.close()


#Not used right now, to be used in future
    def _preserve_cart(self, cart_id):
        """
        Preserve the cartridge to a temp location
        :return: Location of the preserved cartridge
        """
        log.debug("Preserving cartridge")
        temp = tempfile.mkdtemp("_cartridge", dir=self.tmpUploadDir)
        cartridge = self.get(cart_id)
        source = cartridge.cartridgedir
        manifest_file = os.path.join(source, CARTRIDGE_MANIFEST_NAME)
        payload_path = ""
        shutil.copy2(manifest_file, temp)
        shutil.copy2(payload_path, temp)
        return temp

#Not used right now, to be used in future
    def _restore_cart(self, cart_dir):
        """
        Restores the cartridge from a location provided
        """
        pass

    def _validate_cart_dir(self, cart_dir):
        manifest_file = os.path.join(cart_dir, CARTRIDGE_MANIFEST_NAME)
        if os.path.isfile(manifest_file):
            manifest = CartridgeManifest(manifest_file)
            cartridge_payload = manifest.payload
            if cartridge_payload is None:
                log.error("Mandatory key payload not found in cartridge descriptor file" )
                raise ValueError("Mandatory key payload not found in cartridge descriptor file")
            payload_path = os.path.join(cart_dir, cartridge_payload)
            if not os.path.exists(payload_path):
                log.error("Payload %s not found at %s" % (cartridge_payload, cart_dir))
                raise ValueError("Payload %s not found at %s" % (cartridge_payload, cart_dir))
        else:
            raise MandatoryMetadataError("Missing descriptor file %s in the archive" % CARTRIDGE_MANIFEST_NAME)
        return None

    def validate_manifest(self, cart_manifest):
        """
        :param cart_manifest: Validate the cartridge manifest.
        """
        self._validate_cart_handlers(cart_manifest)
        self._validate_cart_provides(cart_manifest)
        self._validate_cart_dependency(cart_manifest)

    def _generate_cart_id(self, cart_manifest):
        cart_name = cart_manifest.name
        cart_version = cart_manifest.version
        cart_cpu_arch = cart_manifest.cpuarch
        if cart_name is None or cart_version is None or len(cart_name) == 0:
            raise MandatoryMetadataError("Missing mandatory name and version in metafile (%s, %s) " % (cart_name, cart_version))

        if cart_cpu_arch is None or len(cart_cpu_arch) == 0:
            raise MandatoryMetadataError("Missing mandatory cpuarch in metafile")
    
        sys_cpuarch = Utils.get_cpuarch()
        if cart_cpu_arch != sys_cpuarch :
            log.error("Unsupported cpu arch %s, system cpu arch is: %s" % 
                        (cart_cpu_arch, sys_cpuarch))
            raise Exception("Unsupported cpu arch %s, system cpu arch is: %s" % 
                        (cart_cpu_arch, sys_cpuarch))
        cartridge_id = cart_name + "_" + str(cart_version) + "_" + cart_cpu_arch
        cartridge_id = cartridge_id.replace(' ', '_')

        # Do sanity check of ID
        pattern = re.compile('^[0-9a-zA-Z_\.\-]+$')
        if not pattern.match(cartridge_id):
            log.error("Syntax error: %s" % cartridge_id)
            raise MandatoryMetadataError("Syntax error, valid characters are [0-9a-zA-Z_]")
        elif len(cart_name) > 64:
            log.error("The cartridge name  must be less than 64 characters")
            raise MandatoryMetadataError("The cartridge name  must be less than 64 characters")

        elif len(str(cart_version)) > 8:
            log.error("The cartridge version  must be less than 8 characters")
            raise MandatoryMetadataError("The cartridge name  must be less than 8 characters")
        return cartridge_id

    def _validate_cart_handlers(self, cart_manifest):
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        cart_handleas = cart_manifest.handleas
        if cart_handleas is None or len(cart_handleas) == 0 :
            raise MandatoryMetadataError("Missing mandatory handleas in metafile")
        pc = PlatformCapabilities.getInstance()
        aufs_support = pc.aufs_supported
        overlayfs_support = pc.overlayfs_supported
        sup_handlers = pc.supported_cartridge_handlers
        for handler in cart_handleas:
            if not handler in sup_handlers:
                log.error("Unsupported handler %s, supported handlers: %s" %
                        (handler, str(sup_handlers)))
                raise Exception("Unsupported handler %s, supported handlers: %s" %
                        (handler, str(sup_handlers)))
        #Validate handlers specified
        for cart_handler in cart_handleas:
            if cart_handler == "overlay":
                if not aufs_support and not overlayfs_support:
                    log.error("Cartridge is specified as overlay but there is no support of AUFS")
                    raise CartridgeDescriptorValidationError("Cartridge is specified as overlay but there is no support of AUFS")

    def _validate_cart_provides(self, cart_manifest):
        log.debug("Validating cartridge provide directives")
        for cart in cart_manifest.provides:
            log.debug("cartridge provides %s" % cart)
            if not 'id' in cart:
                log.error("Cartridge id is missing for provided cartridge")
                raise MandatoryMetadataError("Cartridge id is missing for provided cartridge")
            if not 'version' in cart:
                log.error("Cartridge version is missing for provided cartridge %s", cart['id'])
                raise MandatoryMetadataError("Cartridge version is missing for provided cartridge", cart['id'])
            if len(cart['id'].split(CARTRIDGE_LANG_PREFIX)) == 2:
                runtime = cart['id'].split(CARTRIDGE_LANG_PREFIX)[1]
                runtime_version = cart['version']
                lrt = self.languageruntimes.get(runtime, runtime_version)
                if lrt is None:
                    log.error("No runtime plugin found for the specified runtime:%s, runtime_version: %s"%(runtime,runtime_version))
                    raise ValueError("No runtime plugin found for the specified runtime:%s, runtime_version: %s"%(runtime,runtime_version))
            if cart['id'] in self.cartridgeMap:
                #cartridgeMap is the dictionary of cartridge list with key as id
                for cart_det in self.cartridgeMap[cart['id']]:
                    if cart_det['version'] == cart['version'] :
                        log.error("Cartridge id %s with same version %s already exists in cartridge bundle: %s" % (cart['id'], cart['version'], cart_det['cartridge_id']))
                        raise Exception("Cartridge id %s with same version %s already exists in cartridge bundle: %s" % (cart['id'], cart['version'], cart_det['cartridge_id']))

    def _validate_cart_dependency(self, cart_manifest):
        log.debug("Validating cartridge dependencies")
        if cart_manifest.dependson:
            log.debug("Required dependent cartridges %s" % cart_manifest.dependson)
            for depcart in cart_manifest.dependson:
                cartList = self.cartridgeMap.get(depcart['id'])
                if cartList is None:
                    log.error("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))
                    raise CartridgeDependError("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))
                depmet=False
                for cartInfo in cartList:
                    if cartInfo['version'].startswith(depcart['version']):
                        depmet=True
                        break
                if depmet == False:    
                    log.error("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))
                    raise CartridgeDependError("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))


    def populate_all_dependents(self, cart_dep_list, base_rootfs_list=set(), 
                                                cartridge_lrt_list=set()) :
        """
        This is the recursive function which appends the found rootfs in 
        base_rootfs_list and language runtime in cartridge_lrt_list
        """
        if cart_dep_list is None:
            return
        else:
            for depcart in cart_dep_list:
                cartList = self.cartridgeMap.get(depcart['id'])
                if cartList is None:
                    continue
                for cartInfo in cartList:
                    if cartInfo['version'].startswith(str(depcart['version'])):
                        if cartInfo['type'] == 'baserootfs':
                            #base_rootfs_list.add({"id": depcart['id'], "version":cartInfo['version'], "cartridge_id": cartInfo["cartridge_id"]})
                            base_rootfs_list.add(self.get(cartInfo["cartridge_id"]))
                            continue
                        else:
                            #cartridge_lrt_list.add({"id": depcart['id'], "version":cartInfo['version'], "cartridge_id": cartInfo["cartridge_id"]})
                            cartridge_lrt_list.add(self.get(cartInfo["cartridge_id"]))
                            cart = self.get(cartInfo['cartridge_id'])
                            #Call recursively to find all the dependents
                            self.populate_all_dependents(cart.dependson, base_rootfs_list, cartridge_lrt_list)    


    def validate_cartridge_list(self, cart_dep_list):
        """
        Validates the input cart_dep_list as specified in descriptor file under depends-on cartridges section.
        Raises CartridgeDependError if dpendency is not satisfied
        """
        if cart_dep_list:
            for depcart in cart_dep_list:
                cartList = self.cartridgeMap.get(depcart['id'])
                if cartList is None:
                    log.error("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))
                    raise CartridgeDependError("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))
                depmet=False
                for cartInfo in cartList:
                    if cartInfo['version'].startswith(str(depcart['version'])):
                        depmet=True
                        break
                if depmet == False:
                    log.error("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))
                    raise CartridgeDependError("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))
        
 
    def get(self, cartridge_id):
        return self._cartridges.get(cartridge_id, None)
   
    def update_cartridge_dependency(self, cart_id, cart_version, appid, res_type, cartridge_list=None):
        """
        Updates used_by after resolving the dependency. 
        if cartridge_list is given dependency needs to be resolved withing 
        the cartrdige_list
        """

        log.debug("Updating cartridge dependency for app %s type %s (%s:%s) %s" %
                  (appid, res_type, cart_id, cart_version, cartridge_list))
        found=False
        if cart_id in self.cartridgeMap:
            for cartInfo in self.cartridgeMap[cart_id]:
                if str(cartInfo['version']).startswith(str(cart_version)):
                    if cartridge_list :
                        for c in cartridge_list:
                            if c.id == cartInfo['cartridge_id']:
                                res_det = { "id" : appid, "type" : res_type}
                                if not res_det in cartInfo['used_by']:
                                    cartInfo['used_by'].append(res_det)
                                found=True
                                break
                        if found:
                            break    
                    else:
                        res_det = { "id" : appid, "type" : res_type}
                        if not res_det in cartInfo['used_by']:
                            cartInfo['used_by'].append(res_det)
                        found=True
                        break
            if not found:
                log.error("Cartridge id: %s not found, dependency not updated" , cart_id)
        else:
            log.error("Cartridge id: %s not found, dependency not updated" , cart_id)
                 
            
    def find_cartridge(self, cart_id, cart_version):
        log.debug("Requested id: %s, version : %s", cart_id, cart_version)
        if cart_id is None or cart_version is None:
            return None
        if cart_id in self.cartridgeMap:
            #cartridgeMap is the dictionary of cartridge list with key as id
            for cart_det in self.cartridgeMap[cart_id]:
                log.debug("Id: %s, Version: %s", cart_id, cart_det['version'])
                if cart_det['version'].startswith(cart_version) :
                    return self.get(cart_det['cartridge_id']) 
        return None


    def get_baserootfs_cartridge(self):
        for c in self._cartridges.values():
            if c.type == "baserootfs" :
                return c
        return None

    def _create(self, cartridge_id, cartridge_dir):
        cartridge_mount_path = os.path.join(self._cartridge_mount_path, cartridge_id)
        if not os.path.isdir(cartridge_mount_path):
            os.makedirs(cartridge_mount_path)
            log.debug("Created cartridge mount directory: %s", cartridge_mount_path)
        c = Cartridge(cartridge_id, cartridge_dir, cartridge_mount_path)
        self._cartridges[cartridge_id] = c
        log.debug("Created cartridge entry. %s ", c)
        #update cartridgeMap with the new entries
        for cart_info in c.provides:
            log.debug("Cartridge provides: %s" % cart_info)
            if cart_info['id'] in self.cartridgeMap:
                self.cartridgeMap[cart_info['id']].append( 
                  {"version" : cart_info["version"], "type" : c.type,
                   "cartridge_id" : cartridge_id, "used_by" : [], "size" : c.cart_size})

            else:
                self.cartridgeMap[cart_info['id']] = [{"version":cart_info["version"],
                 "type": c.type, "cartridge_id" : cartridge_id, "used_by" : []}]

        #Update used_by dependency in cartridgeMap 
        insertIdx = -1
        if c.dependson is not None:
            for depcart in c.dependson:
                cartList = self.cartridgeMap[depcart['id']]
                if cartList is None:
                    log.error("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))
                    raise Exception("Dependent cartridge not found %s:%s" % (depcart['id'], depcart['version']))

                log.info("List of versions for %s:  %s" % (depcart['id'], cartList))
                for cartInfo in cartList:
                    log.info("Comapring %s : %s " % (cartInfo['version'], depcart['version']))
                    if cartInfo['version'].startswith(depcart['version']):
                        log.debug("updating used_by for version %s" , cartInfo['version'])
                        res_det = {"id": cartridge_id, "type" : "cartridge"} 
                        if not res_det in cartInfo['used_by']:
                            cartInfo['used_by'].append(res_det)
                        #Update the start order for this cartridge
                        depId = cartInfo['cartridge_id']
                        depIdIdx =  self.loadOrder.index(depId)
                        if depIdIdx > insertIdx :
                            insertIdx = depIdIdx
                        break

        log.debug("Cartridge Map:  %s" % self.cartridgeMap)
        self.loadOrder.insert(insertIdx+1, cartridge_id)
        log.debug("Start Order: %s" % self.loadOrder)
        self.setStartOrder(self.loadOrder)
                 

    def exists(self, cartridge_id):
        if cartridge_id in self._cartridges:
            return True
        return False

    def list(self):
        rval = []
        for c in self._cartridges.values():
            d = c.serialize()
            rval.append(d)
        log.debug("Listing all cartridges : %s" % str(rval))
        return rval


    def _load(self):
        from appfw.runtime.db import RepoFolder, StartConfig 
        from appfw.runtime.hostingmgmt import HostingManager
        hosting_manager = HostingManager.get_instance()
        ns = hosting_manager.get_service("notification-service")
        log.debug("notification service: %s" %str(ns)) 

        repoFolders = []
        loadini_path = os.path.join(self._cartridge_root, "load.ini")
        if not os.path.exists(loadini_path):
           #Create the list of Cartridge Folders as load.ini does not exists
            for d in os.listdir(self._cartridge_root):
                cm = os.path.join(self._cartridge_root, d, CARTRIDGE_MANIFEST_NAME)
                if not os.path.isfile(cm):
                    # This is an invalid cartridge directory. Continue
                    log.error("Invalid cartridge manifest file %s, Ignoring %s", cm, d)
                    continue
                #Add Cartridge foler
                repoFolders.append(RepoFolder(d, os.path.join(self._cartridge_root, d)))
        try:
            self._startConfig = StartConfig(loadini_path, repoFolders)
        except Exception as ex:
            log.error("Not able to load load.ini for cartridges:%s" % str(ex))
            shutil.move(loadini_path, loadini_path+".bak")
            cartLoadList = []
            cartDepList = []
            # Try creating load.ini
            for d in os.listdir(self._cartridge_root):
                cm = os.path.join(self._cartridge_root, d, CARTRIDGE_MANIFEST_NAME)
                if not os.path.isfile(cm):
                    # This is an invalid cartridge directory. Continue
                    log.error("Invalid cartridge manifest file %s, Ignoring %s" % (cm, d))
                    continue
                try:
                    cart_manifest = CartridgeManifest(cm)
                    cartridge_id = self._generate_cart_id(cart_manifest)
                except Exception as ex:
                    log.error("Unable to load cartridge manifest file %s, Ignoring %s Error:%s" %(cm, d, str(ex)))
                    self._corrupted_cartids.append(cartridge_id)
                    continue
                if cart_manifest.dependson:
                    for depcart in cart_manifest.dependson:
                        if not depcart['id'] in cartLoadList:
                            cartDepList.append(depcart['id'])

                for cart_info in cart_manifest.provides:
                    if cart_info['id'] in cartDepList:
                        cartLoadList.insert(0, cartridge_id)
                        break
                if not cartridge_id in cartLoadList:        
                    cartLoadList.append(cartridge_id)

            log.debug("Cartridge Load list: %s" % cartLoadList)
            for cart_id in cartLoadList:
                #Add Cartridge foler
                repoFolders.append(RepoFolder(cart_id, os.path.join(self._cartridge_root, cart_id)))
            try:
                self._startConfig = StartConfig(loadini_path, repoFolders)
            except Exception as ex:
                log.error("Failed to load load.ini of cartridges: %s" % str(ex))
                return
           
        # Walk through the cartridge root and update self
        log.debug("Looking for presintalled cartrdiges at %s", self._cartridge_root)
        if self._startConfig.startOrder is None or self._startConfig.startOrder == "":
            return
        for cart in self._startConfig.startOrder.split(","):
            itemPath = os.path.join(self._cartridge_root, cart)
            if os.path.isdir(itemPath):
                # Create a cartridge entry
                try:
                    #Validating the cartridge manifest
                    cm = os.path.join(itemPath, CARTRIDGE_MANIFEST_NAME)
                    if not os.path.isfile(cm):
                        # This is an invalid cartridge directory. Continue
                        log.error("Invalid cartridge manifest file %s, Ignoring %s" % (cm, itemPath))
                        continue
                    try:
                        cart_manifest = CartridgeManifest(cm)
                    except Exception as ex:
                        log.error("Unable to load cartridge manifest file %s, Ignoring %s Error:%s" %(cm, itemPath, str(ex)))
                        self._corrupted_cartids.append(cart)
                        continue

                    self.validate_manifest(cart_manifest)
                    self._create(cart, itemPath)
                except Exception as ex:
                    log.error("Failed to upload cartridge:%s Error:%s" % (itemPath, str(ex)))
                    self._corrupted_cartids.append(cart)


    def start(self):
        """
        Start the cartridge manager
        Since cartridge are already loaded just post events for failures
        """

        from appfw.runtime.db import RepoFolder, StartConfig 
        from appfw.runtime.hostingmgmt import HostingManager
        hosting_manager = HostingManager.get_instance()
        ns = hosting_manager.get_service("notification-service")

        for cart in self._corrupted_cartids:
            if ns:
                event_message = "Cartridge %s failed to load" % cart
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CARTRIDGE_CORRUPTED,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        

    def auto_install(self, location):
        """
        Automatically install the cartridges which are not installed form the given location.
        """
        log.debug("Auto installing cartridges from %s" % location)
        try:
            if os.path.isdir(location):
                autoInstall_file_path = os.path.join(location, self.autoinstall_file_name)
                cartId_archive_map = self._load_autoinstall(autoInstall_file_path)
                for cart_id in list(set(cartId_archive_map.keys()) - set(self._cartridges.keys())):
                    cartId_archive_map.__delitem__(cart_id)
                with open(autoInstall_file_path, "w") as f:
                    for cart_id, archive in cartId_archive_map.iteritems():
                        self._addentry_autoinstall_data(cart_id, archive, f)
                    cart_archive_list = self._load_cart_orderfile(os.path.join(location,self.autoinstall_order_file_name))
                    if not cart_archive_list:
                        cart_archive_list = os.listdir(location)
                    for cart_archive in cart_archive_list:
                        if tarfile.is_tarfile(os.path.join(location, cart_archive)):
                            if (cart_archive not in cartId_archive_map.values()):
                                cartridge_id = None
                                try:
                                    cartridge_id = self.add(os.path.join(location, cart_archive))
                                except Exception as ex:
                                    log.error("Cartridge %s installation failed due to : %s" % (cart_archive, ex.message))
                                if cartridge_id:
                                    self._addentry_autoinstall_data(cartridge_id, cart_archive, f)
                        else:
                            log.error("The file %s is not a tar ball to proceed with installation"%os.path.join(location, cart_archive))
            else:
                log.info("The given path %s for auto installation is not a directory" % location)
        except Exception as ex:
            log.error("Error during auto installation of cartridges: %s".format(ex.message))


    def _addentry_autoinstall_data(self, cartridge_id, archivename, autoInstall_file):
        """
        For the suddessfully installed cartridge add the entry to auto install file.
        """
        entry = cartridge_id + ":" + archivename + "\n"
        if isinstance(autoInstall_file, file):
            autoInstall_file.write(entry)
        else:
            log.error("Expected file object but found: ", type(autoInstall_file))

    def touchopen(self, filename, *args, **kwargs):
        open(filename, "a").close()
        return open(filename, *args, **kwargs)

    def _load_autoinstall(self, autoInstall_file_path):
        """
        :param autoInstall_file_path: File which contains the all auto install data of cartridges
        :return: Map of already install cartridges and corresponding archive names
        """
        log.debug("Loading the auto install data of cartridges")
        cartId_archive_map = {}
        with self.touchopen(autoInstall_file_path, "r+") as f:
            for line in f.readlines():
                line = line.strip()
                li = line.split(":")
                if len(li) == 2:
                    cartId_archive_map[li[0]] = li[1]
                else:
                    log.error("Ignoring blank or invalid entry ( %s ) ,in auto install file %s" % (line, autoInstall_file_path))
        return cartId_archive_map

    def _load_cart_orderfile(self, cart_order_filepath):
        """
        Order file will have list of archive names separated by new line.
        This method will load all this content as a list and return the same.
        """
        log.debug("Loading the auto install data of cartridges")
        cart_order_list = []
        with self.touchopen(cart_order_filepath, "r+") as f:
            for line in f.readlines():
                line = line.strip()
                cart_order_list.append(line)
        return cart_order_list

    def stop_cartridge_manager(self):
        """
        Calls during CAF shutdown
        Unmounts Cartridges
        """
        for cart in reversed(self._startConfig.startOrder.split(",")):
            if cart:
                try:
                    log.debug("Unmounting cartridge: %s" % cart)
                    c = self.get(cart)
                    c.remove()
                except Exception as e:
                    log.error("Error in removing cartridge: %s. Error: %s" % (cart, str(e)))
                    log.debug("Tracback:", exc_info=True)

    def is_paas_app_supported(self):
        """
        This function will return True if CAF supports PAAS type applications.
        Else it will throw AppTypeNotSupported error
        :return:
        """
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        if 'paas' not in pc.supported_app_types:
            raise AppTypeNotSupportedError("App type PAAS is not supported on this platform!")
        return True

