__author__ = 'srinathc'

import logging
import falcon
import base64

from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from common import ResourceBase, AuthenticatedResource, OauthResourceValidator, make_error_response
from ..oauth.model import *
from ..oauth.oauth import OAuthService
from ..api.token import TokenManager
from ..oauth.utils import extract_form_urlencoded
from oauthlib.common import to_unicode

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()
oauthService = APIService.instance.hosting_manager.get_service("oauth-service")

def service_unavailable(caller, req, resp):
    make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)

def token_endpoint(f):
    if oauthService is not None and oauthService.is_running:
        return oauthService.provider.token_endpoint(f)
    else:
        return service_unavailable

def revoke_endpoint(f):
    if oauthService is not None and oauthService.is_running:
        return oauthService.provider.revoke_endpoint(f)
    else:
        return service_unavailable

def authorize_handler(f):
    if oauthService is not None and oauthService.is_running:
        return oauthService.provider.authorize_handler(f)
    else:
        return service_unavailable

def protect(f):
    if oauthService is not None and oauthService.is_running:
        return oauthService.provider.protect(f)
    else:
        return service_unavailable

@ResourceRoute(OAuthService.OAUTH_SERVICE_API_MAP["get_client"], endpoint="oauthclient")
class ClientResource(AuthenticatedResource):

    def on_get(self, req, resp, client_id):
        '''
           Retrieves oauth client details
           Intent: get oauth client details
           Expected input:
              client_id: string, mandatory, the ID of the client
           Success: oauth client details
           Failure: if client_id not found
        '''
        log.debug("ClientResource.on_get")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        client = oauthService.client_getter(client_id=client_id)
        if not client:
            make_error_response(resp, "client_id not found", "client_id not found", falcon.HTTP_404)
            return

        resp.body = jsonencoder.encode(client)
        resp.status = falcon.HTTP_200

@ResourceRoute(OAuthService.OAUTH_SERVICE_API_MAP["clients"], endpoint="oauthclients")
class ClientsResource(AuthenticatedResource):

    def on_get(self, req, resp):
        '''
           Lists all oauth clients (or only clients of specified app)
           Intent: Lists all oauth clients
           Expected input: optional app_id query parameter
           Success: oauth client details
           Failure: could be due to internal reasons

           Lists all oauth clients (or only clients of specified app)
        '''
        log.debug("ClientsResource.on_get")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        app_id = req.get_param('app_id')

        clients = []
        if app_id:
            client = oauthService.get_client_by_appid(app_id)
            if client:
                clients.append(client.serialize())
        else:
            for client in oauthService.list_clients():
                clients.append(client.serialize())
        resp.body = jsonencoder.encode(clients)
        resp.status = falcon.HTTP_200

    def on_post(self, req, resp):
        '''
           Create an oauth client
           Intent: create an oauth client
           Expected input:
           Success: oauth client details
           Failure: could be due to internal reasons
          
           Generates an OAuth client for an app
        '''
        log.debug("ClientsResource.on_post")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        form_params = extract_form_urlencoded(req)
        if not form_params.has_key("app_id"):
            make_error_response(resp, "app_id not specified", "app_id not specified", falcon.HTTP_400)
            return
        app_id = form_params["app_id"][0]

        token_id = req.get_header("X-Token-Id")
        if not token_id:
            token_id = req.get_param("tokenid")

        token = TokenManager.getInstance().getToken(token_id)
        log.debug("user: "+token.getUserData())
        scopes = form_params["scope"] if form_params.has_key("scope") else None
        log.debug("Registering client with scopes %s", scopes)
        # client.user = user
        client = oauthService.register_app(app_id, token.getUserData(), access_scopes=scopes)
        resp.body = jsonencoder.encode(client.serialize())
        resp.status = falcon.HTTP_200

    def on_delete(self, req, resp):
        '''
           Deletes all oauth clients (or only clients of specified app)
           Intent: Deletes all oauth clients
           Expected input: optional app_id query parameter
           Success: if deletion succeeded
           Failure: could be due to internal reasons

           Deletes all oauth clients (or only clients of specified app)
        '''
        log.debug("ClientsResource.on_delete")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        clients = []
        app_id = req.get_param('app_id')
        if app_id:
            client = oauthService.get_client_by_appid(app_id)
            if client:
                clients.append(client)
        else:
            clients = oauthService.list_clients()

        for client in clients:
            oauthService.client_deleter(client.client_id)
        resp.status = falcon.HTTP_204

@ResourceRoute(OAuthService.OAUTH_SERVICE_API_MAP["get_token"], endpoint="outhtoken")
class TokenResource(AuthenticatedResource):

    def on_get(self, req, resp, access_token):
        '''
           Intent: get an oauth token
           Expected input: a valid token id
           Success: oauth token  details
           Failure: if token does not exist
        '''
        log.debug("TokenResource.on_get")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        token = oauthService.token_getter(access_token=access_token)
        if not token:
            make_error_response(resp, "access_token not found", "access_token not found", falcon.HTTP_404)
            return
        resp.body = jsonencoder.encode(token.serialize())
        resp.status = falcon.HTTP_200

@ResourceRoute(OAuthService.OAUTH_SERVICE_API_MAP["tokens"], endpoint="outhtokens")
class TokensResource(ResourceBase):
    '''
        OAuth token endpoint
        See https://tools.ietf.org/html/rfc6749#section-3.2
    '''

    def on_get(self, req, resp):
        '''
           Lists all access tokens (or only access tokens of specified app)
           Intent: Lists all access tokens
           Expected input: optional app_id query parameter
           Success: details of access tokens
           Failure: could be due to internal reasons

           Lists all access tokens (or only clients of specified app)
        '''
        log.debug("TokensResource.on_get")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        tokens = []
        app_id = req.get_param('app_id')
        if app_id:
            client = oauthService.get_client_by_appid(app_id)
            if client:
                tokens = oauthService.get_tokens_for_client(client.client_id)
        else:
            tokens = oauthService.list_tokens()
        arr = []
        for token in tokens:
            arr.append(token.serialize())
        resp.body = jsonencoder.encode(arr)
        resp.status = falcon.HTTP_200

    def on_post(self, req, resp):
        '''
           Intent: return or generate an oauth token, OAuth 2.0 compliant call
           Expected input: client credentials in authorization header
           Success: oauth token details
           Failure: invalid client, requested permissions not allowed, etc
        '''
        log.debug("TokensResource.on_post")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        auth = req.get_header("Authorization")
        if not auth:
            make_error_response(resp, "Unauthorized", "Unauthorized", falcon.HTTP_401)
            return

        client_id = None
        try:
            _, s = auth.split(' ')
            client_id, client_secret = base64.b64decode(s).split(':')
        except Exception as e:
            make_error_response(resp, "Failed to extract authorization", "Failed to extract authorization", falcon.HTTP_400)
            return

        tokens = oauthService.get_tokens_for_client(client_id)
        for token in tokens:
            expires_in = token.expires_in()
            if expires_in ==-1 or expires_in > 0:
                resp.body = jsonencoder.encode(token.serialize())
                resp.status = falcon.HTTP_200
                return
            else:
                oauthService.token_deleter(access_token=token.access_token)
        return self._generate_token(req, resp)

    @token_endpoint
    def _generate_token(self, req, resp):
        '''
           Intent: generate an oauth token, OAuth 2.0 compliant call
           Expected input: client credentials in authorization header
           Success: oauth token details
           Failure: invalid client, requested permissions not allowed, etc
        '''
        log.debug("Generating access token")

    def on_delete(self, req, resp):
        '''
           deletes all access tokens (or only access tokens of specified app)
           Intent: Deletes all access tokens
           Expected input: optional app_id query parameter
           Success: if deletion is succeeded
           Failure: could be due to internal reasons

           Deletes all access tokens (or only clients of specified app)
        '''
        log.debug("TokensResource.on_delete")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        tokens = []
        app_id = req.get_param('app_id')
        if app_id:
            client = oauthService.get_client_by_appid(app_id)
            if client:
                tokens = oauthService.get_tokens_for_client(client.client_id)
        else:
            tokens = oauthService.list_tokens()

        for token in tokens:
            oauthService.token_deleter(access_token=token.access_token)
        resp.status = falcon.HTTP_204

@ResourceRoute(OAuthService.OAUTH_SERVICE_API_MAP["protected"], endpoint="oauthprotected")
class ProtectedResource(ResourceBase):

    @protect
    def on_get(self, req, resp):
        '''
           Intent: to demonstrate how an API can be guarded with an OAuth token
           Expected input: OAuth token in the authorization header
           Success: oauth token details
           Failure: invalid client, requested permissions not allowed, etc
        '''
        log.debug("ProtectedResource.on_get")
        resp.status = falcon.HTTP_204

@ResourceRoute(OAuthService.OAUTH_SERVICE_API_MAP["revoke"], endpoint="oauthrevoke")
class RevokeResource(AuthenticatedResource):

    @revoke_endpoint
    def on_post(self, req, resp):
        '''
           Intent: revoke an oauth token
           Expected input: a valid oauth token
           Success: the token is deleted
           Failure: invalid token, etc
        '''
        log.debug("RevokeResource.on_post")

@ResourceRoute(OAuthService.OAUTH_SERVICE_API_MAP["verify"], endpoint="oauthverify")
class VerifyResource(OauthResourceValidator):

    def on_post(self, req, resp):
        '''
           Intent: to verify the authenticity of a token w.r.t. a specified
           scope. This is done on any resource server (in oauth terminology)
           Expected input: OAuth token, scopes requesting verification for
           Success: oauth token details
           Failure: invalid token, invalid client, requested permissions not allowed, etc
        '''
        log.debug("VerifyResource.on_post")
        if oauthService is None or not oauthService.is_running:
            make_error_response(resp, "OAuthService is unavailable", "OAuthService is unavailable", falcon.HTTP_503)
            return

        access_token = req.get_param("access_token")
        if not access_token:
            make_error_response(resp, "access_token token specified",
                                "access_token token specified",
                                falcon.HTTP_400)
            return
        token = oauthService.token_getter(access_token=access_token)
        if not token:
            make_error_response(resp, "access_token not found",
                                "access_token not found",
                                http_status_code=falcon.HTTP_404)
            return
        requested_scopes = req.get_param("scopes")
        log.debug("OAUTH requested scopes %s", requested_scopes)
        if requested_scopes:
            req_scopes_list = requested_scopes.split()
            log.debug("OAUTH scopes after split %s", req_scopes_list)
            if token.is_scopes_allowed(req_scopes_list):
                resp.body = token.to_json()
                resp.status = falcon.HTTP_200
            else:
                make_error_response(resp, "Requested scope is not granted",
                                    "Requested scope is not granted",
                                    http_status_code=falcon.HTTP_401)
        else:
            #In case validator is not sending any scopes , do not match against the token scopes
            resp.body = token.to_json()
            resp.status = falcon.HTTP_200



@ResourceRoute(OAuthService.OAUTH_SERVICE_API_MAP["authorize"], endpoint="oauthauthorize")
class AuthResource(object):
    '''
        Intent: authorize a user with password
        Unused in the current scheme of things, will be useful if we do token
        grant flow

        See https://tools.ietf.org/html/rfc6749#section-3.1
    '''

    @authorize_handler
    def on_get(self, req, resp):
        log.debug("AuthResource.on_get")

    @authorize_handler
    def on_post(self, req, resp):
        log.debug("AuthResource.on_post")

