import logging
import json
import os
import falcon
from apiservice import ResourceRoute
from common import AuthenticatedResource, make_response, make_error_response, send_file
from appfw.utils.commandwrappers import *
from jsonencoder import JSONEncoder
from appfw.dockerplugin.apiservice import PluginAPIService
from appfw.utils.utils import Utils
from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper
from ..runtime.platformcapabilities import PlatformCapabilities
from appfw.utils.docker_utils import DockerUtils

jsonencoder = JSONEncoder(ensure_ascii=False)
log = logging.getLogger("runtime.api.resources")

@ResourceRoute("/docker/containers", endpoint="dockercontainers")
class DockerContainersResource(AuthenticatedResource):
    def on_get(self, request, response):
        try:
            # Return the current config of docker remote server
            pc = PlatformCapabilities.getInstance()
            if not pc._remote_docker_supported:
                msg = "Remote docker server access feature is not supported."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return
            docker_api_client = DockerUtils.get_docker_api_client()
            containers_list = DockerUtils.get_containers(docker_api_client)
            out = jsonencoder.encode(containers_list)

            response.status = falcon.HTTP_200
            response.body = out
            response.set_headers({'Content-Type': "application/json",
                                    'Cache-Control': "no-cache"})
        except Exception as ex:
            msg = "Failed to retrieve docker containers on the device"
            log.exception("%s - %s" % (msg, ex))
            make_error_response(response,
                                "%s" % msg,
                                "%s" % msg,
                                falcon.HTTP_500)

@ResourceRoute("/docker/volumes", endpoint="dockervolumes")
class DockerVolumesResource(AuthenticatedResource):
    def on_get(self, request, response):
        try:
            # Return the current config of docker remote server
            pc = PlatformCapabilities.getInstance()
            if not pc._remote_docker_supported:
                msg = "Remote docker server access feature is not supported."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return
            docker_api_client = DockerUtils.get_docker_api_client()
            volumes_list = DockerUtils.get_volumes(docker_api_client)
            out = jsonencoder.encode(volumes_list)

            response.status = falcon.HTTP_200
            response.body = out
            response.set_headers({'Content-Type': "application/json",
                                    'Cache-Control': "no-cache"})
        except Exception as ex:
            msg = "Failed to retrieve docker volumes on the device"
            log.exception("%s - %s" % (msg, ex))
            make_error_response(response,
                                "%s" % msg,
                                "%s" % msg,
                                falcon.HTTP_500)

@ResourceRoute("/docker/images", endpoint="dockerimages")
class DockerImagesResource(AuthenticatedResource):
    def on_get(self, request, response):
        try:
            # Return the current config of docker remote server
            pc = PlatformCapabilities.getInstance()
            if not pc._remote_docker_supported:
                msg = "Remote docker server access feature is not supported."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return
            docker_api_client = DockerUtils.get_docker_api_client()
            images_list = DockerUtils.get_images(docker_api_client)
            out = jsonencoder.encode(images_list)

            response.status = falcon.HTTP_200
            response.body = out
            response.set_headers({'Content-Type': "application/json",
                                    'Cache-Control': "no-cache"})
        except Exception as ex:
            msg = "Failed to retrieve docker images on the device"
            log.exception("%s - %s" % (msg, ex))
            make_error_response(response,
                                "%s" % msg,
                                "%s" % msg,
                                falcon.HTTP_500)

@ResourceRoute("/docker/remoteserver", endpoint="remoteserver")
class DockerRemoteServerResource(AuthenticatedResource):
    def on_get(self, request, response):
        try:
            # Return the current config of docker remote server
            pc = PlatformCapabilities.getInstance()
            if not pc._remote_docker_supported:
                msg = "Remote docker server access feature is not supported."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return
            remotedocker_util = DockerServerHelper.getInstance()
            out = {}
            config = remotedocker_util.get_dockerserver_runtime_config()
            if config["enabled"]:
                out["msg"] = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'client_usage_help_string')
            else:
                out["msg"] = "Docker remote server is disabled"
            out["enabled"] = config["enabled"]
            out["docker_client_tlsverify"] = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'tlsverify', parse_as="bool")
            out = jsonencoder.encode(out)

            response.status = falcon.HTTP_200
            response.body = out
            response.set_headers({'Content-Type': "application/json",
                                    'Cache-Control': "no-cache"})
        except Exception as ex:
            msg = "Failed to retrieve docker remote server configuration"
            log.exception("%s - %s" % (msg, ex))
            make_error_response(response,
                                "%s" % msg,
                                "%s" % msg,
                                falcon.HTTP_500)


    def on_put(self, request, response):
        try:
            '''
            Enable or Disable remote docker server configuration
            Sample input json payload:
            {
                "enabled": true/false
            }
            '''

            pc = PlatformCapabilities.getInstance()
            if not pc._remote_docker_supported:
                msg = "Remote docker server access feature is not supported."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return

            # Check if any application is in activated/running state
            from appfw.runtime.hostingmgmt import HostingManager
            hosting_manager = HostingManager.get_instance()
            applist = hosting_manager.get_service("app-management").list()
            apps_to_deactivate = []
            for app in applist:
                log.debug("app %s state - %s" % (app._connector.id, app._state))
                if app._state in ["ACTIVATED", "RUNNING", "STOPPED", "FAILED"]:
                    apps_to_deactivate.append(app._connector.id)

            if apps_to_deactivate:
                msg = "Deactivate Apps - %s to manage remote docker access" % apps_to_deactivate
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_503)
                return
            payload = json.load(request.stream)
            pluginservice = PluginAPIService.getInstance()
            pluginservice.update_docker_remoteserver(payload)
            remotedocker_util = DockerServerHelper.getInstance()
            remotedocker_util.set_dockerserver_runtime_config(payload)
            # Post an appropriate status change event
            from appfw.utils.cafevent import CAFEvent
            event_type = None
            ns = hosting_manager.get_service("notification-service")
            if payload["enabled"]:
                docker_client_helpstr = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'client_usage_help_string')
                event_type = CAFEvent.TYPE_REMOTE_DOCKER_ENABLED
                event_message = "Remote docker enabled using: %s" %(CAFEvent.SOURCE_RESTAPI)
                response = make_response(response, docker_client_helpstr, falcon.HTTP_200)
            else:
                event_type = CAFEvent.TYPE_REMOTE_DOCKER_DISABLED
                event_message = "Remote docker disabled using: %s" %(CAFEvent.SOURCE_RESTAPI)
                response = make_response(response, 'Successfully disabled remote docker server config', falcon.HTTP_200)

            if ns and event_type:
                log.debug("Posting '%s' event" % (event_type))
                ns.post_event(CAFEvent(event_type=event_type,
                                       event_source=CAFEvent.SOURCE_RESTAPI, event_message=event_message))
        except Exception as ex:
            msg = "Failed to update remote docker server configuration"
            log.exception("%s - %s" % (msg, ex))
            make_error_response(response,
                                "%s - %s" % (msg, ex),
                                "%s - %s" % (msg, ex),
                                falcon.HTTP_500)

@ResourceRoute("/docker/tlscerts", endpoint="clientcreds")
class DockerClientCredsResource(AuthenticatedResource):
    def on_get(self, request, response):
        try:
            # return client tls certs as an attachment and autodownload it
            pc = PlatformCapabilities.getInstance()
            if not pc._remote_docker_supported:
                msg = "Remote docker server access feature is not supported."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return

            docker_tls_supported = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'tlsverify', parse_as="bool")
            if not docker_tls_supported:
                msg = "Docker TLS verification is not supported on this device. There are no TLS Client certs generated."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_503)
                return

            docker_tlsdir = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'tlsdir')
            docker_client_tlsfilename = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'client_tls_tarfile')
            docker_client_tls_fullpath = docker_tlsdir+"/"+docker_client_tlsfilename
            if os.path.exists(docker_client_tls_fullpath):
                send_file(response,
                        docker_client_tls_fullpath,
                        send_as_attachment=True,
                        attachment_filename=docker_client_tlsfilename)
            else:
                msg = "Docker client TLS certs file not found"
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_404)
        except Exception as ex:
            msg = "Failed to download docker client TLS certs file"
            log.exception("%s - %s" % (msg, ex))
            make_error_response(response,
                                "%s" % msg,
                                "%s" % msg,
                                falcon.HTTP_500)


    def on_post(self, request, response):
        try:
            '''
            Generate Docker TLS certs/keys. If tls cert/keys already exist, don't
            generate unless force flag is set.
            {
                "force": true/false
            }
            '''
            pc = PlatformCapabilities.getInstance()
            if not pc._remote_docker_supported:
                msg = "Remote docker server access feature is not supported."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return

            docker_tls_supported = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'tlsverify', parse_as="bool")
            if not docker_tls_supported:
                msg = "Docker TLS verification is not supported on this device. There are no TLS Client certs generated."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_503)
                return

            payload = json.load(request.stream)
            remotedocker_util = DockerServerHelper.getInstance()
            remotedocker_util.setup_docker_tlscerts(payload)
            response = make_response(response, 'Successfully setup docker tls certs', falcon.HTTP_200)
        except Exception as ex:
            msg = "Failed to generate TLS certs for docker client"
            log.exception("%s - %s" % (msg, ex))
            make_error_response(response,
                                "%s" % msg,
                                "%s" % msg,
                                falcon.HTTP_500)


    def on_delete(self, request, response):
        # Delete docker TLS certs/keys
        try:
            pc = PlatformCapabilities.getInstance()
            if not pc._remote_docker_supported:
                msg = "Remote docker server access feature is not supported."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return

            docker_tls_supported = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'tlsverify', parse_as="bool")
            if not docker_tls_supported:
                msg = "Docker TLS verification is not supported on this device. There are no TLS Client certs generated."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_503)
                return

            remotedocker_util = DockerServerHelper.getInstance()
            remotedocker_util.cleanup_docker_tlscerts()
            response = make_response(response, 'Successfully cleaned up docker tls certs', falcon.HTTP_200)
        except Exception as ex:
            msg = "Failed to clean up docker client TLS certs"
            log.exception("%s - %s" % (msg, ex))
            make_error_response(response,
                                "%s" % msg,
                                "%s" % msg,
                                falcon.HTTP_500)
