#!/bin/bash

# Setup VENV
#
#

function HELP {
    echo 'specifying the options to run the test suite'
    echo 'Usage: ./run-tests.sh --lxc/--process/--docker/--all'
    echo '--lxc: lxc container'
    echo '--docker: docker container'
    echo '--process: process container'
    echo '--all: All container types'
    echo '--ip: ip address of the machine where you want to run the test suite[127.0.0.1]'
    echo '--port: port on which CAF is running[8443]'
    echo '--user: username of the machine[jenkins]'
    echo '--pass: password of the machine[jenkins]'
    echo '--testdir: Only tests present in this directory will be executed'
    echo '--testfile: Only tests present in this file will be executed'
    echo '--testcase: Only this testcase will be executed. Must pass --testdir argument for location of the test file'
    echo '--artifacts: Path where artifacts are located. Must specify location in https or Unix file or ftp URI scheme. Example: http://1.2.3.4/ir829 or file:///home/userid/ir829 or file://localhost/ir829 or ftp://1.2.3.4/ir829/'
    echo '--debug: to run the test suite in debugging mode'
    echo '--exitfirst: exit the test suite at first failure'
    echo 'Note: Options "--lxc/--process/--docker/--all" are only applicable if your running the test suite counter to local host'
    exit 1
}
pointer=1
while [[ $pointer -le $# ]]; do
   param=${!pointer}
   if [[ $param != "-"* ]]; then ((pointer++)) # not a parameter flag so advance pointer
   else
      case $param in
         --ip) ((pointer++))
                ip=${!pointer}
                if [ "$ip" ]; then
                  export TEST_CAF_IP=$ip
                fi ;;
         --port) ((pointer++))
                port=${!pointer}
                if [ "$port" ]; then
                  export TEST_CAF_PORT=$port
                fi ;;
         --user) ((pointer++))
                user=${!pointer}
                if [ "$user" ]; then
                  export TEST_CAF_USER=$user
                fi ;;
         --pass) ((pointer++))
                password=${!pointer}
                if [ "$password" ]; then
                  export TEST_CAF_PASS=$password
                fi ;;
         --artifacts) ((pointer++))
                artifacts=${!pointer}
                if [ "$artifacts" ]; then
                  export TEST_CAF_ARTIFACTS=$artifacts
                fi ;;
         --testdir) ((pointer++))
                testdir=${!pointer} ;;
         --testfile) ((pointer++))
                testfile=${!pointer} ;;
         --testcase) ((pointer++))
                testcase=${!pointer} ;;
         -d|--debug) debug=-s ;;
         -x|--exitfirst) exitfirst=-x ;;
         -h|--help) HELP ;;
         --lxc|--docker|--process|--all) pytest_command=$param ;;
         *) ((pointer++))
            echo "Invalid option: $param"
            exit 1 ;;
      esac
    ((pointer++))
   fi
done
if [ ! $TEST_CAF_IP ];then
    export TEST_CAF_IP='127.0.0.1'
fi
if [ ! $TEST_CAF_PORT ];then
    export TEST_CAF_PORT='8443'
fi
if [ ! $TEST_CAF_USER ];then
    export TEST_CAF_USER='jenkins'
fi
if [ ! $TEST_CAF_PASS ];then
    export TEST_CAF_PASS='jenkins'
fi
#check the number of arguments
NUMARGS=$#
if [ $NUMARGS -eq 0 ]; then
    echo 'No args specified, tests will be run for default container type'
else
    if [ "$1" == "--help" ]; then
        HELP
    fi
fi

if [[ ! -z $testdir ]]; then
    TESTDIR=$(readlink -f $testdir)
fi

if [[ ! -z $testfile ]]; then
    TESTFILE=$(readlink -f $testfile)
fi

echo -e \\n"Number of arguments: $NUMARGS"
BASEDIR=$(dirname "$0")
$BASEDIR/setup-venv.sh

# Activate
VENV_DIR=~/.virtualenvs/CAF
ACTIVATE_ENV=$VENV_DIR/bin/activate
if [ -d $VENV_DIR ]
then
    echo 'Activating virtualenv at ' $VENV_DIR
    . $ACTIVATE_ENV
fi

# To get the Cobertura coverage report linked to the source code we run py.test from the iox-dev project root directory
cd $BASEDIR/../../../

#TEST OUTPUT DIR
TEST_OUTPUT_DIR=test_output/caf
if [ ! -d $TEST_OUTPUT_DIR ]
then
    echo 'Creating test output dir : ' $TEST_OUTPUT_DIR
    mkdir -p $TEST_OUTPUT_DIR
fi

echo "Running CAF ut, pep8 and stylecheck"
echo "Current directory: " $PWD
pep8 --max-line-length=120 core/caf/src/appfw/ > pep8.log || true
pyflakes core/caf/src/appfw/ > pyflakes.log || true

if [[ -z $testdir ]] && [[ ! -z $testcase ]]; then
    echo "ERROR: Please specify both the test directory and test case name"
    exit 1
fi


if [[ ! -z $testdir ]] && [[ ! -z $testcase ]]; then
    cd $TESTDIR
    echo "Running the test case: " $testcase
    py.test -v -k $testcase $exitfirst $debug $pytest_command
else
    if [[ ! -z $testdir ]] || [[ ! -z $testfile ]]; then
        echo "Running all the tests in " $TESTDIR $TESTFILE
        py.test -v --cov-config coveragerc --cov core/caf/src/appfw --cov-report xml $TESTFILE $exitfirst $debug $pytest_command
    else
        echo "Running all the tests in core/caf/src/appfw core/caf/tests"
        py.test -v --junitxml pytest.xml --cov-config coveragerc --cov core/caf/src/appfw --cov-report xml core/caf/tests $exitfirst $debug $pytest_command
        mv *.log $TEST_OUTPUT_DIR
        mv *.xml $TEST_OUTPUT_DIR
    fi
fi
