#-----------------------------------------------------
#
# Copyright (c) 2012 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

'''
Created on Sep 27, 2012

@author: rnethi
'''
import os
import shutil
import logging
import ConfigParser
from   stagingplugin import StagingPlugin
from   ..utils.utils import Utils

log = logging.getLogger("runtime.hosting")

"""
Flow Diagram of python plugin:


if (toolkit_required):
    copy the toolkit
    if (launcher_required):
        copy start_connector.py

if (custom start script is provided):
    CMD = 'exec <start_script>'
else:
    if (launcher_required AND toolkit_required):
        CMD = 'exec python start_connector.py'
    else:
        CMD = 'exec python <runner>'
start_command = CMD

"""

class PythonStagingPlugin(StagingPlugin):
    '''
    Base class for standalone python runtime specific processing
    '''
    def __init__(self, baseDir, connectorArchiveFile, stagingReq, config=None):
        super(self.__class__, self).__init__(baseDir, connectorArchiveFile, stagingReq, config)
        self.metadata = self.stagingReq.getMetadata()
        self.useLauncher = self.metadata.use_launcher
        self.toolkit_required = self.metadata.toolkit
        self.env = self._getPythonEnv()
        
    def startStaging(self):
        '''
        Do work and return stagingResponse 
        '''
        self.createConnectorDirectories()
        self.extractArchiveFileToDestination()
        self.createPythonModulesDirectory()
        if self.toolkit_required:
            self.copyToolkitFiles()
        
        self.createStartScript()
        self.createStopScript()
        self.createCustomScripts()
        self.setupCustomLogFiles()
        
    def getStartScript(self):
        '''
        Return the startup script.
        Plugin implementations can use the helper methods generateStartScript() and 
        further customize 
        '''

        #If connector nofile limit has been specified, put that as part of start script
        cfg = self._getSystemConfig()
        customCommands = []
        if cfg.has_option("connector", "connectorOpenFilesLimit"):
            nofileLimit = cfg.getint("connector", "connectorOpenFilesLimit")
            customCommands.append("ulimit -n %d" % nofileLimit) 

        return self.generateStartScript(self.env, customCommands)

    def getPluginEnvironment(self):
        return self.env

    def _getPythonEnv(self):
        """
        Returns environment for connector's app directory and modules directory to PYTHONPATH
        """
        inheritPythonPath = False
        cfg = self._getSystemConfig()
        if cfg.has_option("python-connector", "inherit_runtime_python_path"):
            inheritPythonPath = cfg.getboolean("python-connector", "inherit_runtime_python_path")

        env = {}

        pypath = "$CAF_HOME/"+self.getAppDirectoryName()+ ":$CAF_HOME/"+self.getAppDirectoryName()+"/"+self.getModulesDirectoryName()+":$CAF_HOME/"+self.getModulesDirectoryName()
        if inheritPythonPath:
            pypath = "$PYTHONPATH:"+pypath
        env["PYTHONPATH"] = pypath
        env['PYTHONUNBUFFERED'] = "true"

        #add connector specified environment variables
        conenv = self.stagingReq.getMetadata().app_env
        if conenv:
            for option, value in conenv.items():
                env[option] = value

        return env

    def getStopScript(self):
        '''
        Return the stop script
        Plugin implementations can use the helper methods generateStopScript() and
        further customize 
        '''
        return self.generateStopScript(env=self.env)
    
    def getCommandForStart(self):
        """
        Returns the system command to start the connector
        Language plugin implementations must override this
        """
        
        #We need to generate a start command for launching python script connector
        #for now, assume that python binary is on PATH.
        #TODO: Fix contoller to pass the language runtime path info to staging

        if self.customStartScript:
            return "$CAF_APP_PATH/"+self.customStartScript

        runnerFile = self.stagingReq.getMetadata().app_binary
        
        cmd = []
        cmd.append("python")
        if self.useLauncher and self.toolkit_required:
            cmd.append("$CAF_HOME/start_connector.py")
        else:
            cmd.append("$CAF_APP_PATH/"+runnerFile)

        if ".sh" in runnerFile:
            cmd.remove("python")

        return " ".join(cmd)


    def getModulesDirectoryName(self):
        return "modules"
    
    def getModulesDirectory(self):
        modulesDir = os.path.join(self.getConnectorDirectory(), self.getModulesDirectoryName())
        return modulesDir

    def getRelativeModulesDirectory(self):
        modulesDir = os.path.join(self.getRelativeConnectorDirectory(), self.getModulesDirectoryName())
        return modulesDir
    
    def createPythonModulesDirectory(self):
        if not os.path.exists(self.getModulesDirectory()):
            os.mkdir(self.getModulesDirectory())
    
    def copyToolkitFiles(self):
        """
        Copy toolkit files to modules directory.
        """

        #we have nothing to do if we are running connector inprocess, but copy it anyway
        #it allows us to quickly switch between inprocess and process modes without undeploying connectors
        #if not self.isInProcessContainer():
            #TODO: Make the toolkit as a proper installable python module and copy

        modulesDir = self.getModulesDirectory()

        #copy toolkit service directories        
        toolkitDir = modulesDir
        
        srcToolkitServicesDir = os.path.join(Utils.getRuntimeSourceFolder(), "toolkit-python")
        srcToolkitServicesDir = os.path.join(srcToolkitServicesDir, "services")
        srcToolkitFiles = os.path.join(srcToolkitServicesDir, "*")
        os.system("cp -a "+srcToolkitFiles+" "+toolkitDir)

        #copy sdk directories
        #TODO: clients are not linking ot sdk today. we need to see how to deal with these when
        #client start linking to sdk
        sdkDir = os.path.join(Utils.getRuntimeSourceFolder(), "toolkit-python")
        sdkDir = os.path.join(sdkDir, "client-sdk")
        sdkFiles = os.path.join(sdkDir, "*")
        os.system("cp -a "+sdkFiles+" "+toolkitDir)

        #copy the start script to connector home directory if use-launcher is True in connector_metadata
        if self.useLauncher:
            scriptsFolder = os.path.join(srcToolkitServicesDir, "scripts")
            shutil.copy(os.path.join(scriptsFolder, "start_connector.py"), self.getConnectorDirectory())

        return
    
    def _getSystemConfig(self):
        systemConfigPath = Utils.getSystemConfigPath()
        systemConfig = ConfigParser.SafeConfigParser()
        systemConfig.read(systemConfigPath)
        return systemConfig

