# Copyright (c) 2012-2013 by Cisco Systems, Inc.
# All rights reserved.
#
# This software is the confidential and proprietary information of
# Cisco Systems. It can only be used or disclosed in accordance with
# the term of the license agreement with Cisco Systems.
#
import os
import shutil
import logging
import sys
import ConfigParser
import json
from ..hosting.state import State
from ..utils.utils import Utils, YAML_MANIFEST_NAME, INI_MANIFEST_NAME, APP_DESCRIPTOR_NAME, USER_EXTRACTED_DIR
from error import RepoDatabaseError, MandatoryMetadataError
from descriptormetadata import descriptor_metadata_wrapper
from ..utils.docker_utils import  DOCKER_IMAGE_FILE

log = logging.getLogger("runtime.hosting")


class MissingManifestException(Exception):
    pass


class ConnectorRepository(object):
    '''
    Master database of connectors with file system based persistence.

    Connector platform maintains a repository of deployed connectors in persisted
    form on the platform itself. This is required as the platform could get
    rebooted/restarted for various reasons and it mayn't be practical to go back to
    central manager to query & download deployed connectors.
    
    Controller is the sole owner of this database and is responsible for keeping
    it consistent and up to date.
    
    The database contains the following artifacts for each connector
    
    Deployment related artifacts
      - Connector package - includes connector code, modules & any dependencies. This represents the processed connector package that is finally deployed to a container
      - Connector metadata - Metadata around connector's language runtime(python, java etc), resource requirements, toolkit sdk versions etc
      - Associated container details    
    '''


    def __init__(self, config):
        '''
        Initialize the database
        '''
        self._connectors = {}

        self.rootPath = config.get("controller", "repo")
        self.rootPath = os.path.abspath(self.rootPath)
        
        #Ensure that the root directory exists. 
        if (os.path.exists(self.rootPath)):
            if (not os.path.isdir(self.rootPath)):
                raise RepoDatabaseError("Repository path exists and is not a directory")
        else:
            os.makedirs(self.rootPath)
        log.debug("Repository initialized in the directory:%s", self.rootPath)
	
        # Look for the repo version
        work_folder = os.path.dirname(self.rootPath)
        repo_version = os.path.join(work_folder, "version")
        iox_version = Utils.getIOxVersion()
        if os.path.exists(repo_version) :
            try:
                with open(repo_version, "r") as f:
                    r_version = f.read()
                    r_version = json.loads(r_version)

                caf_supp_vers=["1.0"]
                if "repo" in  iox_version:
                    caf_supp_vers = iox_version["repo"]["supported_versions"]

                r_ver = "1.0"
                if "repo" in r_version:
                    r_ver = r_version["repo"]["repo_version"]
                else:
                    r_version["repo"] = {}
                    r_version["repo"]["repo_version"] = r_ver
                    r_version["repo"]["supported_versions"] = caf_supp_vers
                    #Repo version file does not contains repo so write it
                    o = json.dumps(r_version)
                    with open(repo_version, "w", 0) as f:
                        f.write(o)

                log.info("Caf supported repo versions: %s" % caf_supp_vers)
                log.info("Repo version:%s" % r_ver)

                #if Utils.compare_versions(caf_supp_vers, r_ver) >= 0:
                if r_ver in caf_supp_vers:
                    log.info("Caf version is compatible with repo caf version")
                else:
                    log.error("Repo version: %s is not compatible with CAF supported versions: %s" % (r_ver, caf_supp_vers))
                    raise ValueError("Repo version: %s is not compatible with CAF supported versions: %s" % (r_ver, caf_supp_vers))

            except Exception as ex:
                log.exception("Failed to read version from repo %s:%s" % (repo_version, str(ex)))
                log.info("Moving old repo version to bak and initializing repo version to default")
                shutil.move(repo_version, repo_version+".bak")
                #Repo version file was corrupted so create new one
                r_version = {}
                o = json.dumps(iox_version)
                with open(repo_version, "w", 0) as f:
                    f.write(o)
        else:

            # Version file does not exists create one
            log.info("Repo does not contain the version file creating one...")
            # Store the iox version 
            o = json.dumps(iox_version)
            with open(repo_version, "w", 0) as f:
                f.write(o)

        repo_pid_file = os.path.join(work_folder, "product_id")
        platform_product_id= Utils.getPlatformProductID()
        if not os.path.exists(repo_pid_file) :
            log.info("Storing the product id in %s" % repo_pid_file)
            #create the product_id file 
            with open(repo_pid_file, "w", 0) as f:
                f.write(platform_product_id)
        else:
            verify_pid=True
            if config.has_option("platform", "verify_productid"):
                verify_pid = config.getboolean("platform", "verify_productid")
        
            # Verify if the product id matches with stored one
            if verify_pid:
                self._verifyPID(repo_pid_file, platform_product_id)
            else:
                log.info("Skiping product id check.")

        try:
            self._startConfig = StartConfig(os.path.join(self.rootPath, "start.ini"), self._listFolders())
        except Exception as ex:
            log.exception("Failed to initializes: start.ini :%s" % str(ex))
            log.info("Creating new start.init based on the app folders")
            startini_path = os.path.join(self.rootPath, "start.ini")
            shutil.move(startini_path, startini_path+".bak")
            self._startConfig = StartConfig(os.path.join(self.rootPath, "start.ini"), self._listFolders())
            
        self._initExistingSites()

    def _verifyPID(self, repo_pid_file, platform_product_id):
        """
        Verifies the PID stored in repo_pid_file against the Platform PID
        Raise Exception ValueError if PIDs does not match
        """
        if os.path.exists(repo_pid_file) :
            with open(repo_pid_file, "r") as f:
               r_pid = f.read().strip()
            if r_pid == "default" or  platform_product_id == "default" or r_pid == "" or platform_product_id == "":
                #Did not have product id allow it should not happen in production
                log.info("Allowing for default product id.  Repo pid: %s Platfom product id: %s" % (r_pid, platform_product_id))
                if platform_product_id != "default":
                    #We have got the actual product id replace it with default
                    with open(repo_pid_file, "w", 0) as f:
                        f.write(platform_product_id)
                return True
            if r_pid != platform_product_id:
                log.error("Product ID: %s does not matches with stored PID: %s" %
                        (platform_product_id, r_pid))

                raise ValueError("Product ID: %s does not matches with stored PID: %s" % (platform_product_id, r_pid))
                return False
            else:
                log.info("Platform PID matches with stored PID: %s" % r_pid)
        else:
            log.error("Repo PID file %s does not exists" % repo_pid_file)
            raise ValueError("Repo PID file %s does not exists" % repo_pid_file)
        return True      

    def _initExistingSites(self):
        '''
        Read existing sites and add to the collection
        '''
        if self._startConfig.startOrder is None or self._startConfig.startOrder == "":
            return
        startOrderList =  self._startConfig.startOrder.split(",")
        stOrdList = startOrderList[:]
        for svc_app in stOrdList:
            itemPath = os.path.join(self.rootPath, svc_app)
            log.debug("Loading %s" % itemPath)
            if os.path.isdir(itemPath):
                try:
                    repoFolder = RepoFolder(svc_app, itemPath)
                    self._connectors[svc_app] = Connector(svc_app, repoFolder, restart=True)
                    log.debug("Connectors: %s " % self._connectors)
                except Exception as ex:
                    log.exception("Failed to load application %s Error:%s." % (svc_app, str(ex)))
                    log.info("Removing from %s start list and backing up" % svc_app)
                    startOrderList.remove(svc_app)
                    self.setStartOrder(startOrderList)
                    shutil.move(itemPath, itemPath+".bak")
  

    def listStartOrder(self):
        return self._startConfig.startOrder

    def setStartOrder(self, resStartList):
        resOrder = ",".join(resStartList)        
        self._startConfig.startOrder = resOrder

    def listConnectors(self):
        '''
        Returns an iterator to the list of currently known connectors
        '''
        return iter(self._connectors.values())
    
    def getConnector(self, connectorId):
        '''
        Returns a connector given connector id
        '''
        try:
            return self._connectors[connectorId]
        except KeyError:
            return None

    def setConnector(self, connectorId, value):
        try:
            self._connectors[connectorId] = value
        except KeyError:
            return None

    def createConnector(self, connectorId):
        '''
        Creates a connector. Deployment will happen next.
        After this step, connector will be available in management interfaces, but will only 
        show up its status as "Created".
        '''
        repoFolder = None
        connector = None
        try:
            repoFolder = self._createFolder(connectorId)
            connector = Connector(connectorId, repoFolder)
            
        except:
            e = sys.exc_info()[1]
            log.error("Exception while creating a connector:%s", e)
            if connector:
                self.deleteConnector(connector)
            elif repoFolder:
                repoFolder.delete()
            raise       

        self._connectors[connectorId] = connector
        return connector

    def deleteConnector(self, connector):
        '''
        Deletes a connector
        '''
        
        log.debug("Deleting repo folder %s", str(connector))
        connector._repoFolder.delete()
        self._connectors.pop(connector.id, None)

    
    def _createFolder(self, name):
        '''
        create a folder with the specified name. 
        Throw an error if it already exists
        '''
        folderPath = os.path.join(self.rootPath, name)
        if os.path.exists(folderPath):
            if os.path.isdir(folderPath):
                return RepoFolder(name, folderPath)
            shutil.move(folderPath, folderPath+".bak")
        os.mkdir(folderPath)
        return RepoFolder(name, folderPath)
    
    def _listFolders(self):
        '''
        Returns an iterator of RepoFolder objects corresponding to sub-folders in
        the root of the repo
        '''
        repoFolders = []
        for item in os.listdir(self.rootPath):
            itemPath = os.path.join(self.rootPath, item)
            if os.path.isdir(itemPath):
                repoFolders.append(RepoFolder(item, itemPath))
            else:
                #TODO: Clean-up non-directory entries. we don't expect
                #any of them to be present here
                pass
        return repoFolders


class Connector(object):
    '''
        A connector object represents a deployed connector in the system.
        It holds the master/deployed copy of the connector archive along with the additional metadata
        However, a connector object doesn't represent the hosting environment in which the connector is running
        It will have all the necessary details to connect it back to running container
    '''
    
    def __init__(self, connectorId, repoFolder, restart=False):
        '''
        Initialize a connector
        '''
        self._repoFolder = repoFolder
        self._id = connectorId
        self._metadata = None
        self._reconcile_attempted = False
        # wanted to do this in the load, but the _load happens late in the "deploy" but we want to mark
        # as CREATED earlier
        self._runtimeConfig = RuntimeConfig(self._id, os.path.join(self._repoFolder.getPath(), "db.ini"), restart)
        self._reconcile_attempted = self._runtimeConfig.reconcile_attempted
        self.image_name = None
        self.image_tag = None
        log.debug("Created Connector:%s " % self._id)
    
    def _load(self):
        log.debug("Application loading from repo :%s", self._repoFolder.getPath())
        if os.path.isfile(os.path.join(self._repoFolder.getPath(), APP_DESCRIPTOR_NAME)):
            manifest_file_path = os.path.join(self._repoFolder.getPath(), APP_DESCRIPTOR_NAME)
        elif os.path.isfile(os.path.join(self._repoFolder.getPath(), YAML_MANIFEST_NAME)):
            manifest_file_path = os.path.join(self._repoFolder.getPath(), YAML_MANIFEST_NAME)
        elif os.path.isfile(os.path.join(self._repoFolder.getPath(), INI_MANIFEST_NAME)):
            manifest_file_path = os.path.join(self._repoFolder.getPath(), INI_MANIFEST_NAME)
        else:
            raise MissingManifestException("Could not find manifest file for %r." % self._id)

        try:
            self._metadata = descriptor_metadata_wrapper(self._id, manifest_file_path)
        except Exception as e:
            msg = "Unable to load metadata for connector %r" % self._id
            log.error(msg)
            raise RuntimeError(msg)
            
        if os.path.exists(os.path.join(self._repoFolder.getPath(), USER_EXTRACTED_DIR, "rootfs", DOCKER_IMAGE_FILE )):
            try: 
                log.debug("Loading image detail of %s" % self._id)
                with open(os.path.join(self._repoFolder.getPath(), 
                        USER_EXTRACTED_DIR, "rootfs", DOCKER_IMAGE_FILE), "r") as f:
                    image_details = f.read()
                    image_details = json.loads(image_details)
                    self.image_name = image_details.get("image_name")
                    self.image_tag = image_details.get("image_tag")
            except Exception as e:
                msg = "Unable to load image details %s Error: %s" % (self._id, str(e))
                log.exception(msg)

                    
                     
    @property
    def id(self):
        return self._id
    
    @property
    def reconcile_attempted(self):
        return self._reconcile_attempted
    
    @property
    def metadata(self):
        return self._metadata

    @metadata.setter
    def metadata(self, value):
        self._metadata = value

    @property
    def provides(self):
        return self._metadata.provides

    @property
    def dependsOn(self):
        return self._metadata.dependsOn

    # @property
    # def dependsOnServices(self):
    #     if self.dependsOn:
    #         return self._metadata.dependsOnServices
    #     return None

    @property
    def dependsOnPackages(self):
        if self.dependsOn:
            return self._metadata.dependsOnPackages
        return None

    @property
    def dependsOnCartridges(self):
        if self.dependsOn:
            return self._metadata.dependsOnCartridges
        return None

    @property
    def apptype(self):
        return self._metadata.apptype if self._metadata else ""
    
    @property
    def runtimeConfig(self):
        return self._runtimeConfig

    def getPath(self):
        return self._repoFolder.getPath()

    def set_corrupted(self):
        self._corrupted = True

    @property
    def corrupted(self):
        return self._corrupted


    @property 
    def is_service(self):
        if self._metadata and self._metadata.provides:
            return True
        return False

    @property
    def svc_access_security(self):
        if self._metadata and hasattr(self._metadata, "svc_access_security"):
            return self._metadata.svc_access_security if self._metadata else []
        else:
            return []

    @property
    def svc_security_schemas(self):
        if self._metadata and hasattr(self._metadata, "svc_security_schemas"):
            return self._metadata.svc_security_schemas if self._metadata else []
        else:
            return []

class RepoFolder(object):
    '''
    Represents a repo folder
    '''
    
    def __init__(self, name, folderPath):
        self.folderPath = folderPath
        self.name = name
    
    def getName(self):
        return self.name
    
    def getPath(self):
        return self.folderPath
    
    def getSubFolderPath(self, name):
        return os.path.join(self.folderPath, name)
    
    def createSubFolder(self, name):
        '''
        Creates a subfolder and returns the subfolder path
        '''
        subFolderPath = os.path.join(self.folderPath, name)
        os.mkdir(subFolderPath)
        return subFolderPath
    
    def delete(self):
        log.debug("Deleting folder %s" % self.folderPath)
        shutil.rmtree(self.folderPath, ignore_errors=True)
        
class StartConfig(object):
    """
    Stores the configured runtime status and startupready flag for the container
    
    [start:order]
    svc1, svc2, app1, app2 
    """

    OPTIONS_START = ("start:order", True) # runtime section: (name of the section, isMandatory)
    OPTION_START_ORDER = "order"
    OPTIONS_LIST = [OPTIONS_START] # all known sections

    #repoFolderList to be used in  case start.ini not present 
    def __init__(self, configPath, repoFolderList=None):
        '''
        Load configuration if it exist, if not, create a default one
        '''
        self._configPath = configPath

        # parse configuration
        config = ConfigParser.SafeConfigParser() # parser helper
        config.optionxform = str

        # check if file exists
        if not os.path.exists(configPath):
            if repoFolderList:
                repoFolderNameList = []
                for repoFolder in repoFolderList:
                    repoFolderNameList.append(repoFolder.getName())
                resOrder = ",".join(repoFolderNameList)
                log.debug("Start init not present Buildin start list: %s" , resOrder);

                self._processDefaults(config) # insure default values are present
                config.set(self.OPTIONS_START[0], self.OPTION_START_ORDER, resOrder)
                # file doesn't exist, write file to disk
                with open(configPath, 'w', 0) as configfile:
                    config.write(configfile)
 
        config.read(configPath) # read from file
        self._processDefaults(config) # insure default values are present
        self._config = config;
    

    def _processDefaults(self, config):
        # make sure all sections/optionGroup are present
        for section,mandatory in self.OPTIONS_LIST:
            if not config.has_section(section):
                config.add_section(section)
        if not config.has_option(self.OPTIONS_START[0], self.OPTION_START_ORDER):
            config.set(self.OPTIONS_START[0], self.OPTION_START_ORDER, "")

    
    @property
    def configPath(self):
        return self._configPath
        
    @property
    def startOrder(self):
        if self._config.has_option(self.OPTIONS_START[0], self.OPTION_START_ORDER) :
            return self._config.get(self.OPTIONS_START[0], self.OPTION_START_ORDER)
        else:
            return "" 

    @startOrder.setter
    def startOrder(self, value):
        # if valid, check if its the same
        if value != self._config.get(self.OPTIONS_START[0], self.OPTION_START_ORDER):
            # differ, need to update
            self._config.set(self.OPTIONS_START[0], self.OPTION_START_ORDER, value)
            # file doesn't exist, write file to disk
            with open(self._configPath, 'w', 0) as configfile:
                self._config.write(configfile)


class RuntimeConfig(object):
    """
    Stores the configured runtime status and startupready flag for the container
    
    [connector:runtime]
    status:DEPLOYED
    startupready:[true|false] 
    """

    OPTIONS_RUNTIME = ("connector:runtime", True) # runtime section: (name of the section, isMandatory)
    OPTION_RUNTIME_STATUS = "status"
    OPTION_RUNTIME_STARTUPREADY = "startupready"
    OPTION_RUNTIME_STATUS_VALUES = set([State.RUNNING, State.STOPPED, State.DEPLOYED, State.ACTIVATED])
    OPTION_RUNTIME_APP_GROUP = 'appgroup'

    OPTIONS_APP_DATA = ('connector:data', False)
    OPTION_APP_CUSTOM_OPTION = 'customoption'
    OPTION_APP_UUID = 'uuid'

    OPTIONS_LIST = [OPTIONS_RUNTIME, OPTIONS_APP_DATA] # all known sections

    def __init__(self, connectorId, configPath, restart=False):
        '''
        Load configuration if it exist, if not, create a default one
        restart signifies 
            False: if the application needs to created (new) 
            True: it is already existing
        '''
        self._id = connectorId
        self._configPath = configPath
        self._reconcile_attempted = False

        # parse configuration
        config = ConfigParser.SafeConfigParser() # parser helper
        config.optionxform = str
        try:
            config.read(configPath) # read from file
            self._processDefaults(config, restart) # insure default values are present
        except Exception as ex:
            log.exception("Failure in reading app db.ini :%s" % str(ex))
            if os.path.exists(configPath):
                log.error("Not able to read:%s" % configPath)
                shutil.move(configPath, configPath+".bak")
            # reset and parse configuration
            config = ConfigParser.SafeConfigParser() # parser helper
            config.optionxform = str
            self._processDefaults(config) # insure default values are present
            log.info("Creating new %s with RUNNING state" % configPath)
            config.set(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STATUS, State.RUNNING)
            self._reconcile_attempted = True
        
        # check if file exists
        if not os.path.exists(configPath):
            log.debug("Creating : %s" % configPath)
            if restart:
                #Already existed app but missing db.ini
                config.set(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STATUS, State.RUNNING)
                self._reconcile_attempted = True
            # file doesn't exist, write file to disk
            with open(configPath, 'w', 0) as configfile:
                config.write(configfile)
        
        self._config = config
    
    @property
    def id(self):
        return self._id
    
    @property
    def configPath(self):
        return self._configPath
        
    @property
    def reconcile_attempted(self):
        return self._reconcile_attempted
    
    @property
    def runtimeStatus(self):
        return self._config.get(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STATUS)
    
    @runtimeStatus.setter
    def runtimeStatus(self, value):
        # check if the value is valid
        if value not in self.OPTION_RUNTIME_STATUS_VALUES:
            log.debug("Trying to set an invalid runtime status: " + value)
            return

        # if valid, check if its the same
        if value != self._config.get(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STATUS):
            # differ, need to update
            self._config.set(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STATUS, value)
            # file doesn't exist, write file to disk
            with open(self._configPath, 'w', 0) as configfile:
                self._config.write(configfile)

    @property
    def startupReady(self):
        startupflag = None
        try:
            startupflag = self._config.get(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STARTUPREADY)
        except Exception:
            log.debug("startupready flag is not found: %s", self._id)
        return startupflag

    @property
    def appgroup(self):
        appgroup = False
        if self._config.has_option(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_APP_GROUP):
            appgroup = self._config.getboolean(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_APP_GROUP)
        return appgroup

    @appgroup.setter
    def appgroup(self, value):
        self._config.set(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_APP_GROUP, str(value))
        with open(self._configPath, 'w', 0) as configfile:
            self._config.write(configfile)
        pass

    @startupReady.setter
    def startupReady(self, value):
        self._config.set(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STARTUPREADY, value)
        with open(self._configPath, 'w', 0) as configfile:
            self._config.write(configfile)
        pass
    @property
    def appCustomOptions(self):
        app_custom_option = ""
        if self._config.has_section(self.OPTIONS_APP_DATA[0]):
            if self._config.has_option(self.OPTIONS_APP_DATA[0], self.OPTION_APP_CUSTOM_OPTION):
                app_custom_option = self._config.get(self.OPTIONS_APP_DATA[0], self.OPTION_APP_CUSTOM_OPTION)
                log.debug('Read app custom options. AppId: %s, value: %s' % (self.id, app_custom_option))
        return app_custom_option

    @appCustomOptions.setter
    def appCustomOptions(self, value):
        existing_value = None
        if self._config.has_option(self.OPTIONS_APP_DATA[0], self.OPTION_APP_CUSTOM_OPTION):
            existing_value = self._config.get(self.OPTIONS_APP_DATA[0], self.OPTION_APP_CUSTOM_OPTION)

        # if valid, check if its the same
        if value != existing_value:
            # differ, need to update
            self._config.set(self.OPTIONS_APP_DATA[0], self.OPTION_APP_CUSTOM_OPTION, value)
            with open(self._configPath, 'w', 0) as configfile:
                self._config.write(configfile)
            log.debug('Wrote app custom options. AppId: %s, value: %s' % (self.id, value))

    @property
    def app_uuid(self):
        app_uuid = ""
        if self._config.has_section(self.OPTIONS_APP_DATA[0]):
            if self._config.has_option(self.OPTIONS_APP_DATA[0], self.OPTION_APP_UUID):
                app_uuid = self._config.get(self.OPTIONS_APP_DATA[0], self.OPTION_APP_UUID)
                log.debug('Read app uuid. AppId: %s, value: %s' % (self.id, app_uuid))
        return app_uuid 

    @app_uuid.setter
    def app_uuid(self, value):
        existing_value = None
        if self._config.has_option(self.OPTIONS_APP_DATA[0], self.OPTION_APP_UUID):
            existing_value = self._config.get(self.OPTIONS_APP_DATA[0], self.OPTION_APP_UUID)

        # if valid, check if its the same
        if value != existing_value:
            # differ, need to update
            self._config.set(self.OPTIONS_APP_DATA[0], self.OPTION_APP_UUID, value)
            with open(self._configPath, 'w', 0) as configfile:
                self._config.write(configfile)
            log.debug('Wrote app uuid. AppId: %s, value: %s' % (self.id, value))



    def _processDefaults(self, config, restart=False):
        # make sure all sections/optionGroup are present
        for section,mandatory in self.OPTIONS_LIST:
            if not config.has_section(section):
                if mandatory and restart:
                    raise RepoDatabaseError("Mandatory sections missing from db.ini:%s" % section)
                else:
                    config.add_section(section)

        if not config.has_option(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STATUS):
            config.set(self.OPTIONS_RUNTIME[0], self.OPTION_RUNTIME_STATUS, State.CREATED)
