import threading
import ssl
import logging
import json
import time
import sys
from ws4py.client.threadedclient import WebSocketClient
from ..utils.utils import Utils
from ..utils.infraexceptions import WebSockConnectionError
from appfw.runtime.stats import StatsCollector
from datetime import datetime

log = logging.getLogger("push_service")

__all__ = ["WebSockClientService"]

def retryer(max_retries=5, timeout=2):
    def wraps(func):
        request_exceptions = (
                Exception
        )
        def inner(*args, **kwargs):
            for i in range(max_retries):
                if StatsCollector.getInstance().enabled:
                    restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "websocket")
                    restapi_registry_push.gauge("last_retry_count").set_value(i+1)
                try:
                    result = func(*args, **kwargs)
                except request_exceptions:
                    time.sleep(timeout)
                    continue
                else:
                    return result
            else:
                raise WebSockConnectionError
        return inner
    return wraps

class _WebSockClient(WebSocketClient):
    is_first_connection=0
    def __init__(self, url, on_event, on_error, token, on_closed, headers=None, heartbeat_freq = None, config=None):
        self._config = config
        self._token_id = token
        self._headers = headers
        use_ssl = True
        ssl_options = {}
        if use_ssl:
            ssl_options["cert_reqs"] = ssl.CERT_NONE
            ssl_options["server_side"] = False
        super(_WebSockClient, self).__init__(url, ssl_options, headers=self._headers, heartbeat_freq=heartbeat_freq)
        self.on_event = on_event
        self.on_closed = on_closed
        self.on_error = on_error
        self._closed = threading.Event()
        self._closing = False
        self._closing_lock = threading.RLock()
        self._connected = False


    def opened(self):
        self._connected = True
        self._websocket_open_time = datetime.now()
        if StatsCollector.getInstance().enabled:
            restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "websocket")
            restapi_registry_push.counter("total_connections_opened").inc()
            if _WebSockClient.is_first_connection == 0:
                restapi_registry_push.gauge("first_connection_opened_at").set_value(str(datetime.now()))
                _WebSockClient.is_first_connection = 1
            restapi_registry_push.gauge("last_connection_opened_at").set_value(str(datetime.now()))

    def closed(self, code, reason=None):
        self._closed.set()
        self.on_closed()
        self._connected = False
        if StatsCollector.getInstance().enabled:
            restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "websocket")
            total_websocket_uptime = datetime.now() - self._websocket_open_time
            restapi_registry_push.histogram("websocket_uptime").add(total_websocket_uptime.total_seconds())
            restapi_registry_push.counter("total_connections_closed").inc()
            restapi_registry_push.gauge("last_connection_closed_at").set_value(str(datetime.now()))
            restapi_registry_push.gauge("last_websocket_uptime").set_value(str(total_websocket_uptime))

    def received_message(self, message):
        if not message.is_text:
            log.warning("received data is not text")
            return
        try:
            msg = json.loads(message.data.decode("utf-8"))
        except Exception as ex:
            log.warning("received data is not valid JSON")
            return
        self.on_event(json.loads(str(msg)))

    def close(self, code=1000, reason='', timeout=0):
        super(_WebSockClient, self).close()
        with self._closing_lock:
            self._closing = True
        self._closed.wait(timeout=timeout)

    def connected(self):
        return self._connected


class WebSockClientService(object):
    """
    This class provides interface with websocket client implementation
    User can setup websocket connection with server and manage the lifecycle of the connection
    with api's exposed.User can register for callbacks for error scenarios,message recieved from server
    or when message is sent successfully to the server
    This is threaded implementation of ws4py client
    """
    _to_serialize = ("url_list", "heartbeat_freq", "headers")
    def __init__(self, url_list, on_message_callback, on_error_callback, token=None, headers=None, heartbeat_freq = None, config=None):
        self.url_list = url_list
        self.on_message_callback = on_message_callback
        self.is_closed = threading.Event()
        self.token = token
        self.headers = headers
        self.url = None
        self.heartbeat_freq = heartbeat_freq
        self.url_gen = self.get_url_to_connect()
        self._setup_ws_client()
        self.on_error_callback = on_error_callback

    def get_url_to_connect(self):
        for url in self.url_list:
            yield url

    @retryer(max_retries=3, timeout=5)
    def _ws_client_connect(self, url):
        if StatsCollector.getInstance().enabled:
            restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "websocket")
            restapi_registry_push.counter("total_connection_open_try").inc()
            restapi_registry_push.gauge("last_connection_tried_at").set_value(str(datetime.now()))
            restapi_registry_push.gauge("last_connection_url").set_value(str(url))
        self.ws_client = _WebSockClient(url, self.on_event, self.on_error, self.token, self.on_closed, headers=self.headers, heartbeat_freq=self.heartbeat_freq)
        self.ws_client.daemon = True
        try:
            self.ws_client.connect()
        except Exception as ex:
            self.ws_client.close_connection()
            log.exception("Caught exception %s while connecting to ws server %s", str(ex), self.url)
            if StatsCollector.getInstance().enabled:
                restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "websocket")
                restapi_registry_push.counter("connection_error").inc()
            raise

    def _setup_ws_client(self):
        if StatsCollector.getInstance().enabled:
            restapi_registry_push = StatsCollector.getInstance().get_statsregistry("PUSHNOT", "websocket")
            restapi_registry_push.counter("connection_tries").inc()
        try:
            self.url = next(self.url_gen)
            log.debug("Url to be tried %s", self.url)
        except StopIteration as ex:
            log.exception("Caught Stop Iteration exception on generator %s", str(ex))
            self.on_error(str(ex))
            return False
        except Exception as ex:
            log.exception("Exception while fetching the url %s", str(ex))
            self.on_error(str(ex))
            return False
        try:
            log.debug("Try connection to the server url %s", self.url)
            self._ws_client_connect(self.url)
        except Exception as ex:
            log.exception("Exception while connecting to the server %s", str(ex))
            self._setup_ws_client()
        return True

    def close(self, code=1000, reason="", timeout=0):
        self.is_closed.set()
        self.ws_client.close(code, reason, timeout)

    def on_event(self, message):
        try:
            self.on_message_callback(message)
        except Exception as ex:
            log.exception("Caught exception while calling on_message_callback %s", str(ex))

    def on_error(self, message):
        try:
            self.on_error_callback(message)
        except Exception as ex:
            log.exception("Caught exception while calling on_error_callback %s", str(ex))

    def on_closed(self):
        if not self.is_closed.is_set():
            try:
                self._ws_client_connect(self.url)
                log.debug("Reconnection attempt success")
            except Exception as ex:
                log.exception("Exhausted the max retry attempts %s", str(ex))
                try:
                    log.debug("Trying connection with next available url in the list")
                    self._setup_ws_client()
                except Exception as ex:
                    self.on_error(str(ex))
                    self.is_closed.set()
                    return

    def run_forever(self):
        while not self.is_closed.wait(1):
            pass

    def send_message(self, params, callback=None):
        log.debug("Trying to send message to the ws server %s", self.url)
        if not self.ws_client.connected():
            log.warning("Client is not connected to the server")
            return False
        #Payload has to be in json
        self.ws_client.send(params)
        return True
        #callback("Message sent successfully")

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

if __name__ == "__main__":

    def on_message_cb(msg):
        print "%s" % msg
    def cb(msg):
        print "In callback"
    def on_error(msg):
        print "Connection error"
    payload = {"event":"test", "cisco":"rjil"}

    print "starting ws client"
    url = ["wss://10.78.106.96:8443/api/v1/appmgr/notification"]
    wsc = WebSockClientService(url, on_message_cb, on_error)
    wsc.send_message(json.dumps(payload))
    while 1:
        time.sleep(1)
