__author__ = 'hvishwanath'

import logging
import ast
import json
import os
import tempfile

from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import NetworkConfigurationError, NetworkManagementError
from appfw.utils.infraexceptions import NetworkNotExistingError
from appfw.utils.commandwrappers import *
from netaddr import *
from network_utils import *
from macregistry import MacRegistry
from portregistry import PortRegistry
from appfw.runtime.caf_abstractservice import CAFAbstractService
from appfw.api.systeminfo import SystemInfo
from appfw.hosting.apptypes import AppType

log = logging.getLogger("pdservices")

BRIDGE_AGEING_DEFAULT = "300"
BRIDGE_MAC_FORWARD_BASE_MASK = 0xfff8 

class HostingBridge(object):
    """
    Represents a bridge on the hosting linux platform.
    Each bridge in turn corresponds to an interface on IOS that connects to hosting linux.
    For ex. svcbr_0 is connected to VPG0.
    """
    _to_serialize = ("description", "vlan_id", "external_interface", "bridge_ip", "interface",
                     "default_mode", "supported_modes", "interface_info", "lease_info",
                     "logical_network_info", "dynamically_created", "mirror_mode", "creation_mode",
                     "mac_forward_mode", "mirroring_support", "vlan_hosting",
                     "secure_storage", "arp_ignore")
    
    # bridge_ip modes
    NO_IP_ADDRESS = "no_ip_address"
    DHCP = "dhcp"
    STATIC = "static"
    DPBR_PREFIX = "dpbr_"
    DPBR_NAT_PREFIX = "dpbr_n_"
    DPBR_NAT_DOCKER_PREFIX = "dpbr_docker_n_"

    def __init__(self, interface, bridge_settings):
        # Bridge is the name of the bridge that this object represents (ex. svcbr_0)
        self.interface = interface
        self.bridge_settings = bridge_settings
        self.description = bridge_settings.get("description", "")
        self.supported_modes = bridge_settings["supported_modes"]
        self.external_interface = bridge_settings.get("external_interface")
        self.dhcp_lease_file = bridge_settings.get("dhcp_lease_file")
        self.generated_mac_addr = bridge_settings.get("generated_mac_addr")
        self.dynamically_created = bridge_settings.get("dynamically_created", False)
        self.creation_mode = bridge_settings.get("creation_mode", "user")
                                                       
        self.vlan_id = bridge_settings.get("vlan_id")

        self.direct_connect = False
        if bridge_settings.get("direct_connect") and self.vlan_id and \
                    self.supported_modes == [Network.NETWORK_TYPE_BRIDGE]:
            # Direct connection to external interface is available only if bridge network is requested
            # In this case, we do not call the svcbr_setup/teardown scripts, CAF sets up
            # direct connectivity of dpbr to external interface
            self.direct_connect = True
            self.interface = self.external_interface
            self.bridge_settings["svcbr_setup_script"] = None
            self.bridge_settings["svcbr_teardown_script"] = None
            if self.vlan_id:
                self.interface = self.external_interface + '.' + self.vlan_id

        self.assoc_networks = {}
        self.interface_info = {}

        self.mirroring_support = bridge_settings.get("mirroring_support", False)
        if self.mirroring_support:
            log.debug("bridge %s has mirroring_support", self.interface)
            
        self.arp_ignore = bridge_settings.get("arp_ignore", False)
        if self.arp_ignore:
            log.debug("bridge %s has arp_ignore ON", self.interface)

        self.vlan_hosting = False

        self.default_mode = bridge_settings.get("default_mode", Network.NETWORK_TYPE_BRIDGE)
        if Network.NETWORK_TYPE_BRIDGE not in self.supported_modes:
            self.default_mode = Network.NETWORK_TYPE_NAT
        
        self.mirror_mode = False
        if Network.NETWORK_TYPE_BRIDGE in self.supported_modes:        
            bridgemode_info = bridge_settings.get(Network.NETWORK_TYPE_BRIDGE)
            if bridgemode_info and bridgemode_info.get("mirror_mode") == "yes":
                self.mirror_mode = True

        self.mac_forward_mode = bridge_settings.get("mac_forward_mode", False)
        self.gp_fwd_mask_default = bridge_settings.get("mac_forward_mask_default", 0)
        self.gp_fwd_mask = self.gp_fwd_mask_default 
        if self.mac_forward_mode:
            path = os.path.join('/sys/class/net/', self.interface, 'bridge/group_fwd_mask')
            if os.path.exists(path):
                log.debug("Setting mask %s to %s", BRIDGE_MAC_FORWARD_BASE_MASK, path)
                self.set_gp_fwd_mask(path, BRIDGE_MAC_FORWARD_BASE_MASK)

        self.secure_storage = bridge_settings.get("secure_storage", False)

        self.set_bridge_ip()
        self.set_nat_range()
        self.set_nat_docker_range()
        
        self.netns = bridge_settings.get("netns", None)
        self.required = bridge_settings.get("required", True)
        self.is_internal = bridge_settings.get("internal_use", False)
        self.vpgno = bridge_settings.get("bridge_no", None)
        if self.vpgno != None:
            self.vpgno = str(self.vpgno)

        self.logical_network_info = {}
        self.bridge_ip = self.bridge_settings["bridge_ip"]
          
        log.debug("Bridge_settings: %s", bridge_settings)        

    def set_gp_fwd_mask(self, path, mask):
        try:
            with open(path, 'w', 0) as f:
                f.write(str(mask))
        except IOError:
            log.error("Failed to configure bridge")
            raise IOError

    def set_bridge_ip(self):
        bridge_ip = {"mode": self.NO_IP_ADDRESS}
        if Network.NETWORK_TYPE_NAT in self.supported_modes:
            bridge_ip = self.bridge_settings.get("bridge_ip")
            if bridge_ip == None:
                # Default to dhcp if not provided
                bridge_ip = {"mode": self.DHCP}
                 
            if bridge_ip["mode"] == self.STATIC:
                # subnet mask may be provided openly as 255.255.255.0 or in cidr format: 24
                # Convert and save it in open format here
                ip_addr = bridge_ip.get("ip")
                subnet_mask = bridge_ip.get("subnet_mask")
                if not ip_addr or not subnet_mask:
                    raise NetworkManagementError("IP address or subnet_mask missing for static bridge config.")
                ip_addr_with_mask = IPNetwork(ip_addr + '/' + subnet_mask)
                bridge_ip["subnet_mask"] = str(ip_addr_with_mask.netmask)
                
                if "bridge_gw_ip" not in bridge_ip:
                    bridge_ip["bridge_gw_ip"] = ""
                    
                if "dns" not in bridge_ip:
                    bridge_ip["dns"] = ""
                    
                if "domain" not in bridge_ip:
                    bridge_ip["domain"] = ""    
                
                bridge_gw_ip = bridge_ip.get("bridge_gw_ip")
                if self.bridge_settings[Network.NETWORK_TYPE_NAT].get("setup_private_routing") and not bridge_gw_ip:
                    raise NetworkManagementError("Gateway ip is missing for static bridge config.")
                    
                if bridge_gw_ip:
                    if bridge_gw_ip == ip_addr:
                        raise NetworkManagementError("Invalid Gateway IP: %s, needs to be different than "
                                                        "Bridge IP %s" % (bridge_gw_ip, ip_addr))
                                                        
                    gw_ip_with_mask = IPNetwork(bridge_gw_ip + '/' + subnet_mask)
                    # Bridge ip and gateway ip needs to be on the same subnet 
                    if gw_ip_with_mask.network != ip_addr_with_mask.network:
                        raise NetworkManagementError("Invalid Gateway IP: %s, needs to be in same subnet "
                                                        "as %s/%s" % (bridge_gw_ip, ip_addr, subnet_mask))
                
        self.bridge_settings["bridge_ip"] = bridge_ip
        
    def set_nat_range(self, nat_info=None):      
        if Network.NETWORK_TYPE_NAT in self.supported_modes:
            if not nat_info:
                nat_info = self.bridge_settings[Network.NETWORK_TYPE_NAT]
            
            self.bridge_settings[Network.NETWORK_TYPE_NAT] = nat_info
            
    def set_nat_docker_range(self, nat_info=None):      
        if Network.NETWORK_TYPE_NAT_DOCKER in self.supported_modes:
            if not nat_info:
                nat_info = self.bridge_settings[Network.NETWORK_TYPE_NAT_DOCKER]
            
            self.bridge_settings[Network.NETWORK_TYPE_NAT_DOCKER] = nat_info
            
    def get_lease_info(self):
        if self.bridge_ip["mode"] == self.DHCP:
            self.lease_info = parse_dhcp_client_lease(self.dhcp_lease_file)
            if not self.lease_info["fixed_address"]:
                self.lease_info = get_lease_info_from_static_configuration(self.interface_info)
        elif self.bridge_ip["mode"] == self.STATIC:
            self.lease_info = get_lease_info_from_static_configuration(self.interface_info)
            
            self.lease_info["fixed_address"] = self.bridge_ip["ip"]
            self.lease_info["subnet_mask"] = self.bridge_ip["subnet_mask"]
            
            bridge_gw_ip = self.bridge_ip.get("bridge_gw_ip")
            if bridge_gw_ip:
                self.lease_info["routers"] = bridge_gw_ip
                
            domain = self.bridge_ip.get("domain")
            if domain:
                self.lease_info["domain_name"] = domain
                
            dns = self.bridge_ip.get("dns")
            if dns:
                self.lease_info["dns"] = dns
            
        return self.lease_info
            
    def setup(self):
        """
        Call PD script to create bridge
        Example: scripts/svcbr_setup.sh -i intsvc0 -v 10 -a dhcp -sf $tmpfile.yaml
        Example: scripts/svcbr_setup.sh -i intsvc0 -v 10 -a static -ip 192.168.0.15 -m 255.255.255.0 -sf $tmpfile.yaml
        get status_code, message, dhcp-leases file, bridge_id from PD script in $tmpfile.yaml
        Example $tmpfile.yaml:
        hosting_bridges:
          svcbr_1:
            vlan_id: 10
            phys_interface: intsvc0
            bridge_ip:
              mode: dhcp
              dhcp_lease_file: /etc/platform/dhcp/svcbr_1.leases
            status_code: 0   
            message: "Successfully created"            
        """

        try:
            setup_script = self.bridge_settings.get("svcbr_setup_script")
            bridge_ip = self.bridge_settings["bridge_ip"]
            status_fname = None
            
            if setup_script:
                scripts_dir = Utils.getScriptsFolder()
                setup_script = os.path.join(scripts_dir, setup_script)
                    
                cmd = [setup_script]
                
                # If recovering from persistent storage, we know the bridge_id
                # Otherwise PD script picks the name for us. Naming convention: svcbr_x
                if self.interface:
                    cmd.extend(['-b', self.interface])
                    
                if self.vlan_id:
                    cmd.extend(['-v', str(self.vlan_id)])
                
                cmd.extend(['-i', self.external_interface])
                    
                if self.bridge_ip["mode"] == self.DHCP:
                    cmd.extend(['-a', self.DHCP])
                
                if self.bridge_ip["mode"] == self.STATIC:
                    cmd.extend(['-a', self.STATIC])
                    cmd.extend(['-ip', bridge_ip["ip"]])
                    cmd.extend(['-m', bridge_ip["subnet_mask"]])
                    
                if self.generated_mac_addr:
                    cmd.extend(['-mac', self.generated_mac_addr])
                
                # Create tmpfile status_fname
                status_dir = os.path.dirname(NetworkController.svcbr_status_file)
                if not os.path.isdir(status_dir):
                    os.makedirs(status_dir)

                fd, status_fname = tempfile.mkstemp(suffix='.yaml', prefix='tmp', dir=status_dir)
                    
                cmd.extend(['-of', status_fname])
                log.debug("Executing BRIDGE CREATION CMD:%s" % cmd)
                
                # Call PD script
                output, rcode = call_script(cmd)
               
                # PD script writes results to status_fname. Collect and parse 
                status = get_pd_status(status_fname)
                
                if not status:
                    raise NetworkManagementError("Bad platform response while creating network. "
                                                 "Return code: %s, Output: %s" % (rcode, output))
                                                 
                log.debug("Platform bridge creation script returned: %s" % status)
                
                # Get bridge_id that PD script decided
                bridge_id = status.keys()[0]
                
                # If recovering from persistent storage, saved bridge_id and script output should match
                if self.interface and bridge_id != self.interface:
                    raise NetworkManagementError("Bad platform response while creating bridge: %s" % self.interface)
                
                self.interface = bridge_id
                message = status[self.interface].get("message")
                status_code = status[self.interface].get("status_code")
                
                # PD script failed while creating the bridge
                if status_code != 0:
                    raise NetworkManagementError("Platform error while creating bridge %s: %s" % (self.interface, message))
                
                # Get dhcp_lease_file from PD script yaml file
                returned_bridge_ip = status[self.interface]["bridge_ip"]
                if self.bridge_ip["mode"] == self.DHCP:
                    self.dhcp_lease_file = returned_bridge_ip.get("dhcp_lease_file")
                    
            elif self.direct_connect:
                if self.vlan_id and int(self.vlan_id) > 0 and int(self.vlan_id) <= 4094:

                    # Direct dpbr connectivity requested with valid vlan
                    # We need to create subinterface for external_interface.vlan
                    out, rc = ipcmd("link", "show", self.interface)
                    if rc != 0:
                        out, rc = ipcmd("link", "add", "link", self.external_interface, "name", self.interface, "type", "vlan", "id", self.vlan_id)
                        if rc != 0:
                            raise NetworkManagementError("VLAN interface %s creation has failed. Error: %s" % 
                                                          (self.interface, str(out)))

                        log.debug("VLAN interface %s has been created.", self.interface)


            # Based on the naming convention, each bridge name will have the corresponding vpg number of the gige no.
            # as suffix joined with an underscore   
            dpbr_name = self.DPBR_PREFIX
            dpbr_nat_name = self.DPBR_NAT_PREFIX
            dpbr_nat_docker_name = self.DPBR_NAT_DOCKER_PREFIX
            if self.vpgno == None: 
                if self.vlan_id:
                    self.vpgno = self.vlan_id
                    dpbr_name = self.DPBR_PREFIX + "v" 
                    dpbr_nat_name = self.DPBR_NAT_PREFIX + "v" 
                    dpbr_nat_docker_name= self.DPBR_NAT_PREFIX + "v" 
                elif  "_" in self.interface:
                    self.vpgno = self.interface.split("_")[-1]
                else:
                    # No vlan id provided and bridge name doesn't meet the convention svcbr_0
                    # Find first available vpgno
                    self.vpgno = NetworkController.next_available_vpgno()
                    dpbr_name = self.DPBR_PREFIX + "r"
                    dpbr_nat_name = self.DPBR_NAT_PREFIX + "r"
                    dpbr_nat_docker_name= self.DPBR_NAT_PREFIX + "r"

            self.bridgemode_name = dpbr_name + self.vpgno
            self.natmode_name = dpbr_nat_name + self.vpgno
            self.natmode_docker_name = dpbr_nat_docker_name + self.vpgno
            self._validate()

            self.interface_info = Utils.get_interface_info(self.interface, self.netns)
            if self.interface_info.get("ipv4_address") and self.interface_info.get("subnet_mask"):
                if self.bridge_ip["mode"] == self.NO_IP_ADDRESS:
                    self.bridge_ip["mode"] = self.STATIC
                    self.bridge_ip["ip"] = self.interface_info["ipv4_address"]
                    self.bridge_ip["subnet_mask"] = self.interface_info["subnet_mask"]
                    self.bridge_settings["bridge_ip"] = self.bridge_ip

            self.lease_info = {}
            self.get_lease_info()

            if self.mirroring_support and not self.vlan_id:
                out, rc = brctl("setageing", self.interface, BRIDGE_AGEING_DEFAULT)
                if rc != 0:
                    log.error("reset setageing failed on bridge %s: %s", self.interface, out)

            if self.arp_ignore and not self.vlan_id:
                if not self.netns:
                    set_arp_ignore(self.interface)

            br_type = "hosting_br"
            if self.vlan_id:
                br_type = "vlan"
            if NetworkController.bridge_properties_script and \
               os.path.exists(NetworkController.bridge_properties_script):

                # Bring down interface (ex: L2br, L2br.X) before calling bridge_properties script
                if self.arp_ignore and not self.netns:
                    out, rc = ipcmd("link", "set", "dev", self.interface, "down")
                    if rc != 0:
                        log.error("Couldn't bring down interface %s. Error: %s", self.interface, str(out))

                    log.debug("Interface %s is brought down.", self.interface)

                output, rcode = call_script(NetworkController.bridge_properties_script,
                                            "post_create",
                                            self.interface,
                                            br_type,
                                            str(self.netns))
                if rcode != 0:
                    # log error and continue
                    log.error("Error returned by %s: returncode: %s, message: %s", NetworkController.bridge_properties_script,
                                                                                   rcode, output)
                else:
                    log.debug("Successfully executed %s: returncode: %s, message: %s", NetworkController.bridge_properties_script,
                                                                                       rcode, output)

        except (ValueError, AttributeError) as ex:
            log.error("Unexpected Platform error during network creation: %s" % str(ex))
            self.teardown()
            raise Exception("Unexpected Platform error while creating the network.")
            
        except:
            self.teardown()
            raise
            
        finally:
            if status_fname and os.path.isfile(status_fname):
                os.remove(status_fname)
      

    def teardown(self):
        """
        Call PD teardown script to delete the bridge
        Example: scripts/svcbr_teardown.sh -b svcbr_1 -i intsvc0 -v 10
        """
        
        if not self.interface:
            return

        try:
            teardown_script = self.bridge_settings.get("svcbr_teardown_script")

            if not teardown_script:
                if self.direct_connect and self.vlan_id:
                    # Delete subinterface for external_interface.vlan
                    out, rc = ipcmd("link", "show", self.interface)
                    if rc == 0:
                        out, rc = ipcmd("link", "delete", "dev", self.interface)
                        if rc != 0:
                            log.error("VLAN interface %s delete has failed", self.interface)

                        log.debug("VLAN interface %s has been deleted", self.interface)

                return

            scripts_dir = Utils.getScriptsFolder()
            teardown_script = os.path.join(scripts_dir, teardown_script)
            
            cmd = [teardown_script]
            
            cmd.extend(['-b', self.interface])
            cmd.extend(['-i', self.external_interface])
            if self.vlan_id:
                cmd.extend(['-v', self.vlan_id])
                
            output, rcode = call_script(cmd)
            
            if rcode != 0:
                log.error("Could not teardown bridge %s: %s" % (self.interface, output))
                
        except Exception as ex:
            log.error("Could not teardown bridge %s: %s" % (self.interface, str(ex)))
            
    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def _validate(self):
        out, rc = ipcmd("link", "show", self.interface, netns=self.netns)
        if rc != 0:
            raise NetworkConfigurationError("Hosting bridge %s is not found!? Error: %s" % (self.interface, out))

    def get_bridge_ipv4_address(self):
        if self.interface_info:
            return self.interface_info['ipv4_address']

    def get_bridge_ipv6_address(self):
        if self.interface_info:
            return self.interface_info['ipv6_address']

    def get_bridge_assoc_brnw(self):
        for nw in self.assoc_networks.values():
            if nw.network_type == "bridge":
                return nw.name
        return None

class Network(object):
    """
    Represents a logical network in the Fog Node.
    Each network is connected to a hosting bridge and can be of type bridge/nat.
    Contains methods that help in setting up and tearing down the physical network.
    Depending on the container information passed, this also sets up container specific network infra
    (ex. creating libvirt network domains etc.,)

    This network itself is identified by a name and provided relevant information.
    """

    LIBVIRT_BRIDGE_TEMPLATE = """
    <network>
        <name>{NETWORK_NAME}</name>
        <bridge name='{BRIDGE_NAME}' stp='off' delay='0'/>
    </network>

    """

    LIBVIRT_NAT_TEMPLATE = """
    <network>
      <name>{NETWORK_NAME}</name>
      <forward mode='nat' />
      <bridge name='{BRIDGE_NAME}' stp='off' delay='0'/>
      <dns>
        <forwarder addr='{DNS_SERVER}' />
      </dns>
      <ip address='{GATEWAY_IP}' netmask='{NETMASK}'>
        <dhcp>
          <range start='{IP_START}' end='{IP_END}'/>
          {IP_DHCP_STATIC_ENTRIES}
        </dhcp>
      </ip>
    </network>
    """

    LIBVIRT_IP_DHCP_HOST_TEMPLATE = """
    <host mac='{MAC_ADDRESS}' name='{DOMAIN_NAME}' ip='{IP_ADDRESS}' />
    """

    NETWORK_TYPE_NAT = "nat"
    NETWORK_TYPE_BRIDGE = "bridge"
    NETWORK_TYPE_NAT_DOCKER = "nat_docker"
    #NETWORK_TYPE_HOST = "host"

    _to_serialize = ("name",
                     "description",
                     "repofolder",
                     "source_linux_bridge",
                     "network_type",
                     "gateway_ip",
                     "subnet_mask",
                     "ip_start",
                     "ip_end",
                     "nat_range_cidr",
                     "external_interface",
                     "private_route_table",
                     "app_ip_map",
                     "mirror_mode",
                     "mirroring_support",
                     "vlan_hosting",
                     "arp_ignore")

    def __init__(self,
                 name,
                 description,
                 source_linux_bridge,
                 network_type,
                 network_settings,
                 repofolder,
                 mac_address_block=None,
                 container_info={},
                 phys_interface=None,
                 reconcile=False,
                 docker_conn = None,
                 libvirt_conn = None,
                 auto_br = None):

        self.name = name
        self.libvirt_conn_hdl = libvirt_conn
        self.docker_api_client = docker_conn
        self.description = description
        self.source_linux_bridge = source_linux_bridge
        self.network_type = network_type
        self.repofolder = repofolder
        self.datafile = os.path.join(self.repofolder, "."+self.name)
        self.mac_address_block = mac_address_block
        self.phys_interface = phys_interface
        if source_linux_bridge:
            self.libvirt_network_name = source_linux_bridge.bridgemode_name
            self.external_interface = source_linux_bridge.external_interface
            self.phys_interface = source_linux_bridge.interface
            self.direct_connect = source_linux_bridge.direct_connect
        else:
            # Auto bridge mode
            vpgno = filter(str.isdigit, phys_interface)
            self.libvirt_network_name = NetworkController.auto_br_libvirt_prefix + vpgno
            self.external_interface = phys_interface

        # Depending on the network type, we expect appropriate settings
        self.network_settings = network_settings

        self.gateway_ip = None
        self.subnet_mask = None
        self.ip_start = self.ip_end = None
        self.dhcp_lease_file = None
        self.private_route_table = None
        if self.source_linux_bridge:
            self.private_route_table = str(int(self.source_linux_bridge.vpgno) + 10)
        self.lease_info = None
        self.private_routing = False
        
        self.mirror_mode = False
        self.vethpair = []
        
        self.network_manager = NetworkController.getInstance(None)

        self.mirroring_support = False
        self.vlan_hosting = False
        self.arp_ignore = False
        self.nat_range_cidr =  None

        if self.network_type == self.NETWORK_TYPE_NAT or self.network_type == self.NETWORK_TYPE_NAT_DOCKER:
            self.subnet_mask =self.network_settings["subnet_mask"]
            self.ip_start = self.network_settings["ip_start"]
            self.ip_end = self.network_settings["ip_end"]
            self.gateway_ip = self.network_settings["gateway_ip"]
            self.nat_range_cidr = self.network_settings.get("nat_range_cidr")
            self.lease_info = self.source_linux_bridge.lease_info
            self.private_routing = self.network_settings.get("setup_private_routing", False)
            if self.network_type == self.NETWORK_TYPE_NAT:
                self.libvirt_network_name = self.source_linux_bridge.natmode_name
            if self.network_type == self.NETWORK_TYPE_NAT_DOCKER:
                self.libvirt_network_name = self.source_linux_bridge.natmode_docker_name

            # None means not-defined, don't do anything
            # [] means don't allow any ports
            self.allowed_tcp_ports = self.network_settings.get("allowed_tcp_ports", None)
            self.allowed_udp_ports = self.network_settings.get("allowed_udp_ports", None)

            # Not-defined and [] means the same thing: No ports will be blocked
            self.blocked_tcp_ports = self.network_settings.get("blocked_tcp_ports", [])
            self.blocked_udp_ports = self.network_settings.get("blocked_udp_ports", [])

            if (self.allowed_tcp_ports is not None) or (self.allowed_udp_ports is not None):
                # if allowed ports are specified, we are already blocking all other ports
                # So no additional rules for blocking ports needed
                self.blocked_tcp_ports = []
                self.blocked_udp_ports = []

            # netaddr.IPSet for the nat ip range this NAT network is using
            if self.nat_range_cidr:
                self.nat_range_ipset = IPSet(IPNetwork(self.nat_range_cidr))
            else:
                self.nat_range_ipset = IPSet(IPRange(self.ip_start, self.ip_end))
                self.nat_range_ipset.add(self.gateway_ip) # In case outside of above range
                
        elif self.network_type == self.NETWORK_TYPE_BRIDGE and self.source_linux_bridge:
            self.mirror_mode = self.source_linux_bridge.mirror_mode
            self.mirroring_support = self.source_linux_bridge.mirroring_support
            self.arp_ignore = self.source_linux_bridge.arp_ignore
            self.vlan_hosting = self.source_linux_bridge.vlan_hosting
            vlan_id = self.source_linux_bridge.vlan_id

            # bridgemode_name is in the form: dpbr_0, dpbr_v0, dpbr_r0
            v_prefix = self.source_linux_bridge.bridgemode_name.split("_")[-1]
            self.vethpair = ("veth0_" + v_prefix, "veth1_" + v_prefix)
            self.veth_high_order_mac = None

        elif auto_br:
            self.auto_br = auto_br
            self.mirroring_support = auto_br['mirroring_support']
            self.arp_ignore = auto_br['arp_ignore']

        self.container_info = container_info

        # TODO: This model of one single Network object setting up networks for multiple containers
        # TODO: may not work. For now, this is the best I could think of!

        # Store the network information that you would create for each container
        # map of container=>network name for now.
        self.container_network = dict()

        # Keep track of IPs statically mapped to applications. Used only by NAT networks
        self.app_ip_map = dict()

        # Keep track of MAC<->IP address association as obtained by the MAC Address block
        self.mac_ip_map = dict()

        # UUID of the network created
        self.uuid = None
      
    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                if k == "source_linux_bridge":
                    if f == None:
                        d[k] = "None"
                    else:
                        d[k] = f.interface
                else:
                    d[k] = f
        return d

    def _save_data(self):
        # Store the ipmap for now
        o = json.dumps(self.app_ip_map)
        with open(self.datafile, "w", 0) as f:
            f.write(o)

    def _load_data(self):
        d = dict()
        try:
            if os.path.isfile(self.datafile):
                with open(self.datafile, "r") as f:
                    d = json.load(f)
                log.debug("Loaded network data from %s", self.datafile)
        except Exception as ex:
            log.error("Error loading network data from %s:%s" % (self.datafile, str(ex)))
            log.debug("Traceback:",  exc_info=True)

        self.app_ip_map = d
        
    def get_container_network(self, container_id):
        """
        App is configured to use THIS network.
        The container now needs to find out the corresponding container specific details.
        For now with libvirt, this simply returns the name of libvirt network domain if one is available.
        :param container_id:
        :return:
        """
        return self.container_network.get(container_id)

    def get_container_source_bridge(self):
        return self.source_linux_bridge

    def setup(self, reconcile=False):
        """
        Setup Bridge and NAT networks on self.bridge
        :return:
        """
        try:
            log.debug("Setting up network %s", self.name)
            if self.network_type == self.NETWORK_TYPE_NAT:
                self.setup_nat_network(reconcile)
            elif self.network_type == self.NETWORK_TYPE_NAT_DOCKER:
                self.setup_nat_docker_network(reconcile)
            elif self.network_type == self.NETWORK_TYPE_BRIDGE:
                self.setup_bridge_network(reconcile)
            else:
                raise NetworkConfigurationError("Unsupported network type : %s" % self.network_type)
                
        except Exception as ex:
            log.exception("Failed to setup network: %s" % str(ex))
            self.teardown()
            raise

    def teardown(self, is_getting_shutdown=False):
        """
        Tear down already setup Bridge and NAT networks on self.bridge
        :return:
        """
        log.debug("Tearing down network %s", self.name)
        
        try:
            if self.network_type == self.NETWORK_TYPE_NAT:
                self.teardown_nat_network()
            elif self.network_type == self.NETWORK_TYPE_NAT_DOCKER:
                self.teardown_nat_docker_network(is_getting_shutdown)
            elif self.network_type == self.NETWORK_TYPE_BRIDGE:
                self.teardown_bridge_network()
            else:
                raise NetworkConfigurationError("Unsupported network type : %s" % self.network_type)
        
        except Exception as ex:
            log.error("Could not teardown network %s: %s" % (self.name, str(ex)))


    def network_status(self):
        if self.network_type == self.NETWORK_TYPE_NAT_DOCKER:
            if not self.docker_api_client:
                base_url = Utils.getSystemConfigValue("docker-container", "docker_base_url")
                api_version = Utils.getSystemConfigValue("docker-container", "docker_api_version")
                timeout = Utils.getSystemConfigValue("docker-container", "docker_timeout_seconds", parse_as="int")
                use_tls = Utils.getSystemConfigValue("docker-container", "docker_use_tls", parse_as="bool")
                self.docker_api_client = Utils.get_docker_api_client(base_url, api_version, timeout, use_tls)
                log.debug("docker api client config headers = %s" % self.docker_api_client.headers)
            docker_network_name = self.source_linux_bridge.natmode_docker_name
            log.debug("Docker network name: %s" % docker_network_name)
            if self.docker_api_client:
                #Verify if network already exists
                docker_network_filter = [docker_network_name]
                docker_net_list = self.docker_api_client.networks(names=docker_network_filter)
            log.debug("docker_net_list:%s" % docker_net_list)
            if docker_net_list:
                return True;
            else:
                return False;
        else:
            if self.network_type == self.NETWORK_TYPE_NAT:
                libvirt_network_name = self.source_linux_bridge.natmode_name
            else:
                libvirt_network_name = self.libvirt_network_name

        import libvirt
        try:
            if self.libvirt_conn_hdl is None:
                self.libvirt_conn_hdl = libvirt.open()

            nw = self.libvirt_conn_hdl.networkLookupByName(libvirt_network_name)
            if nw:
                if not nw.isActive():
                    log.info("%s: is down" % libvirt_network_name)
                    return False
                return True
            else:
                return False
        except libvirt.libvirtError as e:
            log.error("Failed to get libvirt network status, Error: %s code: %s" % (str(e), e.get_error_code()))
            log.debug("Traceback:", exc_info=True)
            if e.get_error_code() == libvirt.VIR_ERR_INVALID_CONN or e.get_error_code() == libvirt.VIR_ERR_INTERNAL_ERROR or e.get_error_code() == libvirt.VIR_FROM_STREAMS:
                if self.libvirt_conn_hdl:
                    try:
                        self.libvirt_conn_hdl.close()
                    except Exception as ex:
                        log.exception("Error while handling the libvirt connection: %s"%str(ex))
                    self.libvirt_conn_hdl = None
                try:
                    self.libvirt_conn_hdl = libvirt.open()
                    nw = self.libvirt_conn_hdl.networkLookupByName(libvirt_network_name)
                    if nw:
                        if not nw.isActive():
                            log.info("%s: is down" % libvirt_network_name)
                            return False
                        return True
                    else:
                        return False
                except Exception as e:
                    log.error("Failed to get libvirt network status, Error: %s" % str(e))
                    log.debug("Traceback:",  exc_info=True)
                    return False
            else:
                return False




    def _check_if_enslaved(self, interface, bridge, netns=None):
        """
        Check if the given interface is enslaved by the bridge. If yes, return True, else False.
        Error: Raise NetworkConfigurationError.
        :param interface:
        :param bridge:
        :return:
        """
        out, rc = brctl("show", bridge, netns=netns)
        if rc != 0:
            raise NetworkConfigurationError("Error doing a brctl show on %s: %s" % (bridge, str(out)))

        if interface in out:
            return True

        # Some platforms may have a version of brctl that will not accept arguments
        # In that case, parse the output to check if bridge enslaves the interface
        out, rc = brctl("show", netns=netns)
        if rc != 0:
            raise NetworkConfigurationError("Error doing a brctl show without arguments")

        if out:
            lines = out.split("\n")
            for line in lines:
                # if bridge name and interface name appears in the same line,
                # the interface is enslaved by the bridge
                if bridge in line and interface in line:
                    return True

        return False


    def set_mirroring(self, mirroring):
        """
        Apply "brctl setageing dpbr_X 0" to the network  
        """

        if self.network_type != self.NETWORK_TYPE_BRIDGE:
            if mirroring:
                raise NetworkConfigurationError("Mirroring is only supported for bridge type networks.")
            return

        # Mirroring only enabled for hosting bridges/auto_bridge with mirroring support
        if not self.mirroring_support:
            if mirroring:
                raise NetworkConfigurationError("Cannot activate on network %s that does not support mirroring." % self.name)
            return

        log.debug("app_ip_map %s, mirror_mode %s", self.app_ip_map, self.mirror_mode)

        if not mirroring:
            # App is not asking mirroring, block activation if another app with mirror mode is enabled
            if self.app_ip_map and self.mirror_mode:
                raise NetworkConfigurationError("Cannot activate on network. Another app with mirror mode is enabled on %s." % self.name)
            return

        # App is asking for mirroring, block activation if another app with no mirroring is running on the network
        if self.app_ip_map and not self.mirror_mode:
            raise NetworkConfigurationError("Another app is running on the network %s. Cannot enable mirroring." % self.name)

        out, rc = brctl("setageing", self.libvirt_network_name, "0")
        if rc != 0:
            raise NetworkConfigurationError("setageing failed on bridge %s" % self.libvirt_network_name)

        # If trunk mode, also need to set mirroring for the L2br in addition to the dpbr
        if self.source_linux_bridge and not self.source_linux_bridge.vlan_id:
            out, rc = brctl("setageing", self.source_linux_bridge.interface, "0")
            if rc != 0:
                raise NetworkConfigurationError("setageing failed on bridge %s" % self.source_linux_bridge.interface)

        log.debug("Set mirror mode for network %s is successful", self.libvirt_network_name)
        self.mirror_mode = True

    def reset_mirroring(self):
        """
        Reset the ageing for the network bridge to default
        """
        if self.network_type != self.NETWORK_TYPE_BRIDGE:
            return

        if not self.mirror_mode:
            return

        out, rc = brctl("setageing", self.libvirt_network_name, BRIDGE_AGEING_DEFAULT)
        if rc != 0:
            raise NetworkConfigurationError("reset setageing failed on bridge %s" % self.libvirt_network_name)

        if self.source_linux_bridge and not self.source_linux_bridge.vlan_id:
            out, rc = brctl("setageing", self.source_linux_bridge.interface, BRIDGE_AGEING_DEFAULT)
            if rc != 0:
                raise NetworkConfigurationError("reset setageing failed on bridge %s" % self.source_linux_bridge.interface)

        log.debug("Reset mirror mode for network %s is successful", self.libvirt_network_name)
        self.mirror_mode = False

    def set_mac_forwarding(self, mac_forward_mask):
        """
        Enable mac forwarding
        :param mac_forward_mask: Mask value to set (None indicates no operation)
        :return:
        """
        log.debug("set_mac_forwarding mac_forward_mask=%s", mac_forward_mask) 
        if not self.source_linux_bridge or \
           not self.source_linux_bridge.mac_forward_mode: 
            return
        log.debug("self masks=%s, %s", self.source_linux_bridge.gp_fwd_mask, \
                                       self.source_linux_bridge.gp_fwd_mask_default)
        log.debug("app_ip_map=%s", self.app_ip_map) 

        lb = self.libvirt_network_name

        # Allow activation of app without mac-foward and current bridge mask is same as default
        if mac_forward_mask == None and \
           self.source_linux_bridge.gp_fwd_mask == self.source_linux_bridge.gp_fwd_mask_default:
            log.debug("Null mask: grant new activation without changing current mask.")
            return

        # Block activation requesting mac-forward if another app is already using the interface 
        if self.app_ip_map and \
           mac_forward_mask != self.source_linux_bridge.gp_fwd_mask:
            raise NetworkManagementError("Another app is running on %s. Cannot set mac forwarding from %s to %s" % \
			                             (lb, self.source_linux_bridge.gp_fwd_mask, mac_forward_mask))

        # Setting mask: requested via app descriptor and confirmed with activation payload.
        lb_path = os.path.join('/sys/class/net/', lb, 'bridge/group_fwd_mask')
        log.debug("path: %s; mask value: %s", lb_path, mac_forward_mask)
        try:
            if os.path.exists(lb_path):
                with open(lb_path, 'w', 0) as f:
                    f.write(str(mac_forward_mask))
        except IOError:
            log.error("Failed to configure bridge")
            raise IOError

        self.source_linux_bridge.gp_fwd_mask = mac_forward_mask

    def reset_mac_forwarding(self):
        """
        Disable mac forwarding
        :param:
        :return:
        """
        log.debug("reset_mac_forwarding") 
        if not self.source_linux_bridge or \
           not self.source_linux_bridge.mac_forward_mode: 
            return

        lb = self.libvirt_network_name

        mac_forward_mask = self.source_linux_bridge.gp_fwd_mask_default 

        lb_path = os.path.join('/sys/class/net/', lb, 'bridge/group_fwd_mask')
        log.debug("paths: %s; mask value: %s", lb_path, mac_forward_mask)
        try:
            if os.path.exists(lb_path):
                with open(lb_path, 'w', 0) as f:
                    f.write(str(mac_forward_mask))
        except IOError:
            log.error("Failed to configure bridge")
            raise IOError

        self.source_linux_bridge.gp_fwd_mask = mac_forward_mask

    def app_setup_hook(self, appid, interface_name, mac_address, port_mappings, mac_forward_mask='', mirroring=False, reconcile=False ):
        """
        Do app specific setup
        :param appid:
        :param port_mappings:
        :return:
        """
        log.debug("Network app_setup_hook for %s, interface=%s, mac_forward_mask=%s, mirroring=%s", \
                  appid, interface_name, mac_forward_mask, mirroring)

        self.set_mac_forwarding(mac_forward_mask)

        self.set_mirroring(mirroring)

        self.set_mtu()

        if self.network_type == self.NETWORK_TYPE_BRIDGE:
            self.app_ip_map[appid] = {interface_name: {"mac_address": mac_address}}
            log.debug("%s is bridge network. Nothing to do..", self.name)
            return

        if self.network_type == self.NETWORK_TYPE_NAT_DOCKER:
            self.app_ip_map[appid] = {interface_name: {"mac_address": mac_address}}
            log.debug("%s is docker nat network. Nothing to do..", self.name)
            return

        if self.network_type != self.NETWORK_TYPE_NAT:
            raise NetworkConfigurationError("Unsupported network type %s" % self.network_type)

        # Now we are in NAT mode

        # 1. Set static mapping in libvirt network configuration and update it live

        if "libvirt" in self.container_info:
        
            # Get lease info again, in case dhcp leases file was generated late
            if self.source_linux_bridge.bridge_ip["mode"] == HostingBridge.DHCP:
                self.lease_info = self.source_linux_bridge.get_lease_info()
        
            if self.private_routing and not reconcile:
                # Do if not already done before
                self._setup_private_routing()
        
            libvirt_network_name = self.container_network.get("libvirt")
            c = None

            import libvirt
            try:
                if self.libvirt_conn_hdl:
                    c = self.libvirt_conn_hdl
                else:
                    c = libvirt.open()
                    self.libvirt_conn_hdl = c
                log.debug("Looking up libvirt network domains for %s", libvirt_network_name)
                try:
                    nw = c.networkLookupByName(libvirt_network_name)
                except libvirt.libvirtError:
                    # We should not be in this state. Something wrong
                    log.error("Unable to find libvirt network: %s", libvirt_network_name)
                    raise

                # Get an ip address
                if appid not in self.app_ip_map:
                    self.app_ip_map[appid] = dict()

                appmap = self.app_ip_map.get(appid)
                appiface = appmap.get(interface_name)

                if appiface is None:
                    # Get an ip address and save this data

                    # Note: The mac_ip_map MUST have an entry. If not, let it fail so that we can see what is causing it
                    ipaddress = self.mac_ip_map[mac_address]
                    appiface = {"mac_address": mac_address, "ipv4": ipaddress}
                    appmap[interface_name] = appiface

                    log.debug("Procured ip address %s for app %s", ipaddress, appid)
                    self._save_data()

            except Exception as ex:
                raise

            # 2. Setup port forwarding

            if port_mappings and not reconcile:
                log.debug("Setting up port forwarding based on the port mapping information received..")
                if not self.lease_info["fixed_address"]:
                    raise NetworkConfigurationError("No ip address assigned for bridge %s" % self.source_linux_bridge.interface)
                hostip = self.lease_info["fixed_address"]
                guestip = self.app_ip_map[appid][interface_name]["ipv4"]

                for type in port_mappings:
                    type_mappings = port_mappings.get(type)
                    for m in type_mappings:
                        guestport, hostport = m[0], m[1]

                        # Send the mask as /32. Apply rules ONLY to this ip.
                        setup_port_forwarding(type, hostip, hostport, guestip, guestport, "255.255.255.255")

                        # If there are any exceptions, let it raise!

            log.debug("App setup hook for app %s complete", appid)


    def app_teardown_hook(self, appid, interface_name, mac_address, port_mappings):
        """
        Do app specific setup
        :param appid:
        :param port_mappings:
        :return:
        """
        log.debug("Network app_teardown_hook for %s, interface=%s", appid, interface_name) 

        if self.source_linux_bridge and \
           self.source_linux_bridge.gp_fwd_mask != self.source_linux_bridge.gp_fwd_mask_default: 
            self.reset_mac_forwarding()

        if self.network_type == self.NETWORK_TYPE_BRIDGE:
            self.app_ip_map.pop(appid, None)
            log.debug("%s is bridge network. Nothing to do..", self.name)
            return

        if self.network_type == self.NETWORK_TYPE_NAT_DOCKER:
            self.app_ip_map.pop(appid, None)
            log.debug("%s is docker nat network. Nothing to do..", self.name)     
            return

        if self.network_type != self.NETWORK_TYPE_NAT:
            raise NetworkConfigurationError("Unsupported network type %s" % self.network_type)

        # Now we are in NAT mode

        # 1. Remove the static mapping entry from libvirt's dhcp host map

        if appid not in self.app_ip_map:
            log.debug("AppID %s not found in internal map. Nothing to do...", appid)
            return

        if interface_name not in self.app_ip_map.get(appid):
            log.debug("Interface name %s not found in internal map for app %s. Nothing to do...", interface_name, appid)
            return

        if "libvirt" in self.container_info:
            # 1. Remove static ip-dhcp-host entries
            # Update: Not needed anymore. Will be removed as part of network teardown.

            # 2. Remove port forwarding rules

            if port_mappings:
                log.debug("Removing port forwarding rules based on port mapping information received..")
                hostip = self.source_linux_bridge.lease_info["fixed_address"]
                guestip = self.app_ip_map[appid][interface_name]["ipv4"]

                for type in port_mappings:
                    type_mappings = port_mappings.get(type)
                    for m in type_mappings:
                        guestport, hostport = m[0], m[1]

                        # Send the mask as /32. Apply rules ONLY to this ip.
                        remove_port_forwarding(type, hostip, hostport, guestip, guestport, "255.255.255.255")

                        # If there are any exceptions, let it raise!
            #Remove entries from app_ip_map
            try:
                pop = self.app_ip_map.pop(appid)
                log.debug("App ip map is removed for app %s , removed contents are : %s" % (appid, pop))
            except KeyError:
                log.error("No entry found for the app %s in app ip map to delete"% appid)
                log.debug("Traceback:",  exc_info=True)

            log.debug("App teardown hook for app %s complete", appid)


    def setup_bridge_network(self, reconcile=False):
        """
        Setup a bridge network and attach it to hosting bridge.
        Ensure that dpbr_vpgno is created via the container that this bridge is associated with (self.container)
        Create veth pair and attach this to hosting bridge
        Note that there is NO IP address attached to this dpbr_* bridge.

        PS: dpbr -> Data path bridge
        :return:
        """

        # Handle container specific initialization
        if "libvirt" in self.container_info:
            libvirt_network_name = self.libvirt_network_name
            log.debug("Setting up bridge network for libvirt containers..")
            log.debug("Libvirt network name: %s" % libvirt_network_name)

            import libvirt
            c = None

            try:
                if self.libvirt_conn_hdl:
                    c = self.libvirt_conn_hdl
                else:
                    c = libvirt.open()
                    self.libvirt_conn_hdl = c
                log.debug("Looking up libvirt network domains for %s", libvirt_network_name)
                try:
                    nw = c.networkLookupByName(libvirt_network_name)
                    log.debug("Pre existing network found %s", libvirt_network_name)
                    if not reconcile:
                        log.debug("Will be tearing it down before setting it up with correct configuration..")
                        if nw.isActive():
                            nw.destroy()
                            log.debug("Stopped network %s", nw.name())
                        nw.undefine()
                        log.debug("Undefined network %s", nw.name())
                        nw = None
                except libvirt.libvirtError:
                    nw = None

                if nw is None:
                    log.debug("Couldn't find a network domain by name : %s", libvirt_network_name)
                    log.debug("Attempting to define one..")
                    xml_string = self.LIBVIRT_BRIDGE_TEMPLATE.format(BRIDGE_NAME=libvirt_network_name,
                                                                     NETWORK_NAME=libvirt_network_name)
                    log.debug("Sending XML : %s", xml_string)
                    try:
                        nw = c.networkDefineXML(xml_string)
                        if not nw.isActive():
                            nw.create()

                        log.debug("Defined and started network : %s", nw.name())
                    except libvirt.libvirtError as le:
                        raise le
                else:
                    log.debug("Libvirt network domain %s is already present", libvirt_network_name)
                    if not nw.isActive():
                        nw.create()
                        log.debug("Started libvirt network : %s", nw.name())
                    else:
                        log.debug("Libvirt network %s is already active, nothing to do..", nw.name())
                #nw = c.networkLookupByName(libvirt_network_name)
                self.uuid = nw.UUIDString()
                # Store the network created here
                self.container_network["libvirt"] = libvirt_network_name

            except Exception as ex:
                if "libvirt" in self.container_network:
                    self.container_network.pop("libvirt")
                raise

        else:
            raise NotImplementedError("Not sure what do with these containers : %s" % self.container_info.keys())
        if self.source_linux_bridge and not self.source_linux_bridge.direct_connect:
            # Networking via svcbr bridge
            self.bridge_network_with_svcbr(libvirt_network_name)
        else:
            # Standalone bridge network with no svcbr bridge (auto_bridge_mode)
            self.auto_bridge_network_without_svcbr(libvirt_network_name)

        if self.arp_ignore:
            set_arp_ignore(libvirt_network_name)

        if self.source_linux_bridge and self.source_linux_bridge.mac_forward_mode:
            lb = self.libvirt_network_name
            lb_path = os.path.join('/sys/class/net/', lb, 'bridge/group_fwd_mask')
            mask = self.source_linux_bridge.gp_fwd_mask_default 
            log.debug("lb_path:%s mask:%d", lb_path, mask)
            if os.path.exists(lb_path) and mask: 
                with open(lb_path, 'w', 0) as f:
                    log.debug("Writing %s to %s", mask, lb_path)
                    f.write(str(mask))

        if NetworkController.bridge_properties_script and \
           os.path.exists(NetworkController.bridge_properties_script):
            output, rcode = call_script(NetworkController.bridge_properties_script,
                                        "post_create",
                                        libvirt_network_name,
                                        "bridge",
                                        "None")
            if rcode != 0:
                # log error and continue
                log.error("Error returned by %s: returncode: %s, message: %s", NetworkController.bridge_properties_script,
                                                                               rcode, output)
            else:
                log.debug("Successfully executed %s: returncode: %s, message: %s", NetworkController.bridge_properties_script,
                                                                                   rcode, output)

        if self.source_linux_bridge and not self.source_linux_bridge.netns:
            out, rc = ipcmd("link", "set", "dev", self.source_linux_bridge.interface, "up")
            if rc != 0:
                log.error("Couldn't bring up interface %s. Error: %s", self.source_linux_bridge.interface, str(out))

            log.debug("Interface %s is brougt up.", self.source_linux_bridge.interface)

    def bridge_network_with_svcbr(self, libvirt_network_name):
    
        # Create veth pair and dpbr_0 to source bridge

        # check veth0_ and veth1_ exists
        log.debug("Attempting to create veth pair %s", str(self.vethpair))
        out0, rc0 = ipcmd("link", "show", self.vethpair[0], netns=self.source_linux_bridge.netns)
        out1, rc1 = ipcmd("link", "show", self.vethpair[1])

        if rc0 != rc1:
            # There is some issue is here. One of the pairs seem to be present, while the other is not.
            if rc0 == 0 and rc1 == 1:
                raise NetworkConfigurationError("One end of the veth pair %s exists, while %s is not. Error : %s" %
                                                (self.vethpair[0],
                                                 self.vethpair[1],
                                                 str(out1)))

            if rc1 == 0 and rc0 == 1:
                raise NetworkConfigurationError("One end of the veth pair %s exists, while %s is not. Error: %s" %
                                                (self.vethpair[1],
                                                 self.vethpair[0],
                                                 str(out0)))

        # Both do not exist. So let us create them.
        if rc0 !=0 and rc1 !=0 :
            out, rc = ipcmd("link", "add", self.vethpair[0], "type", "veth", "peer", "name", self.vethpair[1])
            if rc != 0:
                raise NetworkConfigurationError("Couldn't create veth pair %s. Error: %s" % (str(self.vethpair),
                                                                                             str(out)))

            if self.source_linux_bridge.netns != None:
                out, rc = ipcmd("link", "set", self.vethpair[0], "netns", str(self.source_linux_bridge.netns))
                if rc != 0:
                    raise NetworkConfigurationError("Couldn't set %s to network namespace. Error: %s" % (str(self.vethpair[0]),
                                                                                                 str(out)))
            log.debug("Successfully created veth pair %s", str(self.vethpair))

        self.update_veth_mac()

        self.set_mtu()

        # Connect bridgemode_name to hosting_bridge using veth pair
        log.debug("Connecting %s to %s", libvirt_network_name, self.source_linux_bridge.interface)

        if not self._check_if_enslaved(self.vethpair[0], self.source_linux_bridge.interface, self.source_linux_bridge.netns):
            out, rc = brctl("addif", self.source_linux_bridge.interface, self.vethpair[0], netns=self.source_linux_bridge.netns)
            if rc != 0:
                raise NetworkConfigurationError("Couldn't connect %s with %s. Error: %s" %
                                                (self.source_linux_bridge.interface,
                                                 self.vethpair[0],
                                                 str(out)))

        if not self._check_if_enslaved(self.vethpair[1], libvirt_network_name):
            out, rc = brctl("addif", libvirt_network_name, self.vethpair[1])
            if rc != 0:
                raise NetworkConfigurationError("Couldn't connect %s with %s. Error: %s" %
                                                (libvirt_network_name,
                                                 self.vethpair[1],
                                                 str(out)))

        log.debug("Successfully connected %s to %s",
                  libvirt_network_name,
                  self.source_linux_bridge.interface)

        # Set the veth pair up
        out, rc = ipcmd("link", "set", self.vethpair[0], "up", netns=self.source_linux_bridge.netns)
        if rc != 0:
            raise NetworkConfigurationError("Couldn't bring up %s. Error: %s" %
                                            (self.vethpair[0],
                                             str(out)))

        out, rc = ipcmd("link", "set", self.vethpair[1], "up")
        if rc != 0:
            raise NetworkConfigurationError("Couldn't bring up %s. Error: %s" %
                                            (self.vethpair[1],
                                             str(out)))

        if self.mirror_mode:
            out, rc = brctl("setageing", self.libvirt_network_name, "0")
            if rc != 0:
                raise NetworkConfigurationError("setageing failed on bridge %s" % self.libvirt_network_name)

            out, rc = brctl("setageing", self.source_linux_bridge.interface, "0")
            if rc != 0:
                raise NetworkConfigurationError("setageing failed on bridge %s" % self.source_linux_bridge.interface)

        # That's it, we are done.
        log.debug("Bridge mode setup complete")

    def auto_bridge_network_without_svcbr(self, libvirt_network_name):
        """
        For standalone bridge network (auto_bridge_mode)
        Enslave physical interface to dpbr_X without veth pair
        """
        if not self._check_if_enslaved(self.phys_interface, libvirt_network_name):
            out, rc = brctl("addif", libvirt_network_name, self.phys_interface)
            if rc != 0:
                raise NetworkConfigurationError("Couldn't connect %s with %s. Error: %s" %
                                                (libvirt_network_name,
                                                 self.phys_interface,
                                                 str(out)))

    def update_veth_mac(self):
        """
        A Linux software bridge will assume the MAC address of the enslaved
        interface with the numerically lowest MAC addr. In case platform's svcbr_X
        bridge has higher mac address than the veth that CAF creates, it will get the
        veth's mac address, which we don't want. Platforms should try to avoid a scenario
        like this, but if CAF encounters it, it will assign a mac address to veth that starts
        with 0xFE, so that svcbr_X retains its own MAC address.
        More info at: https://www.redhat.com/archives/libvir-list/2010-July/msg00450.html
        """

        veth_mac = get_inf_mac_address(self.vethpair[0])
        svcbr_mac = get_inf_mac_address(self.source_linux_bridge.interface)

        log.debug("%s mac address: %s, %s mac address: %s", self.source_linux_bridge.interface,
                                                            svcbr_mac,
                                                            self.vethpair[0],
                                                            veth_mac)
        if veth_mac < svcbr_mac:

            log.debug("Need to assign a large mac address for %s", self.vethpair[0])

            mac = self.network_manager.macregistry.get_mac_address_for_veth()

            out, rc = ipcmd("link", "set", "dev", self.vethpair[0], "address", mac, netns=self.source_linux_bridge.netns)
            if rc != 0:
                raise NetworkConfigurationError("Couldn't set mac address for %s. Error: %s" %
                                                (self.vethpair[0],
                                                 str(out)))

            self.veth_high_order_mac = mac

    def set_mtu(self):
        """
        Platform is responsible for setting/configuring the mtu of the physical interfaces and svcbr's.
        CAF will match the mtu of the networks and bridges it creates same value as the mtu of the platform's infs.
        set_mtu method will be called at the creation of each CAF network (including CAF startup)
        set_mtu method will also be called whenever an app gets activated on a CAF network (to catch mtu config changes)
        If mtu update is not successful, error is logged, but CAF startup or app activation will not be blocked
        Platform's svcbr may be in a different network namespace than CAF's network namespace
        """
        
        if not self.network_manager.update_network_mtu:
            return

        mtu_ref_inf = self.phys_interface
        netns = None
        if self.source_linux_bridge:
            mtu_ref_inf = self.source_linux_bridge.interface
            netns = self.source_linux_bridge.netns

        target_mtu = get_inf_mtu(mtu_ref_inf, netns)

        if not target_mtu:
            # Error already logged in get_inf_mtu, don't block CAF startup or app activation
            return

        current_mtu = get_inf_mtu(self.libvirt_network_name)

        if target_mtu == current_mtu:
            # mtu of dpbr matches mtu of svcbr
            log.debug("%s mtu=%s matches platform interface %s, skip...", self.libvirt_network_name, target_mtu, mtu_ref_inf)
            return

        # Set mtu of the vnets that dpbr enslaves
        out, rc = brctl("show", self.libvirt_network_name)
        if rc != 0:
            log.error("Error doing brctl show %s", self.libvirt_network_name)
        elif out:
            for line in out.split("\n"):
                if line.strip().startswith("vnet"):
                    set_inf_mtu(line.strip(), target_mtu)

        # Set mtu of dpbr
        set_inf_mtu(self.libvirt_network_name + "-nic", target_mtu)
        set_inf_mtu(self.libvirt_network_name, target_mtu)

        # Set mtu of veth pair
        # veth pair exists for bridge type networks, doesn't exist for nat or auto created bridges
        # vethpair[0] is in the same nw namespace as source_bridge, which may be different than CAF's namespace
        if self.vethpair and not self.direct_connect:
            set_inf_mtu(self.vethpair[0], target_mtu, netns=netns)
            set_inf_mtu(self.vethpair[1], target_mtu)

    def delete_enslaved_interface(self, bridge_name): 
        if self._check_if_enslaved(self.phys_interface, bridge_name):
            br_path = "/sys/class/net/%s/brif/" % bridge_name
            if os.path.isdir(br_path) and len(os.listdir(br_path)) == 2:
                out, rc = brctl("delif", bridge_name, self.phys_interface)
                if rc != 0:
                    raise NetworkConfigurationError("Couldn't remove %s from %s. Error: %s" %
                                                        (bridge_name,
                                                         self.phys_interface,
                                                         str(out)))


    def teardown_bridge_network(self):

        if self.source_linux_bridge and not self.source_linux_bridge.direct_connect:
            # Check if veth pair exists and remove them if so.
            log.debug("Removing veth pair : %s", str(self.vethpair))                
            out0, rc0 = ipcmd("link", "show", self.vethpair[0], netns=self.source_linux_bridge.netns)
            out1, rc1 = ipcmd("link", "show", self.vethpair[1])

            if rc0 == 0 or rc1 == 0:
                # Atleast one of them is present. Let us remove them.
                # Removing one, will remove the corresponding peer also.
                out, rc = ipcmd("link", "del", self.vethpair[1])
                if rc != 0:
                    raise NetworkConfigurationError("Couldn't remove %s. Error: %s" %
                                                    (self.vethpair[0],
                                                     str(out)))

                log.debug("Removed veth pair..")

                # This should automatically remove the interfaces from enslavement of the bridge.

            # Release the high order mac address for veth if assigned
            if self.veth_high_order_mac in self.network_manager.macregistry.HIGH_GENERATED_MAC_LIST:
                self.network_manager.macregistry.HIGH_GENERATED_MAC_LIST.remove(self.veth_high_order_mac)

        else:
            # Standalone bridge network, no veth pair exists
            # remove the interface from enslavement of the bridge.
            self.delete_enslaved_interface(self.libvirt_network_name)


        # Now, let us remove the bridge itself.
        # Handle container specific teardown

        if "libvirt" in self.container_info:
            libvirt_network_name = self.libvirt_network_name
            log.debug("Tearing down bridge network for libvirt containers : %s", libvirt_network_name)

            import libvirt
            c = None

            try:
                if self.libvirt_conn_hdl:
                    c = self.libvirt_conn_hdl
                else:
                    c = libvirt.open()
                    self.libvirt_conn_hdl = c
                log.debug("Looking up libvirt network domains for %s", libvirt_network_name)
                try:
                    nw = c.networkLookupByName(libvirt_network_name)
                except libvirt.libvirtError:
                    nw = None

                if nw is None:
                    log.error("Libvirt network %s is not found. Nothing to do, but this should not happen",
                              libvirt_network_name)
                else:
                    log.debug("Tearing down network : %s", libvirt_network_name)
                    if nw.isActive():
                        nw.destroy()
                        log.debug("Stopped network %s", nw.name())
                    nw.undefine()
                    log.debug("Undefined network %s", nw.name())
                self.uuid = None
            except Exception as ex:
                log.error("Network: Error while tearing down the bridge network. Cause: %s"%ex.message)

        # Remove the network
        if "libvirt" in self.container_network:
            self.container_network.pop("libvirt")

        log.debug("Bridge mode teardown complete")


    def teardown_nat_docker_network(self, is_getting_shutdown=False):
        if is_getting_shutdown:
            # TODO: when docker daemon is shutdown it doesn't seem to tear down the docker network dpbr_docker_n_0
            # So this logic needs to be revisited

            #Do not teardown the docker network let it be handled by docker daemon
            log.debug("CAF is getting shutdown do not tear the docker network explicitly")
            return
        if "docker" in self.container_info:
            if self.docker_api_client:
                try:
                    docker_network_name = self.source_linux_bridge.natmode_docker_name
                    log.debug("Tearing down nat docker network for docker containers: %s", docker_network_name)
                    self.docker_api_client.remove_network(docker_network_name)
                except Exception as ex:
                    log.exception("Failed to remove docker network: %s Error: %s" %  (docker_network_name, str(ex)))

                self.remove_iptables_rules_nat()

    def teardown_nat_network(self):

        # Now, let us remove the bridge itself.
        # Handle container specific teardown

        if "libvirt" in self.container_info:
            if self.private_routing:
                self._teardown_private_routing()

            libvirt_network_name = self.source_linux_bridge.natmode_name
            log.debug("Tearing down nat network for libvirt containers: %s", libvirt_network_name)

            import libvirt
            c = None

            try:
                if self.libvirt_conn_hdl:
                    c = self.libvirt_conn_hdl
                else:
                    c = libvirt.open()
                    self.libvirt_conn_hdl = c
                log.debug("Looking up libvirt network domains for %s", libvirt_network_name)
                try:
                    nw = c.networkLookupByName(libvirt_network_name)
                except libvirt.libvirtError:
                    nw = None

                if nw is None:
                    log.error("Libvirt network %s is not found. Nothing to do, but this should not happen",
                              libvirt_network_name)
                else:
                    log.debug("Tearing down network : %s", libvirt_network_name)
                    if nw.isActive():
                        nw.destroy()
                        log.debug("Stopped network %s", nw.name())
                    nw.undefine()
                    log.debug("Undefined network %s", nw.name())
                self.uuid = None
            except Exception as ex:
                log.error("Network: Error while tearing down the NAT network. Cause: %s"%ex.message)

            self.remove_iptables_rules_nat()

        # Remove the network
        if "libvirt" in self.container_network:
            self.container_network.pop("libvirt")


        log.debug("NAT mode teardown complete")


    def _teardown_private_routing(self):

        # Step 1
        # Remove route entries in private table

        log.debug("Tearing down private routing table : %s", self.private_route_table)
        network_with_prefix = get_network_address_with_prefix(self.lease_info["fixed_address"],
                                                              self.lease_info["subnet_mask"])

        log.debug("Removing routing table entries in table %s", self.private_route_table)
        out, rc = ipcmd("route",
                     "del",
                     network_with_prefix,
                     "dev",
                     self.source_linux_bridge.interface,
                     "src",
                     self.lease_info["fixed_address"],
                     "table",
                     str(self.private_route_table))

        if rc != 0:
            if rc == 2 and "No such process" in out:
                # Private routing not set up before
                return
            else:
                raise NetworkConfigurationError("Couldn't remove route from table %s. Error: %s" %
                                                (self.private_route_table,
                                                 str(out)))


        log.debug("Removing default route added to private table")
        out, rc = ipcmd("route",
                     "del",
                     "default",
                     "via",
                     self.lease_info["routers"],
                     "dev",
                     self.source_linux_bridge.interface,
                     "table",
                     str(self.private_route_table))

        if rc != 0:
            if rc == 2 and "No such process" in out:
                pass
            else:
                raise NetworkConfigurationError("Couldn't remove default route from table %s. Error: %s" %
                                                (self.private_route_table,
                                                 str(out)))


        # Step 2. Remove fwmark
        # ip rule del fwmark 10 table 10

        log.debug("Removing ip rule that applies that sets up routing table for marked packets")
        out, rc = ipcmd("rule",
                     "del",
                     "fwmark",
                     self.private_route_table,
                     "table",
                     self.private_route_table)

        if rc != 0:
            raise NetworkConfigurationError("Couldn't set forward marking to %s. Error: %s" %
                                            (self.private_route_table,
                                             str(out)))


        # Step 3. Remove previously set iptable entries

        # # Rule to tag packet entering dpbr_n_0 with forwarding mark so the correct external NAT ip address can be picked up
        # iptables -t mangle -A PREROUTING -s 11.11.11.0/24 ! -d 11.11.11.0/24 -j MARK --set-mark 10
        # # Rule to tag packet once it has been masqueraded with the correct external IP address to be routed with the alternate routing table
        # iptables -t mangle -A OUTPUT -o dpbr_n_0 -j MARK --set-mark 10

        log.debug("Removing mangling that marks incoming packets with %s", self.private_route_table)
        nat_network = get_network_address_with_prefix(self.ip_start, self.subnet_mask)
        out, rc = iptables("-t",
                           "mangle",
                           "-D",
                           "PREROUTING",
                           "-s",
                           nat_network,
                           "!",
                           "-d",
                           nat_network,
                           "-j",
                           "MARK",
                           "--set-mark",
                           self.private_route_table
                           )
        if rc != 0:
            raise NetworkConfigurationError("Couldn't remove iptable rule that tags incoming packets with %s. Error: %s" %
                                            (self.private_route_table,
                                             str(out)))

        log.debug("Removing mangling that marks outgoing packets with %s", self.private_route_table)
        out, rc = iptables("-t",
                           "mangle",
                           "-D",
                           "OUTPUT",
                           "!",
                           "-d",
                           nat_network,
                           "-o",
                           self.source_linux_bridge.natmode_name,
                           "-j",
                           "MARK",
                           "--set-mark",
                           self.private_route_table
                           )
        if rc != 0:
            raise NetworkConfigurationError("Couldn't remove iptable rule that tags outgoing packets with %s. Error: %s" %
                                            (self.private_route_table,
                                             str(out)))


    def _setup_private_routing(self):
        """
        Add these two rules to table vpgno + 10
        http://wikicentral.cisco.com/display/C3A/819+NAT+Networking+Updates

        Step 1:
        ----------
        ip route add 192.168.0.0/24 dev svcbr_0 src 192.168.0.2 table 10
        ip route add default via 192.168.0.1 dev svcbr_0 table 10

        Step 2:
        -------
        ip rule add fwmark 10 table 10

        Step 3:
        -------
        # Rule to tag packet entering dpbr_n_0 with forwarding mark so the correct external NAT ip address can be picked up
        iptables -t mangle -A PREROUTING -s 11.11.11.0/24 ! -d 11.11.11.0/24 -j MARK --set-mark 10
        # Rule to tag packet once it has been masqueraded with the correct external IP address to be routed with the alternate routing table
        iptables -t mangle -A OUTPUT ! -d 11.11.11.0/24 -o dpbr_n_0 -j MARK --set-mark 10

        :return:
        """
        if not self.lease_info or not self.lease_info.get("fixed_address") or \
           not self.lease_info.get("subnet_mask") or not self.lease_info.get("routers"):
            raise NetworkConfigurationError("Missing ip address and/or gateway ip for bridge %s "
                                            "for private routing." % self.source_linux_bridge.interface)
        
        log.debug("Setting up private routing table : %s", self.private_route_table)
        network_with_prefix = get_network_address_with_prefix(self.lease_info["fixed_address"],
                                                              self.lease_info["subnet_mask"])

        log.debug("Adding route to table %s for the network %s", self.private_route_table, network_with_prefix)
        # Add a route in table 10
        out, rc = ipcmd("route",
                     "add",
                     network_with_prefix,
                     "dev",
                     self.source_linux_bridge.interface,
                     "src",
                     self.lease_info["fixed_address"],
                     "table",
                     str(self.private_route_table))

        if rc != 0:
            if rc == 2 and "File exists" in out:
                # Done this before
                return
            else:
                raise NetworkConfigurationError("Couldn't add route to table %s. Error: %s" %
                                                (self.private_route_table,
                                                 str(out)))


        log.debug("Adding a default route via the default gateway for source bridge")
        out, rc = ipcmd("route",
                     "add",
                     "default",
                     "via",
                     self.lease_info["routers"],
                     "dev",
                     self.source_linux_bridge.interface,
                     "table",
                     str(self.private_route_table))
                     

        if rc != 0:
            if rc == 2 and "File exists" in out:
                pass
            else:
                raise NetworkConfigurationError("Couldn't add default route to table %s. Error: %s" %
                                                (self.private_route_table,
                                                 str(out)))


        # Step 2. Add fwmark
        # ip rule add fwmark 10 table 10

        log.debug("Setting ip rule to apply new routing table for packets marked with %s", self.private_route_table)
        out, rc = ipcmd("rule",
                     "add",
                     "fwmark",
                     self.private_route_table,
                     "table",
                     self.private_route_table)

        if rc != 0:
            raise NetworkConfigurationError("Couldn't set forward marking to %s. Error: %s" %
                                            (self.private_route_table,
                                             str(out)))


        # Step 3. Setup iptable rules to tag packets.
        # # Rule to tag packet entering dpbr_n_0 with forwarding mark so the correct external NAT ip address can be picked up
        # iptables -t mangle -A PREROUTING -s 11.11.11.0/24 ! -d 11.11.11.0/24 -j MARK --set-mark 10
        # # Rule to tag packet once it has been masqueraded with the correct external IP address to be routed with the alternate routing table
        # iptables -t mangle -A OUTPUT ! -d 11.11.11.0/24 -o dpbr_n_0 -j MARK --set-mark 10

        log.debug("Setting up mangling to mark incoming packets with %s", self.private_route_table)
        nat_network = get_network_address_with_prefix(self.ip_start, self.subnet_mask)
        out, rc = iptables("-t",
                           "mangle",
                           "-A",
                           "PREROUTING",
                           "-s",
                           nat_network,
                           "!",
                           "-d",
                           nat_network,
                           "-j",
                           "MARK",
                           "--set-mark",
                           self.private_route_table
                           )
        if rc != 0:
            raise NetworkConfigurationError("Couldn't add iptable rule to tag incoming packets with %s. Error: %s" %
                                            (self.private_route_table,
                                             str(out)))

        log.debug("Setting up mangling to mark outgoing packets with %s", self.private_route_table)
        out, rc = iptables("-t",
                           "mangle",
                           "-A",
                           "OUTPUT",
                           "!",
                           "-d",
                           nat_network,
                           "-o",
                           self.source_linux_bridge.natmode_name,
                           "-j",
                           "MARK",
                           "--set-mark",
                           self.private_route_table
                           )
        if rc != 0:
            raise NetworkConfigurationError("Couldn't add iptable rule to tag outgoing packets with %s. Error: %s" %
                                            (self.private_route_table,
                                             str(out)))

    def add_iptables_rules_nat(self):
        '''
        Add iptables rules for the tcp and udp ports specified in network_config.yaml for the nat bridge
        (if it was successfully created)
        self.libvirt_network_name contains either dpbr_n_X or dpbr_docker_n_X
        '''

        if (not self.network_type == self.NETWORK_TYPE_NAT) and (not self.network_type == self.NETWORK_TYPE_NAT_DOCKER):
            log.debug("Not a NAT network. Skip adding port blocking rules.")
            return

        if (not self.source_linux_bridge) or (not self.libvirt_network_name):
            return

        log.debug("Adding iptables rules for network: %s, nat bridge: %s", self.name, self.libvirt_network_name)

        try:
            # Add the iptables rule if the nat bridge creation was successful
            out, rc = ipcmd("link", "show", self.libvirt_network_name)
            if rc != 0:
                log.debug("The nat bridge %s doesn't exist. Skip adding iptables rules.")
                return

            if (self.allowed_tcp_ports is not None) or (self.allowed_udp_ports is not None):
                log.debug("TCP ports to be allowed: %s, UDP ports to be allowed: %s", self.allowed_tcp_ports, self.allowed_udp_ports)
                self.add_block_all_ports_nat_bridge()
                self.add_allowed_ports_nat_bridge()
            elif self.blocked_tcp_ports or self.blocked_udp_ports:
                log.debug("TCP ports to be blocked: %s, UDP ports to be blocked: %s", self.blocked_tcp_ports, self.blocked_udp_ports)
                self.add_block_ports_nat_bridge()

        except Exception as ex:
            log.error("Failed to add iptables rules for nat bridge: %s. Cause: %s", self.libvirt_network_name, str(ex))

    def add_block_all_ports_nat_bridge(self):
        '''
        Block all ports for the specified nat bridge if it was successfully created
        self.libvirt_network_name contains either dpbr_n_X or dpbr_docker_n_X

        iptables -I INPUT -i dpbr_n_0 -s 192.168.10.0/27 -p icmp -j ACCEPT
        iptables -A INPUT -i dpbr_n_0 -j DROP

        iptables -I OUTPUT -o dpbr_n_0 -d 192.168.10.0/27 -p icmp -j ACCEPT
        iptables -A OUTPUT -o dpbr_n_0 -j DROP
        '''

        log.debug("Block all ports for nat bridge: %s", self.libvirt_network_name)

        try:
            # Delete the rule in case it already exists, ignore any errors
            out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name, "-j", "DROP")

            # Insert the iptables rule to INPUT table to drop all traffic
            out, rc = iptables("-A", "INPUT", "-i", self.libvirt_network_name, "-j", "DROP")

            if rc != 0:
                log.error("Failure to add iptables rule to INPUT table for nat bridge: %s. Error: %s" %
                                                (self.libvirt_network_name, str(out)))


            # Delete the rule in case it already exists, ignore any errors
            out, rc = iptables("-D", "OUTPUT", "-o", self.libvirt_network_name, "-j", "DROP")

            # Insert the iptables rule to OUTPUT table to drop all traffic
            out, rc = iptables("-A", "OUTPUT", "-o", self.libvirt_network_name, "-j", "DROP")

            if rc != 0:
                log.error("Failure to add iptables rule to OUTPUT table for nat bridge: %s. Error: %s" %
                                                (self.libvirt_network_name, str(out)))


            # Delete the rule in case it already exists, ignore errors
            out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                               "-s", self.nat_range_cidr, "-p", "icmp",
                               "-j", "ACCEPT")

            # Insert the rule to INPUT table to allow ping (icmp)
            out, rc = iptables("-I", "INPUT", "-i", self.libvirt_network_name,
                               "-s", self.nat_range_cidr, "-p", "icmp",
                               "-j", "ACCEPT")
            if rc != 0:
                log.error("Failure to add icmp iptables rule to INPUT table for nat bridge: %s. Error: %s" %
                                                (self.libvirt_network_name, str(out)))


            # Delete the rule in case it already exists, ignore errors
            out, rc = iptables("-D", "OUTPUT", "-o", self.libvirt_network_name,
                               "-d", self.nat_range_cidr, "-p", "icmp",
                               "-j", "ACCEPT")

            # Insert the rule to OUTPUT table to allow ping (icmp)
            out, rc = iptables("-I", "OUTPUT", "-o", self.libvirt_network_name,
                               "-d", self.nat_range_cidr, "-p", "icmp",
                               "-j", "ACCEPT")
            if rc != 0:
                log.error("Failure to add icmp iptables rule to OUTPUT table for nat bridge: %s. Error: %s" %
                                                (self.libvirt_network_name, str(out)))

        except Exception as ex:
            log.error("Failed to block all traffic for nat bridge: %s. Cause: %s", self.libvirt_network_name, str(ex))

    def add_allowed_ports_nat_bridge(self):
        '''
        Open the tcp and udp ports specified in network_config.yaml for the nat bridge 
        (if it was successfully created)
        self.libvirt_network_name contains either dpbr_n_X or dpbr_docker_n_X

        iptables -I INPUT -i dpbr_n_0 -s 192.168.10.0/27 -p tcp --dport 8098 -m state --state NEW,ESTABLISHED -j ACCEPT
        iptables -I OUTPUT -o dpbr_n_0 -d 192.168.10.0/27 -p tcp --sport 8098 -m state --state ESTABLISHED -j ACCEPT
        '''

        log.debug("TCP ports to be allowed: %s, UDP ports to be allowed: %s", self.allowed_tcp_ports, self.allowed_udp_ports)

        try:
            if self.allowed_tcp_ports:
                for port in self.allowed_tcp_ports:
                    log.debug("Adding tcp port %s allowing INPUT rule for nat bridge: %s", port, self.libvirt_network_name)

                    # Delete the iptables rule in case it already exists
                    out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                                       "-s", self.nat_range_cidr, "-p", "tcp", "--dport", str(port),
                                       "-m", "state", "--state", "NEW,ESTABLISHED",
                                       "-j", "ACCEPT")

                    # Insert the iptables rule
                    out, rc = iptables("-I", "INPUT", "-i", self.libvirt_network_name,
                                       "-s", self.nat_range_cidr, "-p", "tcp", "--dport", str(port),
                                       "-m", "state", "--state", "NEW,ESTABLISHED",
                                       "-j", "ACCEPT")
                    if rc != 0:
                        log.error("Failure to add tcp port %s allowing INPUT rule to iptables for nat bridge: %s. Error: %s" %
                                                        (port, self.libvirt_network_name, str(out)))


                    log.debug("Adding tcp port %s allowing OUTPUT rule for nat bridge: %s", port, self.libvirt_network_name)

                    # Delete the iptables rule in case it already exists
                    out, rc = iptables("-D", "OUTPUT", "-o", self.libvirt_network_name,
                                       "-d", self.nat_range_cidr, "-p", "tcp", "--sport", str(port),
                                       "-m", "state", "--state", "ESTABLISHED",
                                       "-j", "ACCEPT")

                    # Insert the iptables rule
                    out, rc = iptables("-I", "OUTPUT", "-o", self.libvirt_network_name,
                                       "-d", self.nat_range_cidr, "-p", "tcp", "--sport", str(port),
                                       "-m", "state", "--state", "ESTABLISHED",
                                       "-j", "ACCEPT")
                    if rc != 0:
                        log.error("Failure to add tcp port %s allowing OUTPUT rule to iptables for nat bridge: %s. Error: %s" %
                                                        (port, self.libvirt_network_name, str(out)))

            if self.allowed_udp_ports:
                for port in self.allowed_udp_ports:
                    log.debug("Adding udp port %s allowing INPUT rule for nat bridge: %s", port, self.libvirt_network_name)

                    # Delete the iptables rule in case it already exists
                    out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                                       "-s", self.nat_range_cidr, "-p", "udp", "--dport", str(port),
                                       "-m", "state", "--state", "NEW,ESTABLISHED",
                                       "-j", "ACCEPT")

                    # Insert the iptables rule
                    out, rc = iptables("-I", "INPUT", "-i", self.libvirt_network_name,
                                       "-s", self.nat_range_cidr, "-p", "udp", "--dport", str(port),
                                       "-m", "state", "--state", "NEW,ESTABLISHED",
                                       "-j", "ACCEPT")
                    if rc != 0:
                        log.error("Failure to add tcp port %s allowing INPUT rule to iptables for nat bridge: %s. Error: %s" %
                                                        (port, self.libvirt_network_name, str(out)))


                    log.debug("Adding tcp port %s allowing OUTPUT rule for nat bridge: %s", port, self.libvirt_network_name)

                    # Delete the iptables rule in case it already exists
                    out, rc = iptables("-D", "OUTPUT", "-o", self.libvirt_network_name,
                                       "-d", self.nat_range_cidr, "-p", "udp", "--sport", str(port),
                                       "-m", "state", "--state", "ESTABLISHED",
                                       "-j", "ACCEPT")

                    # Insert the iptables rule
                    out, rc = iptables("-I", "OUTPUT", "-o", self.libvirt_network_name,
                                       "-d", self.nat_range_cidr, "-p", "udp", "--sport", str(port),
                                       "-m", "state", "--state", "ESTABLISHED",
                                       "-j", "ACCEPT")
                    if rc != 0:
                        log.error("Failure to add udp port %s allowing OUTPUT rule to iptables for nat bridge: %s. Error: %s" %
                                                        (port, self.libvirt_network_name, str(out)))
        except Exception as ex:
            log.error("Failed to add port allowing rules for nat bridge: %s. Cause: %s", self.libvirt_network_name, str(ex))

    def add_block_ports_nat_bridge(self):
        '''
        Block the tcp and udp ports specified in network_config.yaml for the nat bridge
        (if it was successfully created)
        self.libvirt_network_name contains either dpbr_n_X or dpbr_docker_n_X
        '''

        try:
            for port in self.blocked_tcp_ports:
                log.debug("Adding tcp  %s blocking rule for nat bridge: %s", port, self.libvirt_network_name)

                # Delete the port blocking rule in case it already exists
                out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                                   "-p", "tcp", "--dport", str(port),
                                   "-j", "DROP")

                # Insert the port blocking rule
                out, rc = iptables("-I", "INPUT", "-i", self.libvirt_network_name,
                                   "-p", "tcp", "--dport", str(port),
                                   "-j", "DROP")

                if rc != 0:
                    log.error("Failure to add tcp port %s blocking rule to iptables for nat bridge: %s. Error: %s" %
                                                    (port, self.libvirt_network_name, str(out)))

            for port in self.blocked_udp_ports:
                log.debug("Adding udp port %s blocking rule for nat bridge: %s", port, self.libvirt_network_name)

                # Delete the port blocking rule in case it already exists
                out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                                   "-p", "udp", "--dport", str(port),
                                   "-j", "DROP")

                # Insert the port blocking rule
                out, rc = iptables("-I", "INPUT", "-i", self.libvirt_network_name,
                                   "-p", "udp", "--dport", port,
                                   "-j", "DROP")

                if rc != 0:
                    log.error("Failure to add udp port %s blocking rule to iptables for nat bridge: %s. Error: %s" %
                                                    (port, self.libvirt_network_name, str(out)))
        except Exception as ex:
            log.error("Failed to add port blocking rules for nat bridge: %s. Cause: %s", self.libvirt_network_name, str(ex))


    def remove_iptables_rules_nat(self):
        '''
        Remove the tcp and udp port rules specified in network_config.yaml from iptables for the nat bridge 
        (if it was successfully torn down)
        self.libvirt_network_name contains either dpbr_n_X or dpbr_docker_n_X
        '''

        if (not self.network_type == self.NETWORK_TYPE_NAT) and (not self.network_type == self.NETWORK_TYPE_NAT_DOCKER):
            log.debug("Not a NAT network. Skip removing port blocking rules.")
            return

        if (not self.source_linux_bridge) or (not self.libvirt_network_name):
            return

        log.debug("Removing iptables rules for network: %s, for nat bridge: %s", self.name, self.libvirt_network_name)

        try:
            # Remove the iptables rule if the nat bridge deletion was successful
            out, rc = ipcmd("link", "show", self.libvirt_network_name)
            if rc == 0:
                log.debug("The nat bridge %s still exists. Keeping the iptables rules.")
                return

            if (self.allowed_tcp_ports is not None) or (self.allowed_udp_ports is not None):
                log.debug("Remove allowed TCP ports: %s, remove allowed UDP ports: %s", self.allowed_tcp_ports, self.allowed_udp_ports)
                self.remove_block_all_ports_nat_bridge()
                self.remove_allowed_ports_nat_bridge()
            elif self.blocked_tcp_ports or self.blocked_udp_ports:
                log.debug("Remove blocked TCP ports: %s, Remove blocked UDP ports: %s", self.blocked_tcp_ports, self.blocked_udp_ports)
                self.remove_blocked_ports_nat_bridge()    

        except Exception as ex:
            log.error("Failed to remove iptables rules for nat bridge: %s. Cause: %s", self.libvirt_network_name, str(ex))


    def remove_block_all_ports_nat_bridge(self):
        '''
        Remove the iptables rules that block all ports for the specified nat bridge 
        (if it was successfully created)
        self.libvirt_network_name contains either dpbr_n_X or dpbr_docker_n_X

        iptables -D INPUT -i dpbr_n_0 -s 192.168.10.0/27 -p icmp -j ACCEPT
        iptables -D INPUT -i dpbr_n_0 -j DROP

        iptables -D OUTPUT -o dpbr_n_0 -d 192.168.10.0/27 -p icmp -j ACCEPT
        iptables -D OUTPUT -o dpbr_n_0 -j DROP
        '''

        log.debug("Remove blocking all ports for nat bridge: %s", self.libvirt_network_name)

        try:
            # Delete the iptables rule that blocks all traffic from INPUT table
            out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name, "-j", "DROP")

            if rc != 0:
                log.error("Failure to remove iptables rule from INPUT table for nat bridge: %s. Error: %s" %
                                                (self.libvirt_network_name, str(out)))


            # Delete the iptables rule that blocks all traffic from OUTPUT table
            out, rc = iptables("-D", "OUTPUT", "-o", self.libvirt_network_name, "-j", "DROP")

            if rc != 0:
                log.error("Failure to remove iptables rule from OUTPUT table for nat bridge: %s. Error: %s" %
                                                (self.libvirt_network_name, str(out)))


            # Delete the icmp iptables rule from the INPUT table
            out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                               "-s", self.nat_range_cidr, "-p", "icmp",
                               "-j", "ACCEPT")
            if rc != 0:
                log.error("Failure to remove icmp iptables rule from INPUT table for nat bridge: %s. Error: %s" %
                                                (self.libvirt_network_name, str(out)))


            # Delete the icmp iptables rule from the OUTPUT table
            out, rc = iptables("-D", "OUTPUT", "-o", self.libvirt_network_name,
                               "-d", self.nat_range_cidr, "-p", "icmp",
                               "-j", "ACCEPT")
            if rc != 0:
                log.error("Failure to remove icmp iptables rule from OUTPUT table for nat bridge: %s. Error: %s" %
                                                (self.libvirt_network_name, str(out)))

        except Exception as ex:
            log.error("Failed to remove iptables rules that block all traffic for nat bridge: %s. Cause: %s", self.libvirt_network_name, str(ex))


    def remove_allowed_ports_nat_bridge(self):
        '''
        Remove the iptables rules that open the tcp and udp ports specified in network_config.yaml for the nat bridge 
        (if it was successfully created)
        self.libvirt_network_name contains either dpbr_n_X or dpbr_docker_n_X

        iptables -D INPUT -i dpbr_n_0 -s 192.168.10.0/27 -p tcp --dport 8098 -m state --state NEW,ESTABLISHED -j ACCEPT
        iptables -D OUTPUT -o dpbr_n_0 -d 192.168.10.0/27 -p tcp --sport 8098 -m state --state ESTABLISHED -j ACCEPT
        '''

        log.debug("Remove TCP ports allowed: %s, Remove UDP ports allowed: %s", self.allowed_tcp_ports, self.allowed_udp_ports)

        try:
            if self.allowed_tcp_ports:
                for port in self.allowed_tcp_ports:
                    log.debug("Removing INPUT rule that allows tcp port %s for nat bridge: %s", port, self.libvirt_network_name)

                    # Delete the iptables rule from INPUT table
                    out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                                       "-s", self.nat_range_cidr, "-p", "tcp", "--dport", str(port),
                                       "-m", "state", "--state", "NEW,ESTABLISHED",
                                       "-j", "ACCEPT")
                    if rc != 0:
                        log.error("Failure to remove INPUT rule that allows tcp port %s for nat bridge: %s. Error: %s" %
                                                        (port, self.libvirt_network_name, str(out)))


                    log.debug("Removing OUTPUT rule that allows tcp port %s for nat bridge: %s", port, self.libvirt_network_name)

                    # Delete the iptables rule from OUTPUT table
                    out, rc = iptables("-D", "OUTPUT", "-o", self.libvirt_network_name,
                                       "-d", self.nat_range_cidr, "-p", "tcp", "--sport", str(port),
                                       "-m", "state", "--state", "ESTABLISHED",
                                       "-j", "ACCEPT")
                    if rc != 0:
                        log.error("Failure to remove OUTPUT rule that allows tcp port %s for nat bridge: %s. Error: %s" %
                                                        (port, self.libvirt_network_name, str(out)))

            if self.allowed_udp_ports:
                for port in self.allowed_udp_ports:
                    log.debug("Removing INPUT rule that allows udp port %s for nat bridge: %s", port, self.libvirt_network_name)

                    # Delete the iptables rule in case it already exists
                    out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                                       "-s", self.nat_range_cidr, "-p", "udp", "--dport", str(port),
                                       "-m", "state", "--state", "NEW,ESTABLISHED",
                                       "-j", "ACCEPT")
                    if rc != 0:
                        log.error("Failure to remove INPUT rule that allows udp port %s for nat bridge: %s. Error: %s" %
                                                        (port, self.libvirt_network_name, str(out)))


                    log.debug("Removing OUTPUT rule that allows udp port %s for nat bridge: %s", port, self.libvirt_network_name)

                    # Delete the iptables rule in case it already exists
                    out, rc = iptables("-D", "OUTPUT", "-o", self.libvirt_network_name,
                                       "-d", self.nat_range_cidr, "-p", "udp", "--sport", str(port),
                                       "-m", "state", "--state", "ESTABLISHED",
                                       "-j", "ACCEPT")
                    if rc != 0:
                        log.error("Failure to remove OUTPUT rule that allows udp port %s for nat bridge: %s. Error: %s" %
                                                        (port, self.libvirt_network_name, str(out)))
        except Exception as ex:
            log.error("Failed to remove port allowing rules for nat bridge: %s. Cause: %s", self.libvirt_network_name, str(ex))

    def remove_blocked_ports_nat_bridge(self):
        '''
        Remove the tcp and udp port rules specified in network_config.yaml from iptables for the nat bridge 
        (if it was successfully torn down)
        self.libvirt_network_name contains either dpbr_n_X or dpbr_docker_n_X
        '''

        try:
            for port in self.blocked_tcp_ports:
                log.debug("Removing tcp port %s blocking rule for nat bridge: %s", port, self.libvirt_network_name)

                # Delete the port blocking rule for tcp from INPUT table
                out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                                   "-p", "tcp", "--dport", str(port),
                                   "-j", "DROP")
                if rc != 0:
                    log.error("Failure to remove tcp port %s blocking rule from iptables for nat bridge: %s. Error: %s" %
                                                    (port, self.libvirt_network_name, str(out)))

            for port in self.blocked_udp_ports:
                log.debug("Removing udp port %s blocking rule for nat bridge: %s", port, self.libvirt_network_name)

                # Delete the port blocking rule for udp from INPUT table
                out, rc = iptables("-D", "INPUT", "-i", self.libvirt_network_name,
                                   "-p", "udp", "--dport", str(port),
                                   "-j", "DROP")
                if rc != 0:
                    log.error("Failure to remove udp port %s blocking rule from iptables for nat bridge: %s. Error: %s" %
                                                    (port, self.libvirt_network_name, str(out)))
        except Exception as ex:
            log.error("Failed to remove port blocking rules for nat bridge: %s. Cause: %s", self.libvirt_network_name, str(ex))


    def setup_nat_docker_network(self, reconcile=False):
        """
        Setup a nat docker network for the container and attach it to hosting bridge.
        :return:
        """
        # Handle container specific initialization
        if not "docker" in self.container_info:
            raise NotImplementedError("Not sure what do with these containers : %s" % self.container_info.keys())
        
        log.debug("Setting up nat network for docker containers..")
        if not self.docker_api_client:
            base_url = Utils.getSystemConfigValue("docker-container", "docker_base_url")
            api_version = Utils.getSystemConfigValue("docker-container", "docker_api_version")
            timeout = Utils.getSystemConfigValue("docker-container", "docker_timeout_seconds", parse_as="int")
            use_tls = Utils.getSystemConfigValue("docker-container", "docker_use_tls", parse_as="bool")
            self.docker_api_client = Utils.get_docker_api_client(base_url, api_version, timeout, use_tls)
            log.debug("docker api client config headers = %s" % self.docker_api_client.headers)
        docker_network_name = self.source_linux_bridge.natmode_docker_name
        log.debug("Docker network name: %s" % docker_network_name)
        if self.docker_api_client:
            #Verify if network already exists
            docker_network_filter = [docker_network_name]
            docker_net_list = self.docker_api_client.networks(names=docker_network_filter)
            log.debug("docker networks api return:%s" % docker_net_list)
            docker_net=None
            for d_net in docker_net_list:
                if d_net['Name'] == docker_network_name:
                    docker_net=d_net
                    reconcile = True
                    break
            if not reconcile and docker_net:
                #Remove the docker network
                log.debug("Going to remove docker network:%s" % docker_net['Name'])
                try:
                    self.docker_api_client.remove_network(docker_network_name)
                    docker_net=None
                except Exception as ex:
                    log.exception("Failed to remove docker network: %s" % docker_network_name)
                    log.debug("Trying to reuse the same network")
                
            if docker_net:
                #Already docker network exists
                self.add_iptables_rules_nat()
                return
            option_kwargs = {
                            "com.docker.network.bridge.name" : docker_network_name
                           }
            import netaddr
            import docker
            #For docker ip start range should include gateway ip in the range
            ip_start = str(IPAddress(self.ip_start) - 1)
            subnet_bits = IPAddress(self.subnet_mask).netmask_bits()
            subnet=ip_start+"/"+str(subnet_bits)
            log.debug("SUBNET:%s" % subnet)

            ipam_pool = docker.types.IPAMPool(
                        subnet=subnet,
                        gateway=self.gateway_ip
                        )
            log.debug("IPAM_POOL:%s" % ipam_pool)
            ipam_config = docker.types.IPAMConfig(pool_configs=[ipam_pool])      
 
            self.docker_api_client.create_network(docker_network_name, driver="bridge",
                         options=option_kwargs, ipam=ipam_config, 
                         enable_ipv6=False)

            self.add_iptables_rules_nat()

    def setup_nat_network(self, reconcile=False):
        """
        Setup a nat network for the container and attach it to hosting bridge.
        Ensure that dpbr_n_vpgno is created via the container that this bridge is associated with (self.container)

        Assign IP address (gateway_ip) dpbr_n_* bridge.

        PS: dpbr -> Data path bridge
        :return:
        """

        # Handle container specific initialization
        if "libvirt" in self.container_info:

            log.debug("Setting up nat network for libvirt containers..")

            libvirt_network_name = self.source_linux_bridge.natmode_name
            log.debug("Libvirt network name: %s" % libvirt_network_name)

            import libvirt
            c = None

            try:
                if self.libvirt_conn_hdl:
                    c = self.libvirt_conn_hdl
                else:
                    c = libvirt.open()
                    self.libvirt_conn_hdl = c
                log.debug("Looking up libvirt network domains for %s", libvirt_network_name)
                try:
                    nw = c.networkLookupByName(libvirt_network_name)
                    log.debug("Pre existing network found %s", libvirt_network_name)
                    if not reconcile:
                        log.debug("Will be tearing it down before setting it up with correct configuration..")
                        if nw.isActive():
                            nw.destroy()
                            log.debug("Stopped network %s", nw.name())
                        nw.undefine()
                        log.debug("Undefined network %s", nw.name())
                        nw = None
                except libvirt.libvirtError:
                    nw = None

                # At this point, even if a network existed, it should have been torn down.
                # Setup afresh
                log.debug("Setting up network domain by name : %s", libvirt_network_name)

                # CSCux44209
                # When libvirt network is updated to have a ip-dhcp-host static entry for this app
                # libvirt re-issues iptable rules that messes up the order in the iptable chain
                # Due to this, a catch all rule gets inserted immediately after the latest portfowarding rules
                # that matches any app's traffic that were previously installed.
                # So, none of the earlier apps will be able to get traffic.
                #
                # To solve this, the approach is to get a block of mac addresses for a nat network,
                # setup ip-dhcp-host static entries ONLY once (at the time of network setup)
                # Which removed the need for it to be done every time app_setup_hook is called (during activation)

                import netaddr
                iprange = netaddr.IPRange(self.ip_start, self.ip_end)

                log.debug("Setting up static DHCP entries for pre allocated mac address blocks..")
                cntr = 0
                ip_dhcp_static_entries = []
                for mac_address, i in zip(self.mac_address_block, iprange):
                    ipaddress = str(i)

                    # Configure DHCP Static mapping
                    updatexml = self.LIBVIRT_IP_DHCP_HOST_TEMPLATE.format(MAC_ADDRESS=mac_address,
                                                                          IP_ADDRESS=ipaddress,
                                                                          DOMAIN_NAME="%s-%s" % (self.name, cntr))
                    log.debug(updatexml)
                    cntr = cntr + 1
                    ip_dhcp_static_entries.append(updatexml)
                    # Also add this to mac_ip_map
                    self.mac_ip_map[mac_address] = ipaddress

                if not ip_dhcp_static_entries:
                    # If there are no entries, just set it to an empty string
                    ip_dhcp_static_entries = ""

                dns_servers = self.lease_info["dns"]
                if "," in dns_servers:
                    dns_servers = dns_servers.split(",")[0]

                xml_string = self.LIBVIRT_NAT_TEMPLATE.format(BRIDGE_NAME=self.source_linux_bridge.natmode_name,
                                                              NETWORK_NAME=libvirt_network_name,
                                                              FORWARDING_DEVICE=self.source_linux_bridge.interface,
                                                              GATEWAY_IP=self.gateway_ip,
                                                              NETMASK=self.subnet_mask,
                                                              IP_START=self.ip_start,
                                                              IP_END=self.ip_end,
                                                              DNS_SERVER=dns_servers,
                                                              IP_DHCP_STATIC_ENTRIES="".join(ip_dhcp_static_entries)
                                                              )
                log.debug("Sending XML : %s", xml_string)
                try:
                    if nw is None:
                        nw = c.networkDefineXML(xml_string)
                        if not nw.isActive():
                            nw.create()
                        log.debug("Defined and started network : %s", nw.name())
                    else:
                        if not nw.isActive():
                            nw.create()
                except libvirt.libvirtError as le:
                    raise le

                self.add_iptables_rules_nat()

                #nw = c.networkLookupByName(libvirt_network_name)
                self.uuid = nw.UUIDString()
                # Store the network created here
                self.container_network["libvirt"] = libvirt_network_name

                self.set_mtu()

            except Exception as ex:
                log.error("Network: Error while setting up the NAT network. Cause: %s"%ex.message)

        else:
            #docker network create --subnet=12.0.0.34/28 mynet
            raise NotImplementedError("Not sure what do with these containers : %s" % self.container_info.keys())



class NetworkController(CAFAbstractService):
    """
    This class encapsulated networking needs on a given platform. It understands the network configuration present in
    system_config.ini, sets up networking on hosting interfaces and manages them.

    It also implements handlers for changing some of the settings at runtime via RESTful APIs.
    """
    __singleton = None # the one, true Singleton

    # Maintains a list of networks created/available on the fog node
    NETWORKS = {}

    # Maintains a list of hosting bridges and their configuration as provided in the config file
    HOSTING_BRIDGES = {}
     
    auto_br_libvirt_prefix = "dpbr_"

    dhcp_filter_template_dict = {
        'block_dhcp_ipv4':
            """
                <filter name='block_dhcp_ipv4' chain='ipv4'>
                  <rule action='drop' direction='out'>
                    <ip protocol='udp' srcportstart='68' dstportstart='67'/>
                  </rule>
                  <rule action='drop' direction='in'>
                    <ip protocol='udp' srcportstart='67' dstportstart='68'/>
                  </rule>
                </filter>
            """,
        'block_dhcp_ipv6':
            """
                <filter name='block_dhcp_ipv6' chain='ipv6'>
                  <rule action='drop' direction='out'>
                    <ip protocol='udp' srcportstart='546' dstportstart='547'/>
                  </rule>
                  <rule action='drop' direction='in'>
                    <ip protocol='udp' srcportstart='547' dstportstart='546'/>
                  </rule>
                </filter>
            """,
        'block_dhcp_ipv4_ipv6':
            """
                <filter name='block_dhcp_ipv4_ipv6' >
                  <filterref filter='block_dhcp_ipv4'/>
                  <filterref filter='block_dhcp_ipv6'/>
                </filter>
    
            """
    }

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(NetworkController, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, params, repofolder, same_dev_flag = True, container_info={"libvirt": {}}):
        """
        same_dev_flag:  True if NC is being initialised from the configuration
        which was last stored on the same device
        """
        # Config is a dictionary representing config/network_config.yaml.
        self.name = params.name
        self._config = params.config
        self._config_file = params.config_file
        self.repofolder = repofolder
        self.uuid_store_file = os.path.join(repofolder, UUID_STORE_FILE)
        self.uuid_map = {}
        if not same_dev_flag:
            log.info("Going to delete network configurations under repo network")
            self._del_old_devnw(repofolder)
        self.libvirt_conn = None
        self._load_data()

        self.base_url = Utils.getSystemConfigValue("docker-container", "docker_base_url")
        self.api_version = Utils.getSystemConfigValue("docker-container", "docker_api_version")
        self.timeout = Utils.getSystemConfigValue("docker-container", "docker_timeout_seconds", parse_as="int")
        self.use_tls = Utils.getSystemConfigValue("docker-container", "docker_use_tls", parse_as="bool")
        self.docker_api_client = None
 
        self.container_info = container_info
        self.default_bridge = None
        self.default_network = None
        self.default_docker_nat_network = None
        self.network_name_prefix = None
        self.enabled = self._config.get("enabled", True)
        self.enable_dhcp_filters = self._config.get("enable_dhcp_filters", False)
        self.local_mac_registry = self._config.get("local_mac_registry", True)
        self._host_mode = self._config.get("host_mode", False)
        self.tcp_pat_port_range = self._config.get("tcp_pat_port_range", None)
        self.udp_pat_port_range = self._config.get("udp_pat_port_range", None)
        self.tcp_blocked_pat_ports = self._config.get("tcp_blocked_pat_ports", [])
        self.udp_blocked_pat_ports = self._config.get("udp_blocked_pat_ports", [])
        self.internal_blocked_ports = self._config.get("internal_blocked_ports", [])
        
        NetworkController.svcbr_status_file = self._config.get("svcbr_status_file", None)
        NetworkController.bridge_properties_script = self._config.get("bridge_properties_script", None)

        self.network_creation_api = self._config.get("network_creation_api", True)
        self.vpg_inf_file = self._config.get("network_creation_inf_file", "/etc/platform/vpg_interfaces")

        if NetworkController.bridge_properties_script:
            NetworkController.bridge_properties_script = os.path.join(Utils.getScriptsFolder(),
                                                         "pdhooks",
                                                         NetworkController.bridge_properties_script)
        
        # VPG interfaces on the platform that can be enslaved
        self.available_phys_infs = Utils.getlines_from_file(self.vpg_inf_file)
        
        # Keeps track of ip address pool used by all configured NAT networks
        # netaddr.IPSet doesn't keep each IP address one-by-one. It saves the cidr subnets 
        # and internally merges them into larger subnets as appropriate
        self.used_nat_range = IPSet()
        self.suggested_nat_range_start = self._config.get("suggested_nat_range_start", None)
        
        self.mirror_mode_supported = self._config.get("mirror_mode_supported", False)
        self.update_network_mtu = self._config.get("update_network_mtu", False)
        self._default_vlan_hosting_bridge = self._config.get("default_vlan_hosting_bridge", None)
        
        self.auto_bridge_mode = self._config.get("auto_bridge_mode", {})
        self.auto_bridge_mode_prefix = self.auto_bridge_mode.get("phys_interface_prefix", None)
        self.auto_bridge_mode_enabled = self.auto_bridge_mode.get("enabled", False)

        self.repofolder = repofolder

        if self.local_mac_registry:
            self.enable_local_mac_registry()
        else:
            self.disable_local_mac_registry()

        self._port_registry = PortRegistry(self.repofolder, self.tcp_pat_port_range, self.udp_pat_port_range,
                                    self.tcp_blocked_pat_ports, self.udp_blocked_pat_ports)
        self.clear_app_port_entry = self._port_registry.clear_app_port_entry

        # Network filters to control the traffic in container
        self.dhcp_filters_defined = []
        self._running = False
        self._network_status = False

    @property
    def suggested_nat_range(self):
        return get_suggested_nat_range(self.suggested_nat_range_start, self.used_nat_range)

    @property
    def host_mode(self):
        return self._host_mode
        
    @property
    def network_status(self):
        return self._network_status

    @classmethod
    def next_available_vpgno(cls):

        used_vpgno_list = cls.used_vpgno_list()
        for vpgno in range(0, 100):
            if vpgno not in used_vpgno_list:
                log.debug("Generated vpgno: %s", vpgno)
                return str(vpgno)

        log.error("No available vpgno left.")
        return None

    @classmethod
    def used_vpgno_list(cls):

        used_vpgno_list = []
        for hb in cls.HOSTING_BRIDGES.values():
            if hb.vpgno:
                used_vpgno_list.append(int(hb.vpgno))

        log.debug("Used vpgno list: %s", used_vpgno_list)
        return used_vpgno_list

    def _del_old_devnw(self, network_repo_folder):
        for item in os.listdir(network_repo_folder):
            # Do not delete the nconf 
            # Delete everything else
            path, nconf_name = os.path.split(self._config_file)
            if item == nconf_name:
                continue
            n_res = os.path.join(network_repo_folder, item)
            log.info("Deleting: %s" % n_res)
            if os.path.isdir(n_res):
                shutil.rmtree(n_res, ignore_errors=True)
            else:
                os.remove(n_res)

    def _get_port_mapping(self, appid, interface_name, network_name, ports, req_port_map, port_map_bridge=False):
        """
        Return port mapping for the asked set of ports.
        :param appid:
        :param network_name:
        :param ports:
        :return:
        """
        nw = self.NETWORKS.get(network_name)
        if nw:
            #iox-natx and iox-nat_dockerx should use same pool
            if nw.network_type ==  Network.NETWORK_TYPE_NAT_DOCKER:
                log.debug("Network: %s need to use iox-natx converting" % network_name)
                regex = "(?P<net_prefix>\w+)-(?P<net_type>\w+)(?P<net_num>\d)"
                m = re.compile(regex)
                m = m.match(network_name)
                net_prefix = m.group("net_prefix")
                net_type =  m.group("net_type")
                net_num = m.group("net_num")
                if net_type == Network.NETWORK_TYPE_NAT_DOCKER:
                    net_type = Network.NETWORK_TYPE_NAT
                    network_name = net_prefix + "-" + net_type + str(net_num)
                    log.debug("Updated nat network name: %s" % network_name)

            return self._port_registry.get_mapping(appid, interface_name, nw.network_type, ports, req_port_map, network_name, port_map_bridge)

        return None


    def _remove_port_mapping(self, appid, interface_name, network_name):

        nw = self.NETWORKS.get(network_name)
        if nw:
            self._port_registry.remove_mapping(appid, interface_name, nw.network_type)
        return None


    def _load_data(self):
        """
        Load any changed config. Strategy is to merge the saved config into the passed config.
        :return:
        """
        lc = None
        path, nconf_name = os.path.split(self._config_file)
        # Just to make backward compatibility
        if nconf_name:
            old_config = os.path.join(self.repofolder, nconf_name)
            if os.path.isfile(old_config):
                if os.stat(old_config).st_size == 0:
                    log.error("File %s is blank, delete the file",  old_config)
                    # remove the empty nconf file.
                    # CAF will retrieve the default network configuration from network_config YAML.
                else:
                    try:
                        with open(old_config, "r") as f:
                            try:
                                lc = json.load(f)
                            except ValueError as e:
                                log.error("Failed to load network configuration stored in %s, delete the file", self._config_file)
                                lc = None
                    except Exception as ex:
                        log.error("Failed to load network configurtion:%s. Will reset." % str(ex))
                        lc=None

                os.remove(old_config)

            if lc:
                try:
                    #self._config.update(lc)
                    #self._config = lc
                    self._config = Utils.merge_configs(lc, self._config)
                    self._save_data()
                    log.debug("Updated network configuration from stored file %s", old_config)
                except Exception as ex:
                    log.error("Failed to save network configuration: %s" % str(ex))

        if os.path.isfile(self.uuid_store_file):
            try:
                with open(self.uuid_store_file, "r") as f:
                    self.uuid_map = json.loads(f.read())
            except Exception as ex:
                log.error("Failed to load network %s. Error:%s" % (self.uuid_store_file, str(ex)))
                self.uuid_map = {}


    def _save_data(self):
        """
        Save config file to disk. Default location is repofolder/.nconf. Will be in json format
        :return:
        """
        #s = json.dumps(self._config)
        try:
            with open(self._config_file, "w", 0) as f:
                f.write(yaml.dump(self._config))
                log.debug("Saved network configuration to %s", self._config_file)
        except Exception as ex:
            log.error("Failed to save network configuration:%s. Error: %s" % 
                            (self._config_file, str(ex)))
            if os.path.exists(self._config_file):
                os.remove(self._config_file)

    @property
    def config(self):
        return self._config


    @config.setter
    def config(self, value):
        # Currently only honor the following settings. The below settings do not require a CAF/Network Controller restart
        # Going forward, accept full config changes that may need NC/CAF restart.
        hm = None
        lmr = None

        if "host_mode" in value:
            hm = value.get("host_mode")
            if isinstance(hm, basestring):
                hm = ast.literal_eval(value.get("host_mode"))

        if "local_mac_registry" in value:
            lmr = value.get("local_mac_registry")
            if isinstance(lmr, basestring):
                lmr = ast.literal_eval(value.get("local_mac_registry"))

        #if "enabled" in value:
        #    enabled = value.get("enabled")
        #    if isinstance(enabled, bool):
        #        self.enabled = enabled
        #        self._config["enabled"] = self.enabled

        if hm is not None:
            self._host_mode = hm
            log.debug("Setting host_mode to %s", hm)

        if lmr is not None:
            self.local_mac_registry = lmr
            log.debug("Setting local mac registry to %s", lmr)
            if self.local_mac_registry:
                self.enable_local_mac_registry()
            else:
                self.disable_local_mac_registry()

        # Also update the config instance
        self._config["host_mode"] = self._host_mode
        self._config["local_mac_registry"] = self.local_mac_registry

        # Persist for future restarts
        self._save_data()
    """
    def set_config(self, config):
        #if self.enabled:
        for name, nw in self.NETWORKS.iteritems():
            if nw.app_ip_map != {}:
                log.error("To shutdown the service, there will be no App/Service should be attached to any network!")
                raise ValueError("To shutdown the service, there should be no App/Service should be attached to any network!")
        try:
            if self.is_running:
                self.stop()
        except Exception as ex:
            log.exception("Network service teardown failed, with reason: %s"%str(ex))
            raise Exception("Network service teardown failed, with reason: %s"%str(ex))
        if self.validate_config(config):
            self._update_config(config)
        try:
            if self.config.get("enabled", None):
                self.start()
            else:
                log.debug("Network service is disabled as part of new config update!")
        except Exception as ex:
            log.exception("Error while setting up the Network service with new config %s, cause: %s"%(config, str(ex)))
            self.stop()
            raise Exception("Error while setting up the Network service with new config %s, cause: %s"%(config, str(ex)))
        #else:
        #    log.error("To update the config Network service needs to be enabled!")
        #    raise ValueError("To update the config Network service needs to be enabled!")
    """
    def _update_config(self, config):
        self.config = config
        self._save_data()

    def validate_config(self, config):
        return True

    def get_config(self):
        return self.config

    @property
    def macregistry(self):
        return self._macregistry

    @property
    def portregistry(self):
        return self._port_registry

    @property
    def default_vlan_hosting_bridge(self):
        return self._default_vlan_hosting_bridge

    def enable_local_mac_registry(self):
        log.debug("Enabling local mac registry..")
        self._macregistry = MacRegistry(self.repofolder)
        self.get_mac_address = self._macregistry.get_mac_address
        self.set_mac_address = self._macregistry.set_mac_address
        self.remove_mac_address = self._macregistry.remove_mac_address
        self.clear_app_mac_entry = self._macregistry.clear_app_mac_entry

    def disable_local_mac_registry(self):
        log.debug("Disabling local mac registry")
        log.debug("Local mac registry disabled!")
        self._macregistry = None
        nonefunc = lambda *args: None
        self.get_mac_address = nonefunc
        self.set_mac_address = nonefunc
        self.remove_mac_address = nonefunc
        self.clear_app_mac_entry = nonefunc


    def get_mac_registry(self):
        if self._macregistry:
            return self._macregistry.serialize()
        else:
            return {"error": "Local Mac registry service is disabled!"}

    def get_port_registry(self):
        if self._port_registry:
            return self._port_registry.serialize()

    def get_dhcp_ipv4_filter(self):
        if 'block_dhcp_ipv4' in self.dhcp_filters_defined:
            return 'block_dhcp_ipv4'
        return None

    def get_dhcp_ipv6_filter(self):
        if 'block_dhcp_ipv6' in self.dhcp_filters_defined:
            return 'block_dhcp_ipv6'
        return None
    
    def get_dhcp_ipv4_ipv6_filter(self):
        if 'block_dhcp_ipv4_ipv6' in self.dhcp_filters_defined:
            return 'block_dhcp_ipv4_ipv6'
        return None
    
    def setup_dhcp_filters(self):
        """
        Will setup the all network filters.
        """
        log.debug("Setting up the DHCP filters!")
        if 'libvirt' in self.container_info:
            if self.libvirt_conn:
                import libvirt
                c = self.libvirt_conn
            else:
                import libvirt
                c = libvirt.open()
                self.libvirt_conn = c
            for dhcp_filter_name in self.dhcp_filter_template_dict.keys():
                try:
                    nwf = None
                    try:
                        if self.uuid_map.get(dhcp_filter_name):
                            nwf = c.nwfilterLookupByUUIDString(self.uuid_map.get(dhcp_filter_name))
                    except libvirt.libvirtError:
                        self.uuid_map.pop(dhcp_filter_name, "")
                        log.debug("Seems like DHCP filter %s is not defined, so defining it"%dhcp_filter_name)
                    if nwf:
                        log.debug("Dhcp filter %s is already defined so not doing anything"%dhcp_filter_name)
                    else:
                        c.nwfilterDefineXML(self.dhcp_filter_template_dict[dhcp_filter_name])
                    self.uuid_map[dhcp_filter_name] = c.nwfilterLookupByName(dhcp_filter_name).UUIDString()
                    self.dhcp_filters_defined.append(dhcp_filter_name)
                except Exception as ex:
                    log.info("Error while defining the DHCP filters. Cause: %s"%ex.message)

    def teardown_dhcp_filters(self):
        """
        Will undefine the all filters defined as part of setup - self.filters
        """
        log.debug("Tearing down the DHCP filters!")
        if 'libvirt' in self.container_info:
            if self.libvirt_conn:
                import libvirt
                c = self.libvirt_conn
            else:
                import libvirt
                c = libvirt.open()
                self.libvirt_conn = c
            for dhcp_filter_name in self.dhcp_filters_defined:
                try:
                    nwf = c.nwfilterLookupByName(dhcp_filter_name)
                    log.debug("DHCP filter %s is defined so tearing it down"%dhcp_filter_name)
                    nwf.undefine()
                except libvirt.libvirtError:
                    log.debug("Seems like DHCP filter %s is not defined, so nothing to teardown"%dhcp_filter_name)
                except Exception as ex:
                    log.info("Error while tearing down the DHCP filters. Cause: %s"%ex.message)

    def set_mtu(self, networkInfo):
        """
        For the CAF networks that the app interfaces are connected to (passed in networkInfo),
        call set_mtu() to make sure the mtu is retained for those bridges
        """

        # MTU processing is only done if platform wants it in network_config.yaml
        if not self.update_network_mtu:
            return

        if not networkInfo:
            return

        for inf in networkInfo:
            if "network_name" in networkInfo[inf]:
                if networkInfo[inf]["network_name"] in self.NETWORKS:
                    self.NETWORKS[networkInfo[inf]["network_name"]].set_mtu()


    def _process_pd_status_file(self, hbs):
        """
        Platform creates default bridges in network_config before CAF starts
        Parameters such as vlan_id or ip address dhcp/static is configurable and unknown to CAF
        Platform passes this information to CAF by creating a status file before starting CAF
        Status file is configurable in network_config (svcbr_status_file)
        If bridge creation was not successful CAF won't start
        Example status file:
        hosting_bridges:
          svcbr_0:
            vlan_id: 15
            bridge_ip:
                mode: static
                ip: 192.168.0.16
                netmask: 255.255.255.0
            # or DHCP
            # bridge_ip:
            #     mode: dhcp    
            #     dhcp_lease_file: /etc/platform/dhcp/svcbr_0.leases
            status_code: 0
            message: Successfully created        
        """
    
        status = get_pd_status(NetworkController.svcbr_status_file, False)
        if not status:
            log.debug("Bad or no platform status file for Networking: %s" % NetworkController.svcbr_status_file)
            return
            
        log.debug("Platform status passed for networking: %s" % status)
        
        # Go through platform status
        for bridge_id in status.keys():
        
            status_code = status[bridge_id].get("status_code")
            message = status[bridge_id].get("message", "")
            
            if status_code != None and int(status_code) != 0:
                raise NetworkConfigurationError("Platform Network config error: %s" % message)
        
            bridge_settings = hbs.get(bridge_id)
            if not bridge_settings: 
                continue
                
            if bridge_settings.get("dynamically_created"):
                continue
        
            vlan_id = status[bridge_id].get("vlan_id")
            if vlan_id:
                bridge_settings["vlan_id"] = str(vlan_id)
                
            returned_bridge_ip = status[bridge_id]["bridge_ip"]
            
            if returned_bridge_ip["mode"] == HostingBridge.DHCP:
                bridge_ip = {"mode": HostingBridge.DHCP}
                bridge_settings["bridge_ip"] = bridge_ip
                bridge_settings["dhcp_lease_file"] = returned_bridge_ip.get("dhcp_lease_file")
                    
            elif returned_bridge_ip["mode"] == HostingBridge.STATIC:
                bridge_ip = {"mode": HostingBridge.STATIC}
                bridge_ip["ip"] = returned_bridge_ip["ip"]
                bridge_ip["subnet_mask"] = returned_bridge_ip["netmask"]
                
                bridge_gw_ip = returned_bridge_ip.get("gateway_ip")
                if bridge_gw_ip:
                    bridge_ip["bridge_gw_ip"] = bridge_gw_ip
                    
                domain = returned_bridge_ip.get("domain")
                if domain:
                    bridge_ip["domain"] = domain

                dns = returned_bridge_ip.get("dns")
                if dns:
                    bridge_ip["dns"] = dns
                
                bridge_settings["bridge_ip"] = bridge_ip
                bridge_settings["dhcp_lease_file"] = None
                
            elif returned_bridge_ip["mode"] == HostingBridge.NO_IP_ADDRESS:
                bridge_ip = {"mode": HostingBridge.NO_IP_ADDRESS}
                bridge_settings["bridge_ip"] = bridge_ip
                bridge_settings["dhcp_lease_file"] = None
        
        # Persist the data
        self._save_data()
        

    def start(self):
        """
        Parse the config file and create logical networks
        :return:
        """
        if not self.enabled:
            return
            
        hbs = self.config.get("hosting_bridges")
        if not hbs:
            hbs = {}
        self.default_bridge = self.config.get("default_bridge")
        self.tcp_pat_port_range = self.config["tcp_pat_port_range"]
        self.udp_pat_port_range = self.config["udp_pat_port_range"]

        self.network_name_prefix = self.config["network_name_prefix"]

        if self.default_bridge and self.default_bridge not in hbs:
            raise NetworkConfigurationError("Configured default bridge %s not present "
                                            "in the list of hosting bridges : %s" % (self.default_bridge,
                                                                                     str(hbs.keys())))
            
        self._process_pd_status_file(hbs)
        
        # Now start creating networks as configured for each bridge
        for i, bridge_id in enumerate(hbs):
            bridge_settings = hbs[bridge_id]
            
            self._validate_phys_interface(bridge_settings)
 
            if Network.NETWORK_TYPE_NAT in bridge_settings["supported_modes"]:
                nat_info = bridge_settings[Network.NETWORK_TYPE_NAT]
                self._validate_nat_ip_range(nat_info)
                
            if Network.NETWORK_TYPE_NAT_DOCKER in bridge_settings["supported_modes"]:
                docker_nat_info = bridge_settings[Network.NETWORK_TYPE_NAT_DOCKER]
                self._validate_nat_ip_range(docker_nat_info)
            # Create a logical hosting bridge and store it
            hb = HostingBridge(bridge_id, bridge_settings)
            try:
                hb.setup()
            except Exception as ex:
                if hb.required:
                    raise
                
                # Error setting up bridge, but it is not required
                # Log errors, attempt teardown (best effort) and continue
                log.warning("Could not set up optional hosting bridge: %s" % str(ex))
                try:
                    hb.teardown()
                except:
                    log.warning("Could not teardown hosting bridge: %s" % str(ex))
                continue
                
            self.HOSTING_BRIDGES[bridge_id] = hb

            if bridge_id == self.default_vlan_hosting_bridge:
                hb.vlan_hosting = True

            try:
                # For each of the supported mode, create a network
                self._create_logical_networks(hb)
            except Exception as ex:
                log.error("Error while creating logical networks: %s" % str(ex))
                hb.teardown()
                self.HOSTING_BRIDGES.pop(bridge_id)
                raise
                
            if hb.generated_mac_addr and self._macregistry:
                self._macregistry.set_mac_address_used(hb.generated_mac_addr)

        # Define the DHCP filters
        if self.enable_dhcp_filters:
            self.setup_dhcp_filters()
                 
        #Set up iptable rules to deny ios access to the apps
        self.setup_iptables()

        # Persist the data
        self._save_data()

        # persist UUID's
        self._save_uuid_map()
        self._running = True                 
        self._network_status = True

    def setup_iptables(self):
        """
        Creates the iptables rule to deny access to ios for containers
        """
        try:
            log.debug("Setting up ios deny iptable rules")
            default_bridge = self.get_default_bridge()
            host_bridge = self.get_hosting_bridge(default_bridge)

            #Read IPV4 address. 
            #ipv4host = host_bridge.get_bridge_ipv4_address()
            #Read IPV6 address 
            #ipv6host = host_bridge.get_bridge_ipv6_address()

            if host_bridge is not None:
                br_intf = host_bridge.interface

            self.clear_iptables(log_level="DEBUG")
            for int_block_port in self.internal_blocked_ports:

                log.debug("Add the iptable rule for denying conainer access to ios")
                out, rc = iptables("-I",
                               "FORWARD",
                               "1",
                               "-p",
                               "tcp",
                               "--destination-port",
                                "%d" % int_block_port,
                               "-j",
                               "DROP"
                               )
                if rc != 0:
                    log.error("Error Adding FORWARD/DROP ios rule. Error: %s" % str(out))

                out, rc = ip6tables("-I",
                               "FORWARD",
                               "1",
                               "-p",
                               "tcp",
                               "--destination-port",
                                "%d" % int_block_port,
                               "-j",
                               "DROP"
                               )
                if rc != 0:
                    log.error("Error Adding FORWARD/DROP ios rule. Error: %s" % str(out))
		
                if br_intf is not None:
                    out, rc = iptables("-I",
                               "FORWARD",
                               "1",
                               "-p",
                               "tcp",
                               "--destination-port",
                                "%d" % int_block_port,
                               "-i",
                               br_intf,
                               "-j",
                               "ACCEPT"
                               )
                    if rc != 0:
                        log.error("Error adding FORWARD/ACCEPT ios rule. Error: %s" % str(out))

                    #if ipv6host:
                    out, rc = ip6tables("-I",
                               "FORWARD",
                               "1",
                               "-p",
                               "tcp",
                               "--destination-port",
                                "%d" % int_block_port,
                               "-i",
                               br_intf,
                               "-j",
                               "ACCEPT"
                               )
                    if rc != 0:
                        log.error("Error adding FORWARD/ACCEPT ios rule. Error: %s" % str(out))
        except Exception as ex:
            log.exception("Error in creating iptable rules")

    def clear_iptables(self, log_level="ERROR"):
        """
        Deletes the iptables rule to deny access to ios for containers
        """
        try:
            default_bridge = self.get_default_bridge()
            host_bridge = self.get_hosting_bridge(default_bridge)
            #Read IPV4 address. 
            #ipv4host = host_bridge.get_bridge_ipv4_address()
            br_intf = host_bridge.interface
            #Read IPV6 address 
            #ipv6host = host_bridge.get_bridge_ipv6_address()

            log.debug("Setting up ios deny iptable rules")

            for int_block_port in self.internal_blocked_ports:

                log.debug("Removing the iptable rule for denying conainer access to ios")
                out, rc = iptables("-D",
                               "FORWARD",
                               "-p",
                               "tcp",
                               "--destination-port",
                                "%d" % int_block_port,
                               "-j",
                               "DROP"
                               )
                if rc != 0:
                    if log_level == "ERROR" :
                        log.error("Error deleting FORWARD/DROP ios rule. Error: %s" % str(out))
                    else:
                        log.debug("Error deleting FORWARD/DROP ios rule. Error: %s" % str(out))

                out, rc = ip6tables("-D",
                               "FORWARD",
                               "-p",
                               "tcp",
                               "--destination-port",
                                "%d" % int_block_port,
                               "-j",
                               "DROP"
                               )
                if rc != 0:
                    if log_level == "ERROR" :
                        log.error("Error deleting FORWARD/DROP ios rule. Error: %s" % str(out))
                    else:
                        log.debug("Error deleting FORWARD/DROP ios rule. Error: %s" % str(out))

                out, rc = iptables("-D",
                               "FORWARD",
                               "-p",
                               "tcp",
                               "--destination-port",
                                "%d" % int_block_port,
                               "-i",
                               br_intf,
                               "-j",
                               "ACCEPT"
                               )
                if rc != 0:
                    if log_level == "ERROR" :
                        log.error("Error deleting FORWARD/ACCEPT ios rule. Error: %s" % str(out))
                    else:
                        log.debug("Error deleting FORWARD/ACCEPT ios rule. Error: %s" % str(out))

                #if ipv6host:
                out, rc = ip6tables("-D",
                               "FORWARD",
                               "-p",
                               "tcp",
                               "--destination-port",
                               "%d" % int_block_port,
                               "-i",
                               br_intf,
                               "-j",
                               "ACCEPT"
                               )
                if rc != 0:
                    if log_level == "ERROR" :
                        log.error("Error deleting FORWARD/ACCEPT ios v6 rule. Error:%s" % str(out))
                    else:
                        log.debug("Error deleting FORWARD/ACCEPT ios v6 rule. Error:%s" % str(out))


        except Exception as ex:
            log.exception("Error in deleting iptables rule")

    def _create_logical_networks(self, hb, nw_types=None):
    
        if type(hb.supported_modes) is not list or hb.supported_modes == []:
            raise NetworkConfigurationError("Supported modes provided empty or invalid.")
    
        # For each of the supported mode, create a network
        for m in hb.supported_modes:
                
            if nw_types and m not in nw_types:
                continue
        
            i = int(hb.vpgno)
            network_name = "%s-%s%s" % (self.network_name_prefix, m, i)
            if hb.vlan_id and int(hb.vlan_id) == i:
                network_name = "%s-%s-v%s" % (self.network_name_prefix, m, i)
            elif hb.direct_connect:
                network_name = "%s-%s-r%s" % (self.network_name_prefix, m, i)
            reconcile = False
            if self.libvirt_conn:
                import libvirt
                c = self.libvirt_conn
            else:
                import libvirt
                c = libvirt.open()
                self.libvirt_conn = c
            if self.uuid_map.get(network_name):
                try:
                    n = c.networkLookupByUUIDString(self.uuid_map.get(network_name))
                    if n:
                        reconcile = True
                except libvirt.libvirtError:
                    self.uuid_map.pop(network_name, "")
                    pass
            # Get network settings corresponding to the type
            network_settings = hb.bridge_settings.get(m)
            desc = hb.description + " - " + m

            # If the type is NAT, get a pre allocated mac address block
            if m == Network.NETWORK_TYPE_NAT or m==Network.NETWORK_TYPE_NAT_DOCKER :
                ip_start = network_settings["ip_start"]
                ip_end = network_settings["ip_end"]
                log.debug("Setting up a mac address block for NAT network %s", network_name)
                mac_address_block = self._macregistry.get_macaddress_block(network_name, i, ip_start, ip_end)
                libvirt_nw_name = hb.natmode_name
            else:
                log.debug("No mac address block needed for non nat network %s", network_name)
                mac_address_block = []
                libvirt_nw_name = hb.bridgemode_name
            #if not reconcile:
            # Create a logical network and store it
            if not self.libvirt_conn:
                import libvirt
                self.libvirt_conn = libvirt.open()
            if m==Network.NETWORK_TYPE_NAT_DOCKER:
                import docker
                if self.base_url and self.api_version:
                    self.docker_api_client = Utils.get_docker_api_client(self.base_url, self.api_version, self.timeout, self.use_tls)
                    log.debug("docker api client config headers = %s" % self.docker_api_client.headers)
                network = Network(name=network_name, description=desc, source_linux_bridge=hb,
                                  network_type=m, network_settings=network_settings, repofolder=self.repofolder,
                                  mac_address_block=mac_address_block, container_info={"docker":{}}, reconcile=False, docker_conn=self.docker_api_client)
            else:

                network = Network(name=network_name, description=desc, source_linux_bridge=hb,
                                  network_type=m, network_settings=network_settings, repofolder=self.repofolder,
                                  mac_address_block=mac_address_block, container_info=self.container_info, reconcile=False, libvirt_conn=self.libvirt_conn)

            network.setup(reconcile)
            if network.uuid:
                self.uuid_map[network_name] = network.uuid
            self._save_uuid_map()
            self.NETWORKS[network_name] = network
            hb.assoc_networks[m] = network
            hb.logical_network_info[network_name] = {"type": m}
            if m == Network.NETWORK_TYPE_NAT or m==Network.NETWORK_TYPE_NAT_DOCKER:
                hb.logical_network_info[network_name].update(network_settings)
                
                # For the created NAT network merge its nat ip range to the used nat ip pool
                self.used_nat_range = self.used_nat_range | network.nat_range_ipset
                
            # If this is the default mode on the default bridge, mark this as default network
            if hb.interface == self.default_bridge and m == hb.default_mode:
                self.set_default_network(network_name)

            #Set the default docker network
            if  hb.interface == self.default_bridge and hb.default_mode == Network.NETWORK_TYPE_NAT and m==Network.NETWORK_TYPE_NAT_DOCKER:
                self.set_default_docker_nat_network(network_name)

    def _save_uuid_map(self):
        log.debug("Persisting the UUID map!")
        with open(self.uuid_store_file, "w", 0) as f:
            f.write(json.dumps(self.uuid_map))

    def stop(self, is_getting_shutdown=False):
        """
        Teardown all networks and hosting bridges
        :return:
        """
        if not self.enabled:
            return

        self.clear_iptables()

        for name, nw in self.NETWORKS.iteritems():
            if nw.app_ip_map != {}:
                log.debug("app ip map: %s" % str( nw.app_ip_map))
                log.error("To shutdown the service, there will be no App/Service should be attached to any network!")
                raise ValueError("To shutdown the service, there should be no App/Service should be attached to any network!")
        for n in self.NETWORKS.values():
            n.teardown(is_getting_shutdown)

        for h in self.HOSTING_BRIDGES.values():
            h.teardown()

        self.NETWORKS.clear()
        self.HOSTING_BRIDGES.clear()
        self.used_nat_range.clear()
        self.default_network = None
        # tear down the DHCP filteres defined
        if self.enable_dhcp_filters:
            self.teardown_dhcp_filters()
        self._save_uuid_map()
        self.uuid_map.clear()
        if self.libvirt_conn:
            self.libvirt_conn.close()
        self._running = False
        self._network_status = False

    def validate_req_ports(self, app_network):
        """
        Validates if the requested ports are available and not blocked or in use
        """
        log.debug("validate_req_ports:  %s" % app_network)
        if not app_network:
            return
        try:
            for intf in app_network:
                interface_name = intf.get('interface-name')
                ports = intf.get('ports')
                if not ports:
                    continue
                log.debug("ports: %s", ports)
                log.debug("Interface name: %s", interface_name)
                for type in ports.keys():
                    if type!="tcp" and type !="udp":
                        log.debug("Invalid port key:%s" % type)
                        raise ValueError("Invalid port type %s. Cannot proceed!" % type)
                    type_ports = ports.get(type)
                    type_mapping_list = []
                    for p in type_ports:
                        req_ports = p
                        description = None
                        if isinstance(p, dict):
                            req_ports = p.get('port')
                            description = p.get('description')
                        self._port_registry.validate_ports(req_ports, type)
            log.debug("Port requirement validation successful")
            return 
        except Exception as ex:
            log.error("Port requirement validation failed:%s" % str(ex))
            raise ex

    def app_setup_hook(self, appid, network_name, interface_name, mac_address, ports, req_port_map, mac_forward_mask='', mirroring=False, reconcile=False, port_map_bridge=False):
        """
        Do app related setup based on the network it has requested.
        :param appid:
        :param network_name:
        :param ports:
        :return: port_mapping
        """
        
        log.debug("NetworkController app_setup_hook: appid: %s, network_name: %s, interface_name: %s, \
                   mac_address: %s, ports: %s, req_port_map: %s, mac_forward_mask: %s, mirroring: %s", appid, network_name,
                   interface_name, mac_address, ports, req_port_map, mac_forward_mask, mirroring)
        nw = self.NETWORKS.get(network_name)
        if nw:
            port_mappings = self._get_port_mapping(appid, interface_name, network_name, ports, req_port_map, port_map_bridge)
            nw.app_setup_hook(appid, interface_name, mac_address, port_mappings, mac_forward_mask, mirroring, reconcile)
            return port_mappings
        
        # App wants to activate on a bridge network that doesn't exist yet
        # If auto_bridge_mode is on and there is a match, create the network here
        auto_br = self._get_auto_bridge(network_name)
        if auto_br:
            if not self.libvirt_conn:
                import libvirt
                self.libvirt_conn = libvirt.open()
            network = Network(name=network_name, description=auto_br['description'], source_linux_bridge=None,
                              network_type='bridge', network_settings={}, repofolder=self.repofolder, 
                              container_info=self.container_info, phys_interface=auto_br['phys_interface'], libvirt_conn=self.libvirt_conn, auto_br=auto_br)
                              
            network.setup()
            self.NETWORKS[network_name] = network
            port_mappings = self._get_port_mapping(appid, interface_name, network_name, ports, req_port_map)
            network.app_setup_hook(appid, interface_name, mac_address, port_mappings, mac_forward_mask, mirroring)
            return port_mappings

        return None


    def app_teardown_hook(self, appid, network_name, interface_name, mac_address, port_mappings):
        """
        Do app related teardown based on the network it was connected to
        :param appid:
        :param network_name:
        :return:
        """
        nw = self.NETWORKS.get(network_name)
        if nw:
            nw.app_teardown_hook(appid, interface_name, mac_address, port_mappings)
            
            # Delete standalone bridge networks during app deactivation
            # if no other app is using it
            if nw.app_ip_map == {}:
                if not nw.source_linux_bridge: 
                    nw.teardown()
                    self.NETWORKS.pop(network_name, None)
                elif nw.source_linux_bridge.creation_mode=="caf":
                    log.debug("Going to delete hosting bridge which was auto created")
                    self.delete_hosting_bridge(nw.source_linux_bridge.interface)
                else:
                    nw.reset_mirroring()

            return

        return None

    def get_default_network(self, net_docker=False):
    
        if not self.default_network:
            log.debug("Default network not set.")
            return None
    
        if not net_docker:
            return self.default_network
        else:
            if self.NETWORKS.get(self.default_network).network_type == Network.NETWORK_TYPE_NAT:
                return self.default_docker_nat_network
            else:
                return self.default_network
        

    def get_default_docker_nat_network(self):
        return self.default_docker_nat_network

    def get_secure_storage_network(self, app_type):
        if (SystemInfo.is_secure_storage_supported() == True):
            for nw in self.NETWORKS.values():
                d = nw.serialize()
                if nw.source_linux_bridge and nw.source_linux_bridge.secure_storage:
                    log.debug("Found Secure Storage Network: %s", d)
                    if (app_type == AppType.DOCKER and nw.network_type == "nat_docker" or
                        app_type == AppType.LXC and nw.network_type == "nat" or
                        app_type == AppType.VM and nw.network_type == "nat"):
                        log.debug("Returning for %s network Secure Storage Network: %s", nw.network_type, d)
                        return nw
        log.debug("Secure Storage Network unavailable for this platform")
        return None

    def get_default_bridge(self):
        return self.default_bridge

    def get_container_network(self, network_id, container_id):
        """
        App is configured to use a particular logical network.
        A container now needs to find out the corresponding container specific details.
        For now with libvirt, this simply returns the name of libvirt network domain if one is available.
        :param network_id:
        :return:
        """
        nw = self.NETWORKS.get(network_id)
        if nw:
            return nw.get_container_network(container_id)
        
        # If auto_bridge_mode is on, match the container network
        auto_br = self._get_auto_bridge(network_id)
        if auto_br:
            return auto_br["phys_interface"].replace(self.auto_bridge_mode_prefix, self.auto_br_libvirt_prefix)

        return None

    def set_default_network(self, network_id):
        if network_id not in self.NETWORKS:
            log.error("Network %s not found. Cannot set it as the default network!", network_id)
            raise NetworkConfigurationError("Network %s not found! Cannot set it as the default network!" % network_id)

        self.default_network = network_id
        log.info("Default network is set to %s", network_id)
        
    def set_default_docker_nat_network(self, network_id):
        if network_id not in self.NETWORKS:
            log.error("Network %s not found. Cannot set it as the default network!", network_id)
            raise NetworkConfigurationError("Network %s not found! Cannot set it as the default network!" % network_id)

        self.default_docker_nat_network = network_id
        log.info("Default docker nat network is set to %s", network_id)


    @property
    def used_phys_interfaces(self):
        """
        Physical interfaces can be used by svcbr's defined in network config,
        svcbr's dynamically created (Connexa style) or by auto bridge mode networks
        This method returns the list of physical interfaces enslaved and in use
        """
        rval = []
        for nw in self.NETWORKS.values():
            phys_inf = nw.external_interface
            if phys_inf and phys_inf not in rval:
                rval.append(phys_inf)
                
        return rval

    def list_networks(self):
        rval = []
        for n in self.NETWORKS.values():
            if n.source_linux_bridge and n.source_linux_bridge.is_internal:
                continue
            d = n.serialize()
            rval.append(d)
     
        # If auto_bridge_mode is on, announce the possible bridge networks
        rval.extend(self._get_auto_bridge_list())
            
        log.debug("Listing all Networks : %s" % str(rval))
        return rval
        
    def _get_auto_bridge_list(self):
        """
        If auto_bridge_mode is enabled in network config, this method finds the unused
        physical interfaces that match the given prefix and announces a corresponding
        bridge network on that physical interface. (if not already exists)
        
        Actual bridge network creation doesn't take place here. 
        It happens during app activation.
        """

        rval = []
        if not self.auto_bridge_mode_enabled or not self.auto_bridge_mode_prefix:
            # Auto bridge mode not turned on
            return rval
        
        description = self.auto_bridge_mode.get("description", None)
        if not description:
            description = "Auto bridge network"
            
        auto_br_prefix = self.auto_bridge_mode.get("iox_bridge_prefix", None)
        if not auto_br_prefix:
            auto_br_prefix = "iox-auto-bridge"
            
        mirroring_support = self.auto_bridge_mode.get("mirroring_support", False)

        arp_ignore = self.auto_bridge_mode.get("arp_ignore", False)

        for interface_name in Utils.get_interfaces(self.auto_bridge_mode_prefix):
        
            if interface_name in self.used_phys_interfaces:
                continue
        
            br_name = interface_name.replace(self.auto_bridge_mode_prefix, auto_br_prefix)
            
            br_dict = {'name': br_name, 
                       'source_linux_bridge': 'None', 
                       'network_type': 'bridge',
                       'description' : description + " via " + interface_name,
                       'phys_interface': interface_name,
                       'mirroring_support': mirroring_support,
                       'arp_ignore': arp_ignore
                      }
            
            if br_name not in self.NETWORKS:
                rval.append(br_dict)

        return rval
        
        
    def _get_auto_bridge(self, network_name):
        """
        If auto_bridge_mode is enabled in network config, this method finds the unused
        physical interfaces that match the given prefix and whether a bridge network
        with network_name can be announced. (if not already exists)
        
        Actual bridge network creation doesn't take place here. 
        It happens during app activation.
        """
        
        for br in self._get_auto_bridge_list():
            if br['name'] == network_name:
                return br
                
        return None

    def list_hosting_bridges(self):
        rval = []
        for h in self.HOSTING_BRIDGES.values():
            d = h.serialize()
            rval.append(d)
        log.debug("Listing all Hosting Bridges : %s" % str(rval))
        return rval
        
    def get_logical_vlan_network(self, vlan_tag):
        """
        vlan_tag : "all" - returns the bridge network associated     \
        with parent default_vlan_hosting_bridge bridge
        otherwise returns  bridge from the bridge network 
        with the vlan_tag..
        """

        ext_interface = self.default_vlan_hosting_bridge
        if vlan_tag == "all":
            hb = self.HOSTING_BRIDGES.get(ext_interface)
            if hb:
                return hb.assoc_networks[hb.default_mode]
            else:
                log.error("No hosting bridge found for interface:%s vlan:%s" % (ext_interface,  vlan_tag))
                return None
                
        for h in self.HOSTING_BRIDGES.values():
            if h.external_interface == ext_interface and \
               h.vlan_id == vlan_tag:
                return h.assoc_networks[h.default_mode]
        return None
                

    def _validate_nat_ip_range(self, nat_info, exclude_nw_name=None):
        """
        Example nat_info:
        {"nat_range_cidr": "192.168.10.32/27"}
        Example nat_info:
        {'ip_range': '192.168.223.10-192.168.223.254', 'subnet_mask': '255.255.255.0', 'gateway_ip': '192.168.223.1'}
        Example nat_info:
        {'ip_range': '192.168.10.0-192.168.10.32', 'subnet_mask': '255.255.255.224', 'gateway_ip': '192.168.10.1'}
        Example nat_info:
        {'ip_range': '192.168.10.2-192.168.10.30', 'subnet_mask': '255.255.255.224', 'gateway_ip': '192.168.10.1'}
        """
        nat_range_cidr = nat_info.get("nat_range_cidr")
        ip_range = nat_info.get("ip_range")
        
        # If no range is provided, use suggested_nat_range
        if not nat_range_cidr and not ip_range:
            nat_range_cidr = self.suggested_nat_range
        
        if nat_range_cidr:
            # Calculate ip_start, ip_end, gateway_ip, ip_range, subnet_mask from nat_range_cidr
            nw_ip, mask = nat_range_cidr.split('/')
            nat_nw = IPNetwork(nat_range_cidr)
            if nw_ip != str(nat_nw[0]):
                suggested = str(nat_nw[0]) + '/' + mask
                raise NetworkManagementError("Invalid NAT subnet: %s. Did you mean %s?" 
                                              % (nat_range_cidr, suggested))
            gateway_ip = str(nat_nw[1])
            ip_start = str(nat_nw[2])
            ip_end = str(nat_nw[-2])
            ip_range = str(nat_nw[2]) + '-' + str(nat_nw[-2])
            subnet_mask = str(nat_nw.netmask)
            # IPSet that corresponds to this nat range
            new_range = IPSet(nat_nw)
        else:
            # Not cidr format validate the ip range
            ip_start, ip_end = ip_range.split('-')
            subnet_mask = nat_info["subnet_mask"]
            ip_start_with_mask = IPNetwork(ip_start + '/' + subnet_mask)
            ip_end_with_mask = IPNetwork(ip_end + '/' + subnet_mask)
            
            # ip_start and ip_end need to be in the same subnet
            if ip_start_with_mask.network != ip_end_with_mask.network:
                raise NetworkManagementError("Invalid NAT ip range: %s %s" % (ip_range, subnet_mask))

            # ip_start one after network address
            if ip_start_with_mask.ip == ip_start_with_mask.network:
                ip_start = str(IPAddress(ip_start) + 1)
            
            # ip_end one before broadcast address
            if ip_end_with_mask.ip == ip_end_with_mask.broadcast:
                ip_end = str(IPAddress(ip_end) - 1)
            
            # IPSet that corresponds to this nat range            
            new_range = IPSet(IPRange(ip_start, ip_end))
                
            gateway_ip = nat_info.get("gateway_ip")
            if gateway_ip:
                gateway_ip_with_mask = IPNetwork(gateway_ip + '/' + subnet_mask)
                
                # gateway_ip and ip_start need to be in the same subnet
                if gateway_ip_with_mask.network != ip_start_with_mask.network:
                    raise NetworkManagementError("Invalid Gateway IP: %s, NAT Range: %s %s" % (gateway_ip, ip_range, subnet_mask))
                    
                # gateway_ip cannot be network address or broadcast address    
                if gateway_ip_with_mask.ip == gateway_ip_with_mask.network or \
                         gateway_ip_with_mask.ip == gateway_ip_with_mask.broadcast:
                    raise NetworkManagementError("Invalid Gateway IP: %s, NAT Range: %s %s" % (gateway_ip, ip_range, subnet_mask))
                   
                if gateway_ip == ip_start:
                    ip_start = str(IPAddress(ip_start) + 1)
                    
                new_range.add(gateway_ip)  # In case not in range
            else:
                # If gateway_ip not set assume the first address of the range as gateway address
                gateway_ip = ip_start
                ip_start = str(IPAddress(ip_start) + 1)
        
        # Compare new_range with existing used NAT pool to detect overlaps
        for nw in self.NETWORKS.values():
            # If nat nw already exists, do not compare with yourself
            if exclude_nw_name and nw.name == exclude_nw_name:
                continue
            if nw.network_type == Network.NETWORK_TYPE_NAT:
                if not new_range.isdisjoint(nw.nat_range_ipset):
                    raise NetworkManagementError("Requested NAT range "
                                 "overlaps with existing network: %s" % nw.name)
                 
        nat_info["gateway_ip"] = gateway_ip
        nat_info["ip_start"] = ip_start
        nat_info["ip_end"] = ip_end
        nat_info["ip_range"] = ip_range
        nat_info["subnet_mask"] = subnet_mask
        
    def _validate_bridgemode_info(self, bridge_config):
        """
        If mirror_mode bridge network is requested validate that platform supports it.
        Also make sure mirror mode not enabled when a NAT network is also present.
        """
        bridgemode_info = bridge_config[Network.NETWORK_TYPE_BRIDGE]
        if bridgemode_info.get("mirror_mode") == "yes":
            if not self.mirror_mode_supported:
                raise NetworkManagementError("Port mirroring for a bridge network not supported on this platform.")
                
            if Network.NETWORK_TYPE_NAT in bridge_config["supported_modes"]:
                raise NetworkManagementError("Port mirroring for bridge network cannot be used when a NAT network is also selected.")
        
    def _validate_phys_interface(self, bridge_config):
        """
        Check if requested physical interface and vlan already in use by another bridge
        """
        requested_phys_inf = bridge_config.get("external_interface")
        requested_vlan_id = bridge_config.get("vlan_id")
        
        for hb in self.HOSTING_BRIDGES.values():
            if hb.external_interface == requested_phys_inf and hb.vlan_id == requested_vlan_id:
                raise NetworkManagementError("Requested physical interface/vlan already "
                                 "in use by: %s - %s" % (hb.interface, hb.description))
         
    def add_hosting_bridge(self, bridge_config, persist=True):
        """
        Add hosting bridge defined by given bridge_config
        Example bridge_config:
        {
        "description": "Southbound Network",
        "vlan_id": "10",
        "external_interface": "intsvc0",
        "supported_modes": ["nat", "bridge"],
        "bridge_ip": {"mode": "static", "ip": "192.168.0.15", "subnet_mask": "255.255.255.0"},
        "nat": {"nat_range_cidr": "192.168.10.32/27"}
        "bridge": {"mirror_mode": "yes"}
        }
        """
        
        bridge_config["dynamically_created"] = True
        bridge_config["svcbr_setup_script"] = self.config.get("svcbr_setup_script")
        bridge_config["svcbr_teardown_script"] = self.config.get("svcbr_teardown_script")
        
        if self._macregistry:
            bridge_config["generated_mac_addr"] = self._macregistry.get_mac_address_for_bridge()
        
        self.available_phys_infs = Utils.getlines_from_file(self.vpg_inf_file)
        if bridge_config["external_interface"] not in self.available_phys_infs:
            raise NetworkManagementError("Invalid physical interface: %s" % bridge_config["external_interface"])
            
        self._validate_phys_interface(bridge_config)
        
        if Network.NETWORK_TYPE_NAT in bridge_config["supported_modes"]:
            nat_info = bridge_config[Network.NETWORK_TYPE_NAT]
            self._validate_nat_ip_range(nat_info)
            
        if Network.NETWORK_TYPE_NAT_DOCKER in bridge_config["supported_modes"]:
            docker_nat_info = bridge_config[Network.NETWORK_TYPE_NAT]
            self._validate_nat_ip_range(docker_nat_info)
            
        if Network.NETWORK_TYPE_BRIDGE in bridge_config["supported_modes"]:
            bridgemode_info = bridge_config.get(Network.NETWORK_TYPE_BRIDGE)
            if bridgemode_info:
                self._validate_bridgemode_info(bridge_config)
        
        hb = HostingBridge(None, bridge_config)
        hb.setup()        
        self.HOSTING_BRIDGES[hb.interface] = hb
        
        try:
            self._create_logical_networks(hb)
        except Exception as ex:
            log.error("Error while creating logical networks: %s" % str(ex))
            hb.teardown()
            self.HOSTING_BRIDGES.pop(hb.interface)
            raise
        
        if persist:
            # Update persistent config and save
            if not self.config["hosting_bridges"]:
                self.config["hosting_bridges"] = {}
            self.config["hosting_bridges"][hb.interface] = hb.bridge_settings
            self._save_data()
        
        return hb.interface
            
    def create_vlan_network(self, vlan_tag):
        """
        Add hosting bridge defined by given bridge_config
        Example bridge_config:
        {
        "description": "Southbound Network",
        "vlan_id": "10",
        "external_interface": "intsvc0",
        "supported_modes": ["nat", "bridge"],
        "bridge_ip": {"mode": "static", "ip": "192.168.0.15", "subnet_mask": "255.255.255.0"},
        "nat": {"nat_range_cidr": "192.168.10.32/27"}
        "bridge": {"mirror_mode": "yes"}
        }
        """
        bridge_config = {}
        if self.default_vlan_hosting_bridge is None:
            log.error("Default external bridge for vlan creation not found")
            raise Exception("Default external bridge for vlan creation not found")

        bridge_config["external_interface"]=self.default_vlan_hosting_bridge
        bridge_config["description"]="Dynamic vlan bridge"
        bridge_config["vlan_id"]=vlan_tag
        bridge_config["supported_modes"]=["bridge"]
        bridge_config["bridge_ip"]={"mode":HostingBridge.NO_IP_ADDRESS}
        bridge_config["creation_mode"] = "caf"
        bridge_config["direct_connect"] = True

        if self.default_vlan_hosting_bridge in self.HOSTING_BRIDGES:
            if self.HOSTING_BRIDGES[self.default_vlan_hosting_bridge].mirroring_support:
                bridge_config["mirroring_support"] = True
            if self.HOSTING_BRIDGES[self.default_vlan_hosting_bridge].arp_ignore:
                bridge_config["arp_ignore"] = True
            if self.HOSTING_BRIDGES[self.default_vlan_hosting_bridge].mac_forward_mode:
                bridge_config["mac_forward_mode"] = True
                bridge_config["mac_forward_mask_default"] = \
			        self.HOSTING_BRIDGES[self.default_vlan_hosting_bridge].gp_fwd_mask_default

        log.debug("Creating dynamic hosting bridge with vlan. %s" % bridge_config)

        # Since create_vlan_network is done at activation time, we don't need to persist the created bridge
        hb_id =  self.add_hosting_bridge(bridge_config, persist=False)
        return self.get_hosting_bridge(hb_id)
        


    def delete_hosting_bridge(self, bridge_id):
        """
        Delete the hosting bridge with given bridge_id and its associated logical networks
        """
        
        if bridge_id not in self.HOSTING_BRIDGES.keys():
            raise NetworkNotExistingError("Network bridge not found: %s" % bridge_id)
            
        hb = self.HOSTING_BRIDGES[bridge_id]
        if not hb.bridge_settings.get("dynamically_created"):
            raise NetworkManagementError("Cannot delete default network %s." % bridge_id)
        
        for nw in hb.assoc_networks.values():
            if nw.app_ip_map != {}:
                raise NetworkManagementError("Cannot delete. Please deactivate app(s) using the network: %s" % nw.app_ip_map.keys())
        
        # Call teardown() for associated logical networks 
        for nw in hb.assoc_networks.values():
            nw.teardown()
            self.NETWORKS.pop(nw.name)
            if nw.network_type == Network.NETWORK_TYPE_NAT:
                # Remove this network's nat range from the used nat pool
                self.used_nat_range = self.used_nat_range - nw.nat_range_ipset
            
        hb.teardown()    
        self.HOSTING_BRIDGES.pop(bridge_id)

        if hb.generated_mac_addr and self._macregistry:
            self._macregistry.release_mac_address(hb.generated_mac_addr)
        
        if bridge_id in self.config["hosting_bridges"]:
            # Update persistent config and save
            self.config["hosting_bridges"].pop(bridge_id)

            self._save_data()

        
    def update_hosting_bridge(self, bridge_id, bridge_config):
        """
        Update the hosting bridge with given bridge_id and bridge_config
        Only updateable fields currently are description and nat range
        """
        
        if bridge_id not in self.HOSTING_BRIDGES.keys():
            raise NetworkNotExistingError("Network bridge not found: %s" % bridge_id)
            
        hb = self.HOSTING_BRIDGES[bridge_id]
            
        # Update nat range if it is provided and if it is in supported modes    
        nat_info = bridge_config.get(Network.NETWORK_TYPE_NAT)
        if nat_info and Network.NETWORK_TYPE_NAT in hb.supported_modes:
            nw = hb.assoc_networks[Network.NETWORK_TYPE_NAT]
            self._validate_nat_ip_range(nat_info, nw.name)
            if nat_info != hb.bridge_settings[Network.NETWORK_TYPE_NAT]:
                if nw.app_ip_map != {}:
                    raise NetworkManagementError("Cannot change NAT IP range. Deactivate "
                          "app(s) using the network: %s" % nw.app_ip_map.keys())

                nw.teardown()
                self.uuid_map.pop(nw.name, "")
                self.used_nat_range = self.used_nat_range - nw.nat_range_ipset
                hb.set_nat_range(nat_info)
                self._create_logical_networks(hb, [Network.NETWORK_TYPE_NAT])

                
        # Update description if it is provided
        # Also change the description of the associated logical networks
        desc = bridge_config.get("description")
        if desc and str(desc) != hb.description:
            hb.description = str(desc)
            hb.bridge_settings["description"] = hb.description
            for n in hb.assoc_networks.values():
                n.description = str(desc) + " - " +  n.network_type
        
        # Update persistent config and save
        self.config["hosting_bridges"][hb.interface] = hb.bridge_settings
        
        self._save_data()
        self._save_uuid_map()

    def get_network(self, network_id):
        """
        Return a logical network with network_id if it exists
        :param network_id:
        :return:
        """
        
        if network_id in self.NETWORKS:
            return self.NETWORKS[network_id]
        
        auto_br = self._get_auto_bridge(network_id)
        if auto_br:
            if not self.libvirt_conn:
                import libvirt
                self.libvirt_conn = libvirt.open()
            return Network(name=auto_br['name'], description=auto_br['description'],
                         source_linux_bridge=None, network_type='bridge', 
                        phys_interface=auto_br["phys_interface"], 
                        network_settings={}, repofolder=self.repofolder, 
                        libvirt_conn=self.libvirt_conn)
            
        return None
        

    def get_network_status(self):
        """
        Return True if networks are up
        """
        if len(self.NETWORKS) == 0:
            return False
        for nw in self.NETWORKS.values():
            if nw.network_status() == False:
                self._network_status = False
                log.error("Network: %s is down" % nw)
                return False
        self._network_status = True
        return True

    def get_hosting_bridge(self, bridge_id):
        """
        Return a logical instance of hosting bridge with bridge_id if it exists
        :param bridge_id:
        :return:
        """
        return self.HOSTING_BRIDGES.get(bridge_id)


    @classmethod
    def getInstance(cls, config):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = NetworkController(config)
        return cls.__singleton

'''
if "__main__" == __name__:
    import yaml
    import json

    logging.basicConfig(
         level=logging.INFO,
         datefmt='%H:%M:%S'
    )

    # set up logging to console
    console = logging.StreamHandler()
    console.setLevel(logging.INFO)
    # set a format which is simpler for console use
    # formatter = logging.Formatter('%(name)-12s: %(levelname)-8s %(message)s')
    # console.setFormatter(formatter)
    # add the handler to the root logger
    logging.getLogger().addHandler(console)

    # container_info = {
    #     "libvirt": {
    #         "connection_str": "lxc:///"
    #     }
    # }

    network_config_file = Utils.getNetworkConfigFile()
    network_config = yaml.safe_load(file(network_config_file))

    ns = NetworkController(network_config, "/tmp")

    ns.start()

    print "Hosting Bridges : \n--------------------------\n"
    print json.dumps(ns.list_hosting_bridges(),indent=2)
    print "Network List : \n--------------------------\n"
    print json.dumps(ns.list_networks(), indent=2)
    print "Default Network: %s" % ns.get_default_network()

    # ns.teardown()
    #
    # print "Hosting Bridges : \n--------------------------\n"
    # print json.dumps(ns.list_hosting_bridges(),indent=2)
    # print "Network List : \n--------------------------\n"
    # print json.dumps(ns.list_networks(),indent=2)
    # print "Default Network: %s" % ns.get_default_network()
    '''
