'''

@author: havishwa

Copyright (c) 2014-2015 by Cisco Systems, Inc.
All rights reserved.
'''

from appfw.utils import infraexceptions
from appfw.runtime.error import MandatoryMetadataError
from appfw.utils.utils import ImmutableMap

class CartridgeManifest(object):
    """
    Represents cartridge metadata. Captures cartridge information such as  general info, supported
    language runtime, version, cpu architecture, env variables etc.,
    """

    # Mandatory sections
    #TODO: Add mandatory sections based on final design/conclusion
    OPTIONS_INFO = (("info",), True)

    OPTIONS_CARTRIDGE = (("cartridge",), True)
    OPTIONS_CARTRIDGE_TYPE = (("cartridge", "type"), True)
    OPTIONS_CARTRIDGE_CPUARCH = (("cartridge", "cpuarch"), True)
    OPTIONS_CARTRIDGE_FORMAT = (("cartridge", "format"), True)
    OPTIONS_CARTRIDGE_HANDLER = (("cartridge", "handle-as"), True)
    OPTIONS_CARTRIDGE_PROVIDES = (("cartridge", "provides"), True)
    OPTIONS_CARTRIDGE_PACKAGE = (("cartridge", "payload"), True)

    # Non mandatory sections
    OPTIONS_CARTRIDGE_ENV = (("cartridge", "env"), False)
    OPTIONS_CARTRIDGE_DEPENDSON = (("cartridge", "depends-on"), False)
    OPTIONS_DESCRIPTOR_SCHEMA_VERSION = (("descriptor-schema-version",), False)

    OPTIONS_LIST = [OPTIONS_INFO, OPTIONS_CARTRIDGE,
                    OPTIONS_CARTRIDGE_TYPE,
                    OPTIONS_CARTRIDGE_CPUARCH,
                    OPTIONS_CARTRIDGE_FORMAT,
                    OPTIONS_CARTRIDGE_HANDLER,
                    OPTIONS_CARTRIDGE_PROVIDES,
                    OPTIONS_CARTRIDGE_PACKAGE,
                    OPTIONS_CARTRIDGE_ENV,
                    OPTIONS_CARTRIDGE_DEPENDSON,
                    OPTIONS_DESCRIPTOR_SCHEMA_VERSION]

    def __init__(self, mfilepath):
        self.manifestfile = mfilepath
        tempmap = self.parse()
        self._mdmap = self._processDefaults(tempmap)
        self._validate_mandatory(self._mdmap)


    def parse(self):
        import yaml
        rval = None
        fp = None
        try:
            fp = file(self.manifestfile, "r")
            rval = yaml.safe_load(fp)
            return rval
        except Exception as ex:
            raise infraexceptions.MalformedManifestError(str(ex))
        finally:
            if fp:
                fp.close()


    def _validate_mandatory(self, tempmap):
        for option_group in self.OPTIONS_LIST:
            # TODO: Develop apptype as an entity which enables other entities
            # to register with apptype specific logic. For instance,
            # PaaSMetadataValidator, PaaSStager etc., would register with PaaS app type
            # For now, introduce custom logic for validation in Metadata itself
            keymap, mandatory = option_group
            if mandatory:
                if self.get_from_dict(tempmap, keymap) is None:
                    raise MandatoryMetadataError("Mandatory metadata section is missing : %s" %
                                                 ".".join(keymap))

    @classmethod
    def get_from_dict(cls, datadict, maplist, default=None):
        """
        Return the value from dict by recursively querying
        for keys specified in maplist
        #ex. datadict = {"a" : {"b" : "c"} }
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        """
        rval = default
        try:
            #rval = reduce(dict.__getitem__, maplist, datadict)
            rval = reduce(lambda d, k: d[k], maplist, datadict)
        except Exception:
            pass

        return rval

    @classmethod
    def set_in_dict(cls, datadict, maplist, value):
        """Set the value of a key in the dict by recursively
        iterating with keys specified in maplist
        ex.
        #>>> datadict
        #{'a': {'b': 'c'}}
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        #>>> set_in_dict(datadict, ["a", "b"], "$$$")
        #>>> get_from_dict(datadict, ["a","b"])
        #'$$$'
        """
        rval = cls.get_from_dict(datadict, maplist[:-1])
        if rval:
            rval[maplist[-1]] = value


########## Section level properties #################

    @property
    def manifest_version(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_DESCRIPTOR_SCHEMA_VERSION[0], "1.0")

    @property
    def info(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_INFO[0])

    @property
    def cartridge(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_CARTRIDGE[0])

#######################################################

    @property
    def name(self):
        return self.info.get("name")

    @property
    def description(self):
        return self.info.get("description", "")

    @property
    def author(self):
        return self.info.get("author-name", "")

    @property
    def authorLink(self):
        return self.info.get("author-link", "")

    @property
    def version(self):
        return self.info.get("version", "")

    #################################

    @property
    def type(self):
        return str(self.cartridge.get("type"))

    @property
    def format(self):
        return str(self.cartridge.get("format"))

    @property
    def handleas(self):
        return self.cartridge.get("handle-as")

    @property
    def provides(self):
        return self.cartridge.get("provides")

    @property
    def payload(self):
        return self.cartridge.get("payload")

    @property
    def dependson(self):
        return self.cartridge.get("depends-on")

    @property
    def runtime(self):
        return str(self.cartridge.get("runtime"))

    @property
    def runtime_version(self):
        return str(self.cartridge.get("runtime_version"))

    @property
    def cpuarch(self):
        return self.cartridge.get("cpuarch")

    @property
    def rootfs(self):
        return self.cartridge.get("rootfs")

    @property
    def dirname(self):
        return self.cartridge.get("dirname")

    @property
    def env(self):
        return self.cartridge.get("env")

    def _processDefaults(self, mdmap):

        for section, mandatory in self.OPTIONS_LIST:
            if section in mdmap:
                mdmap[section] = ImmutableMap(mdmap[section])
        return ImmutableMap(mdmap)
