#-----------------------------------------------------
#
# Copyright (c) 2012 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

'''

@author: alhu
'''
import logging
import json
import falcon
import time

from token import TokenManager, LimitError, IOXTokenManager
from apiservice import ResourceRoute
from common import ResourceBase, AuthenticatedResource, make_error_response, add_no_cache_header
from auth import check_authentication, check_auth_token, check_login_allowed
from jsonencoder import JSONEncoder

log = logging.getLogger("runtime.api.tokenservice")
jsonencoder = JSONEncoder()

@ResourceRoute("/tokenservice", endpoint="tokenservice")
class TokenServiceResourceHandler(ResourceBase):
    """This class is the resource handler for tokenservice resource."""

    def on_post(self, request, response):
        '''
        Creates a token. The request needs to be authenticated.
        '''
        tokenInfo = None

        try:
            auth_header = request.auth
            log.debug("Host:%s request uri:%s" % (request.host, request.uri))  
            if not check_login_allowed(auth_header):
                log.debug("Too many fail login attempts")
                response = make_error_response(response,
                                               'Too many failed login requests please try after some time',
                                               'Too many failed login requests please try after some time',
                                               falcon.HTTP_403)
                return
            
            auth_user = check_authentication(auth_header, request)
            if auth_user:
                if auth_user == "LOGIN ATTEMPT WITH DEFAULT CREDS":
                    response = make_error_response(response,
                                                           'Logged in with default user credentials. Change default user password and login again.',
                                                           'Logged in with default user credentials. Change default user password and login again.',
                                                           '299 Change default user password')
                    return
                else:
                    tokenInfo = TokenManager.getInstance().createToken(auth_user)
            else:
                log.debug("Failed to authenticate user")
                response = make_error_response(response,
                                                       'Failed to authenticate user',
                                                       'Failed to authenticate user',
                                                       falcon.HTTP_401)
                return

        except LimitError:
            response = make_error_response(response,
                                                   'Max session limit reached',
                                                   'Max session limit reached',
                                                   falcon.HTTP_503)
            return

        if tokenInfo is None:
            response = make_error_response(response,
                                                   "Token cannot be created",
                                                   "Token cannot be created",
                                                   falcon.HTTP_500)
            return

        tokenResp = {}
        tokenResp["token"] = {}
        tokenResp["token"]["id"] = tokenInfo.id

        response.set_header('Strict-Transport-Security', "max-age=31536000; includeSubDomains; preload")
        response.status = falcon.HTTP_200
        response.body = json.dumps(tokenResp)

    @falcon.before(check_auth_token)
    def on_get(self, request, response):
        '''
        Lists all the active tokens. Returns an empty body for now.
        '''
        # TODO: Currently it returns an empty list. When RBAC is
        #       implemented, the returned list should be based on
        #       the user's privilege.
        tokens = TokenManager.getInstance().listTokens()
        out = jsonencoder.encode(tokens)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

@ResourceRoute("/tokenservice/{token}", endpoint="token")
class TokenResourceHandler(AuthenticatedResource):
    """This class is the resource handler for token resource."""

    def on_get(self, request, response, token):
        tokenInfo = TokenManager.getInstance().getToken(token)
        if tokenInfo is None:
            response = make_error_response(response,
                                           "Token with token-id: %s, is not found" % token,
                                           "Token not found",
                                           falcon.HTTP_404)
            return

        tokenResp = {}
        tokenResp["token"] = {}
        tokenResp["token"]["id"] = tokenInfo.id
        tokenResp["token"]["expiry_period"] = int(tokenInfo._exp_tm - time.time())

        response = add_no_cache_header(response)
        response.status = falcon.HTTP_200
        response.body = json.dumps(tokenResp)

   
    def on_delete(self, request, response, token):
        '''
        Deletes the given token id.
        '''
        if token:
            tokenInfo = TokenManager.getInstance().getToken(token)
            if tokenInfo is not None:
                TokenManager.getInstance().deleteToken(token)
                response.status = falcon.HTTP_200
                response.body = ''
                return

        response = make_error_response(response,
                                       "Token with token-id: %s, is not found" % token,
                                       "Token not found",
                                       falcon.HTTP_404)

@ResourceRoute("/ioxv/tokenservice/{appid}", endpoint="ioxvtoken")
class IOXTokenResourceHandler():

    def on_get(self, request, response, appid):

        tokenInfo = IOXTokenManager.getInstance().getTokenByAppId(appid)
        if tokenInfo is None:
            response = make_error_response(response,
                                           "IOX Token for app with appid: %s, is not found" % appid,
                                           "Token not found",
                                           falcon.HTTP_404)
            return

        tokenResp = {}
        tokenResp["token"] = {}
        tokenResp["token"]["id"] = tokenInfo.id
        tokenResp["token"]["expiry_period"] = int(tokenInfo._exp_tm - time.time())

        response.status = falcon.HTTP_200
        response.body = json.dumps(tokenResp)
