'''
Created on 2013

@author: samagraw

Copyright (c) 2013-2014 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import os
import subprocess
import tempfile, shutil
from   ..utils.utils import Utils
from systeminfo import SystemInfo

log = logging.getLogger("runtime")

class SyslogServerConfig(object):

    cfgFile = Utils.getSystemConfigValue("logging", "syslogConfigPath", "/etc/syslog.conf")

    logLevelToString = {
        'emerg':'emergency',
        'alert':'alert',
        'crit':'critical',
        'err':'error',
        'warn':'warning',
        'notice':'notice',
        'info':'info',
        'debug':'debug'}

    '''
    Configure syslog-server entry
    when update is unique id of syslog server string (combination of ip, port and proto), -> edit an exisiting entry
    when update is "", -> add a new entry
    '''
    @classmethod
    def updateConfig(cls, data, update):
        log.debug("Configuring syslog-server entry :%s", data)
        
        try:
            reverseLookUp = dict(zip(SyslogServerConfig.logLevelToString.values(), \
                                     SyslogServerConfig.logLevelToString.keys()))
            logLevel = reverseLookUp[data['level']]
            uuid = SystemInfo.get_systemid()
            template = 'template("[' + uuid + '] [$HOST] [$LEVEL] $PROGRAM $MSG")'
            dstHost = 'destination d_loghost   { ' + data['protocol'].lower() + '("' + data['ipaddr'] + '" port(' + str(data['port']) +') ' + template + '); }; '
            filter = 'filter p_loghost { level(' + logLevel + '..emerg) and facility(user, local1); }; '
            dstLog = 'log { source(s_sys); filter(p_loghost);  destination(d_loghost); };\n'

            cfg = dstHost + filter + dstLog;
            log.debug("Syslog server config for conf file: %s", cfg)
                                                   
            with open(SyslogServerConfig.cfgFile) as fileHdlr:
                lines = fileHdlr.readlines()
                                                      
            tmpfile = tempfile.NamedTemporaryFile(delete=True)
            for line in lines:             
                if "destination d_loghost" in line:  
                    if update:
                        ipAddr = line.split('("')[1].split('"')[0]
                        portNum = line.split('port(')[1].split(')')[0]
                        protocol = line.split('{ ')[1].split('(')[0]
                        if ipAddr==update.split(',')[0] and portNum==update.split(',')[1] and protocol==update.split(',')[2].lower():
                            log.info("Updating Syslog-server entry: %s %s %s", data['ipaddr'], data['port'], data['protocol']) 
                            tmpfile.write(cfg)
                        else:
                            tmpfile.write(line)
                    else:
                        tmpfile.write(line)
                else:
                    tmpfile.write(line)

            # Write in a particular format in syslog-ng.conf file
            if update == "":
                log.debug("Adding new syslog-server entry")
                tmpfile.write(cfg)
            
            tmpfile.flush()
            shutil.copy2(tmpfile.name, SyslogServerConfig.cfgFile)
            tmpfile.close()
            log.info("Sending a SIGHUP signal to syslog to load new config")
            subprocess.check_output("/bin/kill -HUP `cat /var/run/syslogd.pid`", shell=True)
        except IOError:
            os.remove(tmpfile)
            log.error("Exception while updating syslog-server configuration")

    #Get syslog-server information
    @classmethod
    def getConfig(cls):
        log.debug("Get syslog-server configuration")
        serverList = []
        
        try:
            with open(SyslogServerConfig.cfgFile) as fileHdlr:
                lines = fileHdlr.readlines()
            
            for line in lines:
                if "destination d_loghost" in line:
                    if line[0] != '#':
                        ipAddr = line.split('("')[1].split('"')[0]
                        portNum = line.split('port(')[1].split(')')[0]
                        protocol = line.split('{ ')[1].split('(')[0].upper()
                        logLevel = SyslogServerConfig.logLevelToString[line.split('level(')[1].split('..')[0]]

                        serverList.append({"ipaddr" : ipAddr,
                                           "port" : int(portNum),
                                           "protocol" : protocol,
                                           "level" : logLevel})
            
            log.debug("Syslog-server list: %s", serverList)
        except IOError:
            log.error("Exception while fetching syslog-server configuration")

        return serverList
    
    #Delete syslog-server configuration
    @classmethod
    def deleteConfig(cls, data):
        log.debug("Deleting syslog-server entry: %s", data)
        try:
            with open(SyslogServerConfig.cfgFile) as fileHdlr:
                lines = fileHdlr.readlines()
              
            tmpfile = tempfile.NamedTemporaryFile(delete=True)
            for line in lines:
                if "destination d_loghost" in line:
                    ipAddr = line.split('("')[1].split('"')[0]
                    portNum = int(line.split('port(')[1].split(')')[0])
                    protocol = line.split('{ ')[1].split('(')[0].upper() 
                    if ipAddr==data['ipaddr'] and portNum==data['port'] and protocol==data['protocol']:
                        log.debug("Deleting Syslog-server entry: %s %s %s", data['ipaddr'], data['port'], data['protocol'])
                    else:
                        tmpfile.write(line) 
                else:  
                    tmpfile.write(line)
                                  
            tmpfile.flush()
            shutil.copy2(tmpfile.name, SyslogServerConfig.cfgFile)
            tmpfile.close()
            log.info("Sending a SIGHUP signal to syslog to load new config")
            subprocess.check_output("/bin/kill -HUP `cat /var/run/syslogd.pid`", shell=True)
        except IOError:
            os.remove(tmpfile)
            log.error("Exception while deleting syslog-server configuration")
