__author__ = 'pedeshpa'

import logging
import falcon
import copy
import getopt
import re
from apiservice import APIService, ResourceRoute
from common import AuthenticatedResource, make_response, make_error_response, add_no_cache_header
from ..utils.utils import Utils
from ..utils.docker_command_utils import docker_command_dict
from jsonencoder import JSONEncoder

jsonencoder = JSONEncoder()

log = logging.getLogger("runtime.api")

@ResourceRoute("/platform/capability", endpoint="platformcapability")
class PlatformCapabilityResourceHandler(AuthenticatedResource):
    def on_get(self, request, response):
        detail = request.get_param("detailed")
        if isinstance(detail, basestring):
            if detail.lower() == "false":
                detail = False
            elif detail.lower() == "true":
                detail = True
            else:
                detail = False
        else:
            detail = False

        if APIService.instance.app_manager:
            pc = APIService.instance.app_manager.get_platform_capability(detail)
        else:
            from appfw.runtime.platformcapabilities import PlatformCapabilities
            pc = PlatformCapabilities.getInstance().get_platform_capability(detail)
            log.debug("Platform cap: %s" % pc)
        if not pc:
            response = make_error_response(response,
                                           description="Unable to retrieve platform capability info",
                                           message="Unable to retrieve platform capability info")
            return

        try:
            out = jsonencoder.encode(pc)
            headers = {
                'Content-Type': "application/json",
                'Strict-Transport-Security': "max-age=31536000; includeSubDomains; preload"
            }
            response = add_no_cache_header(response)
            response = make_response(response, out, falcon.HTTP_200, headers)
        except Exception as ex:
            log.exception("Error while json encoding response : %s", str(pc))
            response = make_error_response(response,
                                           description=str(ex),
                                           message="Error encoding response to json")
        return


@ResourceRoute("/platform/rsmgr/profile_definitions", endpoint="rsmgrInfo")
class PlatformResourceProfileHandler(AuthenticatedResource):
    def on_get(self, request, response):

        if APIService.instance.app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return


        try:
            pc = APIService.instance.app_manager.get_platform_resource_profiles()
        except Exception as ex:
            log.exception("Error fetching platform resource profile definitions %s", str(ex))
            response = make_error_response(response,
                                           description=str(ex),
                                           message="Unable to retrieve resource profile definitions")
            return

        try:
            out = jsonencoder.encode(pc)
            headers = {
                'Content-Type': "application/json",
            }
            response = add_no_cache_header(response)
            response = make_response(response, out, falcon.HTTP_200, headers)
        except Exception as ex:
            log.exception("Error while json encoding response : %s", str(pc))
            response = make_error_response(response,
                                           description=str(ex),
                                           message="Error encoding response to json")
        return


@ResourceRoute("/platform/rsmgr/allocations", endpoint="rsmgrInfo")
class PlatformResourceAllocations(AuthenticatedResource):
    def on_get(self, request, response):
        if APIService.instance.app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return


        try:
            pc = APIService.instance.app_manager.get_platform_resource_allocations()
        except Exception as ex:
            log.exception("Exception while retrieving allocations per app, %s", str(ex))
            response = make_error_response(response,
                                           description=str(ex),
                                           message="Unable to retrieve resource allocation info")
            return

        try:
            out = jsonencoder.encode(pc)
            headers = {
                'Content-Type': "application/json",
                'Cache-Control': "no-cache"
            }
            response = make_response(response, out, falcon.HTTP_200, headers)
        except Exception as ex:
            log.exception("Error while json encoding response : %s", str(pc))
            response = make_error_response(response,
                                           description=str(ex),
                                           message="Error encoding response to json")
        return


@ResourceRoute("/platform/runtime_options", endpoint="runtime options info")
class PlatformRuntimeOptions(AuthenticatedResource):
    def on_get(self, request, response):
        """
        This method will parse the given runtime options and return the status accordingly
        """
        resp_template = {
            "validation_status":{
                "is_valid": True,
                "input_options": "",
                "error_msg": "",
                "un_supported_options": []
            },
             "supported_options": []
        }
        runtime_options = request.get_param("options", default="")
        try:
            if APIService.instance.app_manager:
                pc = APIService.instance.app_manager.resource_manager.get_platform_cap
            else:
                from appfw.runtime.platformcapabilities import PlatformCapabilities
                pc = PlatformCapabilities.getInstance()
            # Reason we are using DEEPCOPY here is, if we are modify the values of list- those will effect values in platform capability values
            black_listed_options = pc.black_listed_docker_run_options
            resp_template["supported_options"] = pc.get_merged_docker_runtime_options()
            if runtime_options:
                un_supported_options = []
                temp = runtime_options
                while True:
                    try:
                        options, args = Utils.parse_runtime_options(temp)
                        break
                    except getopt.GetoptError as ex:
                        un_supported_options.append(ex.opt)
                        index = temp.find(ex.opt)
                        temp = temp[index:]
                        small_option_data = re.findall("-.[ =]", temp)
                        small_option_index = -1
                        if small_option_data:
                            small_option_index = temp.find(small_option_data[0])
                        big_option_index = temp.find("--")
                        if small_option_index != -1 or big_option_index != -1:
                            if small_option_index == -1:
                                temp = temp[big_option_index:]
                            elif big_option_index == -1:
                                temp = temp[small_option_index:]
                            elif small_option_index > big_option_index:
                                temp = temp[big_option_index:]
                            else:
                                temp = temp[small_option_index:]
                        else:
                            break
                if un_supported_options:
                    for option in black_listed_options:
                        #If the length of the option is greater than the 1, then we can assume that option is long arg
                            #Otherwise option will be short arg
                        temp_option = option.lstrip("-")
                        if len(temp_option) > 1:
                            if runtime_options.find("--"+temp_option+" ") != -1 or runtime_options.find("--"+temp_option+"=") != -1:
                                un_supported_options.append(temp_option)
                        else:
                            if runtime_options.find("-"+temp_option+" ") != -1 or runtime_options.find("-"+temp_option+"=") != -1:
                                un_supported_options.append(temp_option)
                    resp_template["validation_status"]["un_supported_options"] = un_supported_options
                    raise Exception("Given runtime options are invalid!")
                if black_listed_options:
                    for option in options:
                        command = option[0]
                        if command in black_listed_options or command.lstrip("-") in black_listed_options:
                            resp_template["validation_status"]["un_supported_options"].append(command.lstrip("-"))
                if resp_template["validation_status"]["un_supported_options"]:
                    raise ValueError("Provided options %s are black listed !"%resp_template["validation_status"]["un_supported_options"])
        except Exception as ex:
            resp_template["validation_status"]["is_valid"] = False
            log.exception("Error while parsing the runtime options %s", str(ex))
            response = make_error_response(response,
                                           description=str(ex),
                                           message="Given runtime options are invalid!")
            if isinstance(ex, getopt.GetoptError):
                resp_template["validation_status"]["un_supported_options"].append(ex.opt)
            resp_template["validation_status"]["error_msg"] = str(ex)
        resp_template["validation_status"]["input_options"] = runtime_options
        out = jsonencoder.encode(resp_template)
        headers = {
            'Content-Type': "application/json",
            'Cache-Control': "no-cache"
        }
        response = make_response(response, out, falcon.HTTP_200, headers)


