#! /bin/sh
#
# svcbr_setup.sh
# Invoked by CAF at the creation of a logical network, this script will setup
# the hosting bridge and the physical or VLAN interface underneath it.
#
# Usage: svcbr_setup.sh [options]
#     Options:
#     -b or -bridge     - hosting bridge. The script will pick the first availalbe svcbr bridge
#                         if "-b" option is not provided
#     -i or -interface  - physical interface
#     -v or -vlan       - VLAN ID. The script will create the VLAN interface and enslave it
#                         to svcbr_x bridge if VLAN ID is provided
#     -a or -addressing - IP addressing scheme, "dhcp", "static", or "no_ip_address"(default)
#     -ip or -ipaddr    - static IP assigned to svcbr_x
#     -m or -netmask    - netmask for svcbr_x
#     -mac or -hwaddr    - MAC address for svcbr_x
#     -af or -append-stsfile  - append the bridge status file
#     -of or -output-stsfile  - overwrite the bridge status file
#
# Required utilities: bridge-util, iproute2 (ip link), DHCP client udhcpc or equivalent
#
# February 2016, Ting Yu
#
# Copyright (c) 2016 by cisco Systems, Inc.
# All rights reserved
#

SVCBR_IF=""
PHY_IF=""
VLAN_ID=""

SVCBR_PREFIX=svcbr
SVCBR_STS_FILE_DIR=/opt/cisco/caf/config/tmp
SVCBR_STS_FILE=${SVCBR_STS_FILE_DIR}/bridge_status.yaml
SVCBR_STS_FILE_APPEND=0

#DHCP_CLNT=/sbin/dhclient
DHCP_CLNT="/sbin/udhcpc -b -i "
DHCP_LEASE_FILE_DIR=/etc/platform/dhcp

LOGGER="/usr/bin/logger -t svcbr_setup"

RTN_CODE=0                       # 0 for success, 1 for failure
RTN_MSG="Successfully created"   # error message if any

# sanity check the input parameters
verify_input_params () {
    if [ -z "$PHY_IF" ]; then
        RTN_MSG="physical interface is missing"
        return 1
    fi

    if [ ! -z "$SVCBR_ADDRESSING" ]; then
        if  [ "$SVCBR_ADDRESSING" != "dhcp" ] && [ "$SVCBR_ADDRESSING" != "static" ]; then
            RTN_MSG="bridge ip address assignment selection $SVCBR_ADDRESSING is not valid"
            return 1
        fi
        if [ "$SVCBR_ADDRESSING" = "static" ] && [ -z "$SVCBR_STATIC_IP" ]; then
            RTN_MSG="bridge static ip address configuration is missing"
            return 1
        fi
    else
        SVCBR_ADDRESSING="no_ip_address"
    fi
    return 0
}

setup_physical_intf () {
    # create VLAN interface if needed
    if [ ! -z $VLAN_ID ]; then
        # sanity check the given VLAN ID
        if [ $VLAN_ID -lt 1 ] || [ $VLAN_ID -gt 4094 ]; then
            RTN_MSG="input VLAN tag $VLAN_ID is not valid"
            return 1
        fi
        VLAN_SUBIF=${PHY_IF}.${VLAN_ID}
        ip link show $VLAN_SUBIF > /dev/null 2>&1
        if [ $? -ne 0 ]; then
            ip link add link $PHY_IF name $VLAN_SUBIF type vlan id $VLAN_ID > /dev/null 2>&1
            if [ $? -ne 0 ]; then
                RTN_MSG="VLAN interface $VLAN_SUBIF creation has failed"
                return 1
            fi
            echo "VLAN interface $VLAN_SUBIF has been created"
            ${LOGGER} "VLAN interface $VLAN_SUBIF has been created"
        fi
        INTF_TOADD=$VLAN_SUBIF
    else
        INTF_TOADD=$PHY_IF
    fi
    ip link set dev $INTF_TOADD up > /dev/null 2>&1
    # clear the IP addresss of the physical interface
    ip addr flush dev $INTF_TOADD > /dev/null 2>&1
    ${LOGGER} "clear the IP address of interface $INTF_TOADD"
    return 0
}

# find the first available bridge interface to use
find_svcbr_to_use () {
    if [ ! -z "$SVCBR_IF" ]; then
        # check if the requested svcbr bridge is available
        ip link show $SVCBR_IF > /dev/null 2>&1
        if [ $? -ne 0 ]; then
            return 0
        else
            echo "requested bridge $SVCBR_IF is already in use"
            RTN_MSG="requested bridge $SVCBR_IF is already in use"
            return 1
        fi
    fi
    # get the number of bridges currently available
    num_svcbr=`ip link | awk '{print $2}' | grep svcbr | wc -l`
    # find the bridge ID to use
    br_idx=0
    while true; do
        SVCBR_IF=${SVCBR_PREFIX}_${br_idx}
        ip link show $SVCBR_IF > /dev/null 2>&1
        if [ $? -ne 0 ]; then
            break
        else
            br_idx=`expr $br_idx + 1`
            if [ $br_idx -ge $num_svcbr ]; then
                break
            fi
        fi
    done
    SVCBR_IF=${SVCBR_PREFIX}_$br_idx
    return 0
}

# create hosting bridge svcbr_x
create_svcbr_intf () {
    RTN_MSG=`brctl addbr $SVCBR_IF 2>&1`
    if [ $? -ne 0 ]; then
        RTN_MSG="fail to create bridge $SVCBR_IF"
        return 1
    fi

    # enslave the vlan sub-interface to the bridge
    brctl show | grep $SVCBR_IF | grep $INTF_TOADD  > /dev/null 2>&1
    if [ $? -ne 0 ]; then
        brctl addif $SVCBR_IF $INTF_TOADD
        if [ $? -ne 0 ]; then
            RTN_MSG="fail to add $INTF_TOADD to $SVCBR_IF"
            return 1
        fi
    fi
    ip link set dev $SVCBR_IF up > /dev/null 2>&1
    echo "bridge $SVCBR_IF has been created"
    ${LOGGER} "bridge $SVCBR_IF has been created"
    # set MAC address if provided
    if [ ! -z "$SVCBR_MACADDR" ]; then
        ip link set dev $SVCBR_IF address $SVCBR_MACADDR > /dev/null 2>&1
    fi
    return 0
}

#convert netmask to cidr format
mask2cidr() {
    nbits=0
    idx=1
    while [ $idx -le 4 ]; do
       octet=`echo $1 | cut -d. -f${idx}`
       case $octet in
            255) nbits=$((nbits+8));;
            254) nbits=$((nbits+7));;
            252) nbits=$((nbits+6));;
            248) nbits=$((nbits+5));;
            240) nbits=$((nbits+4));;
            224) nbits=$((nbits+3));;
            192) nbits=$((nbits+2));;
            128) nbits=$((nbits+1));;
            0);;
            *) echo "netmask $1 is not recognized"; return 1
        esac
        idx=$((idx+1))
    done
}

# configure IP address of svcbr_x
config_svcbr_ipaddr () {
    if [ -z "$SVCBR_ADDRESSING" ]; then
        # no ip address for svcbr_x, assuming it is associated with a bridged network
        return 0
    elif [ "$SVCBR_ADDRESSING" = "dhcp" ]; then
        # clear the current IP
        ip addr flush dev $SVCBR_IF> /dev/null 2>&1

        # run dhcp on the bridge interface
        mkdir -p $DHCP_LEASE_FILE_DIR
        DHCP_LEASE_FILE=${DHCP_LEASE_FILE_DIR}/${SVCBR_IF}.leases
        DHCP_PID_FILE=/var/run/udhcpc.${SVCBR_IF}.pid
        $DHCP_CLNT $SVCBR_IF -p $DHCP_PID_FILE > /dev/nul 2>&1
        SVCBR_IP=`ip addr show $SVCBR_IF | grep "inet\b" | awk '{print $2}' | cut -d/ -f1`
    elif [ "$SVCBR_ADDRESSING" = "static" ]; then
        addr=${SVCBR_STATIC_IP}
        # set netmask if provided
        if [ ! -z $SVCBR_NETMASK ]; then
            mask2cidr $SVCBR_NETMASK
            if [ $? -eq 0 ]; then
                addr=`echo "${SVCBR_STATIC_IP}/$nbits"`
            fi
        fi
        ip addr add $addr dev $SVCBR_IF > /dev/null 2>&1
        if [ $? -ne 0 ]; then
            RTN_MSG="fails to assign static IP address $addr to $SVCBR_IF"
            return 1
        fi
        SVCBR_IP=`ip addr show $SVCBR_IF | grep "inet\b" | awk '{print $2}' | cut -d/ -f1`
        echo "$SVCBR_IF has been configured with IP address $addr"
    fi
    return 0
}

# generate a bridge status YAML file for CAF
generate_svcbr_sts_file () {
    mkdir -p $SVCBR_STS_FILE_DIR
    if [ "${SVCBR_STS_FILE_APPEND}" != "1" ]; then
        rm -f $SVCBR_STS_FILE
    fi
    if [ ! -f "${SVCBR_STS_FILE}" ]; then
        echo "hosting_bridges:" > $SVCBR_STS_FILE
    fi
    echo "  ${SVCBR_IF}:" >> $SVCBR_STS_FILE
    echo "    vlan_id: $VLAN_ID" >> $SVCBR_STS_FILE
    echo "    bridge_ip:" >> $SVCBR_STS_FILE
    echo "        mode: $SVCBR_ADDRESSING" >> $SVCBR_STS_FILE
    if [ "$SVCBR_ADDRESSING" = "dhcp" ]; then
        echo "        dhcp_lease_file: ${DHCP_LEASE_FILE}" >> $SVCBR_STS_FILE
    elif [ "$SVCBR_ADDRESSING" = "static" ]; then
        echo "        ip: $SVCBR_STATIC_IP" >> $SVCBR_STS_FILE
        echo "        netmask: $SVCBR_NETMASK" >> $SVCBR_STS_FILE
    fi
    echo "    status_code: $RTN_CODE"  >> $SVCBR_STS_FILE
    echo "    message: $RTN_MSG"   >> $SVCBR_STS_FILE
}

# get input options
if [ $# = 0 ]; then
    echo "missing input parameters"
    exit 1
fi

while [ "$#" != 0 ];
do
    case $1 in
        -b | -bridge )
            SVCBR_IF=$2;
            ;;
        -i | -interface )
            PHY_IF=$2;
            ;;
        -v | -vlan )
            VLAN_ID=$2;
            ;;
        -a | -addressing )
            SVCBR_ADDRESSING=$2;
            ;;
        -ip | -ipaddr )
            SVCBR_STATIC_IP=$2;
            ;;
        -m | -netmask )
            SVCBR_NETMASK=$2;
            ;;
        -mac | -hwaddr )
            SVCBR_MACADDR=$2;
            ;;
        -af | -append-stsfile )
            SVCBR_STS_FILE_DIR=`dirname $2`
            SVCBR_STS_FILE=$2
            SVCBR_STS_FILE_APPEND=1
            ;;
        -of | -output-stsfile )
            SVCBR_STS_FILE_DIR=`dirname $2`
            SVCBR_STS_FILE=$2
            ;;
        -* )
            ;;
    esac
    shift
done

report_then_abort () {
    RTN_CODE=1
    generate_svcbr_sts_file
    echo $RTN_MSG
    exit 1
}

# find the bridge interface to use
find_svcbr_to_use
if [ $? -ne 0 ]; then
    report_then_abort
fi

verify_input_params
if [ $? -ne 0 ]; then
    report_then_abort
fi

# setup physical interface, create VLAN if needed
setup_physical_intf
if [ $? -ne 0 ]; then
    report_then_abort
fi

# create the bridge interface
create_svcbr_intf
if [ $? -ne 0 ]; then
    report_then_abort
fi

config_svcbr_ipaddr
if [ $? -ne 0 ]; then
    report_then_abort
fi

# all set for CAF to create the rest of the logical networks
RTN_MSG="Successfully created"
generate_svcbr_sts_file
