#!/bin/sh
# Will setup the network for Given container

create_dir(){
    directory=$1
    if [ -d "$directory" ]; then
	    echo "Given dir $directory is already there"
    else
	    echo "Given dir $directory is not found, so creating one"
	    mkdir -p $directory
    fi
}
HELP(){
    echo 'This script sets up networking for docker style applications'
    echo 'Will be executed at the time of app startup'
    echo 'Usage: ./docker_networksetup.sh -d $DOMID -p $PIDREPO -i $INTERFACE -r $RESOLV_CONF -s $SCRIPT_PATH'
    echo '-d: domain id of the conatiner'
    echo '-p: PID repo to store the DHCP client daemon pids'
    echo '-i: Interface to which ip needs to be assigned'
    echo '-r: DNS resolve file path. Should be the location w.r.t the apps rootfs'
    echo '-s: DHCP client - custom script path'
    echo '-c: Provide DHCP client identifier'
    echo 'Note: Before running this script, make sure you have network name space created in host for the container'
    echo 'To create network namespace of the container, run this command: ln -s /proc/{CON_INIT_PID}/ns/net /var/run/netns/{DOM_ID}'
    exit 1
}
echo "Running the docker network up script"
# Making sure process is running with root user
if [ "$(id -u)" != "0" ]; then
        echo "This script must be run as root" 1>&2
        exit 1
fi
pointer=1
while [ $pointer -le $# ]; do
   #param=$1
   if [ "`expr \"$1\" : \"- .*\"`" = "0" ];then
      case $1 in
         -d | --domid)
                shift
                DOMAIN_ID=$1
                ;;
         -p | --pidrepo)
                shift
                PID_REPO=$1
                ;;
         -i | --interface)
                shift
                INTERFACE=$1
                ;;
         -r | --resolvconf)
                shift
                R_CONF=$1
                ;;
         -s | --script)
                shift
                SCRIPT_PATH=$1
                ;;
         -c | --clientid)
                shift
                CLIENT_ID=$1
                ;;
         -f | --ipfamily)
                shift
                IP_FAMILY=$1
                ;;
         -pid | --processid)
                shift
                PID=$1
                ;;
         -l | --seclabel)
                shift
                SEC_LABEL=$1
                ;;
         -h|--help) HELP ;;
         *)
            echo "Invalid option: $1"
            exit 1 ;;
      esac
   else
      echo "Invalid parameters passed: check --help option"
      exit 1
   fi
   shift
done
if [ "x$DOMAIN_ID" = "x" ] || [ "x$PID_REPO" = "x" ] || [ "x$INTERFACE" = "x" ] || [ "x$R_CONF" = "x" ] || [ "x$IP_FAMILY" = "x" ] || [ "x$PID" = "x" ];then
	echo "Mandatory Arguments: DOMAIN_ID: $DOMAIN_ID, PID_REPO: $PID_REPO,INTERFACE: $INTERFACE, RESOLV_CONF: $R_CONF, IP family: $IP_FAMILY, PID:$PID are missing"
	exit 1
fi

# Check for the udhcpc binary is exists
if which udhcpc >/dev/null; then
    echo "Udhcpc binary exists"
else
    echo "Udhcpc binary not found!"
    exit 1
fi

# If security label is provided, use it to run the process
if [ "x$SEC_LABEL" != "x" ];then
    echo $SEC_LABEL > /proc/self/attr/current
fi

if [ "x$IP_FAMILY" = "xipv4" ];then
    echo "Given IP family is : $IP_FAMILY"
else
    echo "Unsupported IP family: $IP_FAMILY is given"
    exit 1
fi
# Check and create pid repo
create_dir $PID_REPO
export R_CONF=$R_CONF

CLIENT_PARAM=""
if [ "x$CLIENT_ID" = "x" ]; then
    CLIENT_PARAM=""
else
    CLIENT_PARAM="-C ${CLIENT_ID}"
fi

# Entering the container network space and running udhcpc on a interface and trying to get ip for it.
if [ "x$SCRIPT_PATH" = "x" ];then
    ip netns exec $DOMAIN_ID udhcpc -R -b -p $PID_REPO/$INTERFACE$IP_FAMILY -i $CLIENT_PARAM $INTERFACE
else
    ip netns exec $DOMAIN_ID udhcpc -R -b -p $PID_REPO/$INTERFACE$IP_FAMILY -s $SCRIPT_PATH $CLIENT_PARAM -i $INTERFACE
fi
net_return=$?
if [ $net_return != 0 ]; then
	echo "Issue while running udhcpc. So exiting"
	if [ ! -d "/proc/$PID" ]; then
        echo "App got crashed, before CAF can setup the networking!"
        exit 3
    fi
	exit $net_return
fi
exit 0
