'''
Copyright (c) 2014 by Cisco Systems, Inc.
All rights reserved.

@author: madawood
'''

import logging
import shutil
import tempfile
import tarfile
import os
import subprocess
import zipfile
import copy
import json
from stager import StagingResponse
from dockerstager import DockerStager
from ..utils.utils import Utils, USER_EXTRACTED_DIR, LAYER_CONTENTS_DIR, LAYER_ARCHIVE_FILE, DOCKER_METADATA_DIR
from ..utils.infraexceptions import *
from ..utils.docker_utils import DockerUtils, DOCKER_MANIFEST_FILE_NAME, DOCKER_LAYERS_MANIFEST_FILE, \
    docker_manifest_contents, docker_config_contents
import tarfile
from ..utils.infraexceptions import StagingError

log = logging.getLogger("runtime.hosting")


class NativeDockerStager(DockerStager):
    """
    Stager for docker type application container
    """
    __singleton = None

    def __new__(cls, *args, **kwargs):
        # subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(NativeDockerStager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = NativeDockerStager(*args)
        return cls.__singleton

    def __init__(self, config, languageRuntimes=None):
        super(self.__class__, self).__init__(config, languageRuntimes)

    def stageConnector(self, stagingReq):
        """
        The docker rootfs tarball in extract_archive is extracted into rootfs in the repo directory.
        Then the layer tarballs are extracted into layercontents in each layer directory to form blobs of data.
        """

        log.debug("Begining staging for docker app. AppID: %s" % stagingReq.getConnectorId())
        response = dict()
        staging_location = os.path.join(self._staging_location, stagingReq.getConnectorId())
        if not os.path.isdir(staging_location):
            os.makedirs(staging_location)
        pkg = stagingReq.getPackageObject()
        layer_reg_serv = stagingReq.getValue("layer_reg_serv")
        metadata = stagingReq.getValue("metadata")
        target = stagingReq.getValue("target")
        args = stagingReq.getValue("args")
        repodir = self._config.get("controller", "repo")
        pkg.dest_dir = os.path.join(repodir, stagingReq.getConnectorId(), USER_EXTRACTED_DIR)
        rootfs_tar = os.path.join(repodir, stagingReq.getConnectorId(), USER_EXTRACTED_DIR,
                                  metadata.startup.get("rootfs", "rootfs.tar"))
        if not os.path.isfile(rootfs_tar):
            log.info("ROOTFS archive is not found so, we will be reverse engineer it from layers.")
            if os.path.isfile(os.path.join(repodir, stagingReq.getConnectorId(), USER_EXTRACTED_DIR, "rootfs",
                                           DOCKER_LAYERS_MANIFEST_FILE)):
                manifest_file = os.path.join(repodir, stagingReq.getConnectorId(), USER_EXTRACTED_DIR, "rootfs",
                                             DOCKER_LAYERS_MANIFEST_FILE)
                try:
                    with open(manifest_file) as f:
                        layer_list = json.load(f)
                    eng_rootfs = self.engineer_rootfs(metadata, layer_reg_serv, layer_list, stagingReq.getConnectorId(),
                                                 os.path.join(repodir, stagingReq.getConnectorId()), target, args)
                    log.debug("Engineered rootfs: %s" % eng_rootfs)
                    response["docker_rootfs"] = eng_rootfs   
                except Exception as ex:
                    log.exception(
                        "Error while creating the docker rootfs archive, Cause: %s" % (ex.message))
                    raise ValueError(
                        "Error while creating the docker rootfs archive, Cause: %s" % (ex.message))
            else:
                log.error(
                    "Docker manifest file: %s is not exists!" % os.path.join(repodir, stagingReq.getConnectorId(),
                                                                             USER_EXTRACTED_DIR, "rootfs",
                                                                             DOCKER_LAYERS_MANIFEST_FILE))
                raise ValueError(
                    "Docker manifest file: %s is not exists!" % os.path.join(repodir, stagingReq.getConnectorId(),
                                                                             USER_EXTRACTED_DIR, "rootfs",
                                                                             DOCKER_LAYERS_MANIFEST_FILE))
        app_config_file_name = Utils.find_app_config_filename(pkg.dest_dir)
        if app_config_file_name:
            app_config_file_name = os.path.join(pkg.dest_dir, app_config_file_name)
        response["staged-package-path"] = pkg.dest_dir
        response["app-config-file-name"] = app_config_file_name
        return StagingResponse(stagingReq.getConnectorId(), stagingReq.getPackagePath(), staging_location, response)

    def engineer_rootfs(self, metadata, layer_reg_serv, layer_list, app_id, app_repo, target, args):
        """
        When docker rootfs is not provides then, CAF will create the data and re-create the docker rootfs archive.
        """
        log.debug("Creating docker rootfs from layers")
        tmp_loc = Utils.getSystemConfigValue("controller", "upload_dir", "/tmp")
        temp_dir = tempfile.mkdtemp(dir=tmp_loc, prefix="rootfs_arch_")
        delete_rootfs_after_load = Utils.getSystemConfigValue("docker-container",
                                                "delete_rootfs_after_load", False, "bool")
        try:
            rootfs_archive_dir = os.path.join(temp_dir, "rootfs_archive_dir")
            os.mkdir(rootfs_archive_dir)
            self._check_delete_layers_part_of_app_repo(layer_list, app_repo)
            layers_not_found = layer_reg_serv.check_layers_exists(layer_list)
            if not layers_not_found:
                self.handle_layers_in_layer_repo(metadata, app_id, app_repo, layer_reg_serv, layer_list,
                                                 rootfs_archive_dir, target, args)
            else:
                log.error("Layers: %s are missing from the layer registry" % layers_not_found)
                raise MandatoryDataMissingError("Layers: %s are missing from the layer registry" % layers_not_found)
            # By this time all layers and metadata is available at rootfs_archive_dir, so now create a docker rootfs archive
            rootfs_archive = os.path.join(temp_dir, metadata.startup.get("rootfs", "rootfs.tar"))
            with tarfile.open(rootfs_archive, "w") as tar:
                tar.add(rootfs_archive_dir, arcname="")
            if delete_rootfs_after_load:
                tmp_rootfs_dir = os.path.join(tmp_loc, app_id)
                if not os.path.exists(tmp_rootfs_dir):
                    os.mkdir(tmp_rootfs_dir)
                shutil.move(rootfs_archive, tmp_rootfs_dir)
                log.debug("Engineered rootfs in tmpfs : %s" % os.path.join(tmp_rootfs_dir, metadata.startup.get("rootfs", "rootfs.tar"))) 
                return os.path.join(tmp_rootfs_dir, metadata.startup.get("rootfs", "rootfs.tar"))
            else:
                shutil.move(rootfs_archive, os.path.join(app_repo, USER_EXTRACTED_DIR))
                return os.path.join(app_repo, USER_EXTRACTED_DIR, metadata.startup.get("rootfs", "rootfs.tar"))
        except Exception as ex:
            log.exception("Error while reverse engineering the ROOTFS. Cause: %s" % ex.message)
            raise ValueError("Error while reverse engineering the ROOTFS. Cause: %s" % ex.message)
        finally:
            shutil.rmtree(temp_dir, ignore_errors=True)

    def _check_delete_layers_part_of_app_repo(self, layer_list, app_repo):
        """
        Check and deletes the layers, if they are part of the app repo.
        """
        log.debug("Check if the layers as part of the app repo!")
        rootfs_dir = os.path.join(app_repo, USER_EXTRACTED_DIR, "rootfs")
        if os.path.exists(rootfs_dir):
            for layer in layer_list:
                if os.path.exists(os.path.join(rootfs_dir, layer)):
                    shutil.rmtree(os.path.join(rootfs_dir, layer.encode("utf-8")), ignore_errors=True)

    def handle_layers_in_layer_repo(self, metadata, app_id, app_repo, layer_reg_serv, layer_list, rootfs_archive_dir,
                                    target, args):
        """
        Handle the layers from layers repo, by creating layer.tar and calaulating tha SHA cheksum of the layers
            in order to create docker deployable archive.
        """
        log.debug("Building docker rootfs archive from layers present in layer-repo!")
        docker_metadata_dir = os.path.join(app_repo, USER_EXTRACTED_DIR, "rootfs", DOCKER_METADATA_DIR)
        docker_sha256_list = []
        if os.path.exists(docker_metadata_dir):
            log.debug("Docker layer metadata is provided so we will be using this to create docker rootfs archive")
            self._compose_archive_docker_metadata(app_id, docker_metadata_dir, layer_reg_serv, rootfs_archive_dir)
        else:
            log.debug("No docker metadata is provided, so we will create the metadata and rootfs archive from layer registry!")
            layer_repo = layer_reg_serv.repo
            for layer in layer_list:
                layer_info = layer_reg_serv.get(layer)
                layer_dir = os.path.join(rootfs_archive_dir, layer_info["docker_id"])
                os.mkdir(layer_dir)
                layer_repo_contents = os.path.join(layer_repo, layer.encode("utf-8"), LAYER_CONTENTS_DIR)
                sha256_layer = self._create_copy_layer_archive(layer_repo_contents, layer_dir)
                docker_sha256_list.append((layer_info["docker_id"], sha256_layer))
            config_json = self._compose_docker_config_data(metadata, docker_sha256_list, rootfs_archive_dir, target,
                                                           args)
            image_file = os.path.join(app_repo, USER_EXTRACTED_DIR, "rootfs", "image.json")
            image_name=None
            image_tag=None
            if os.path.exists(image_file):
                log.debug("Loading image detail of %s" % image_file)
                with open(image_file, "r") as f:
                    image_details = f.read()
                    image_details = json.loads(image_details)
                    image_name = image_details.get("image_name")
                    image_tag = image_details.get("image_tag")

            self._compose_docker_manifest_data(docker_sha256_list, app_id, config_json, rootfs_archive_dir, image_name, image_tag) 

    def _compose_archive_docker_metadata(self, app_id, docker_metadata_dir, layer_reg_serv, rootfs_archive_dir):
        """
        Compose the docker deployable archive from metadata provided in the package.
        """
        try:
            layer_repo = layer_reg_serv.repo
            Utils.copytree_contents(docker_metadata_dir, rootfs_archive_dir, symlinks=True)
            docker_metadata_dir = rootfs_archive_dir
            manifest_file = os.path.join(docker_metadata_dir, DOCKER_MANIFEST_FILE_NAME)
            if os.path.isfile(manifest_file):
                with open(manifest_file) as f:
                    manifest_data = json.load(f)
                config_file = manifest_data[0]["Config"]
                config_file = os.path.join(docker_metadata_dir, config_file)
                if os.path.isfile(config_file):
                    with open(config_file) as f:
                        config_data = json.load(f)
                    docker_layer_list = manifest_data[0]["Layers"]
                    layer_list = config_data["rootfs"]["diff_ids"]
                    for index in range(len(layer_list)):
                        layer_list[index] = layer_list[index].split(":")[1].strip()
                    for index in range(len(docker_layer_list)):
                        docker_layer_list[index] = docker_layer_list[index].split("/")[0].strip()
                    layers_missing = layer_reg_serv.check_layers_exists(layer_list)
                    if layers_missing:
                        raise ValueError("Layers: %s are missing from the layer registry!"%layers_missing)
                    else:
                        count = 0
                        for layer in zip(layer_list, docker_layer_list):
                            layer_repo_contents = os.path.join(layer_repo, layer[0], LAYER_CONTENTS_DIR)
                            dst_dir = os.path.join(docker_metadata_dir, layer[1])
                            sha256_layer = self._create_copy_layer_archive(layer_repo_contents, dst_dir)
                            layer_list[count] = "sha256:"+sha256_layer
                            count = count + 1
                        config_data["rootfs"]["diff_ids"] = layer_list
                        with open(config_file, "w") as f:
                            json.dump(config_data, f)
                else:
                    raise ValueError("Docker metadata config file: %s is not present:"%config_file)
            else:
                raise ValueError("Docker metadata manifest file: %s is not present:"%manifest_file)
        except Exception as ex:
            log.exception("Error while creating docker archive from docker metadata. Cause: %s"%ex.message)
            raise ex

    def _compose_docker_config_data(self, metadata, docker_sha_list, rootfs_dir, target, args):
        """
        Compose the config data for the docker archive
        This the structure of the config data we will generate.
             {
              "architecture": None,
              "config": {
                "Hostname": "",
                "Domainname": "",
                "User": "user:group",
                "AttachStdin": False,
                "AttachStdout": False,
                "AttachStderr": False,
                "Tty": True,
                "OpenStdin": False,
                "StdinOnce": False,
                "Env": [
                  "PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin",
                   "JAVA_HOME=/usr/bin/java"
                ],
                "Cmd": ["/bin/sleep", "infinity"],
                "ArgsEscaped": True,
                "Image": "",
                "Volumes": None,
                "WorkingDir": "",
                "Entrypoint": None,
                "OnBuild": None,
                "Labels": None
              },
              "container": "",
              "created": "",
              "docker_version": "",
              "history": [],
              "os": "linux",
              "rootfs": {
                "type": "layers",
                "diff_ids": ["sha256:a30b835850bfd4c7e9495edf7085cedfad918219227c7157ff71e8afe2661f63",
                                "sha256:6267b420796f78004358a36a2dd7ea24640e0d2cd9bbfdba43bb0c140ce73567"]
              }
        }
        """
        log.info("Creating docker config data!")
        docker_config_contents_local = copy.deepcopy(docker_config_contents)
        startup_section = metadata.startup
        user = startup_section.get("user", "")
        group = startup_section.get("group", "")
        currentdir = startup_section.get("workdir", "")
        env = metadata.app_env
        docker_config_contents_local["architecture"] = Utils.get_cpuarch()
        if group:
            user = user.strip() + ":" + group.strip()
        docker_config_contents_local["config"]["user"] = str(user)
        for key, val in env.iteritems():
            docker_config_contents_local["config"]["Env"].append(str(key.strip() + "=" + val.strip()))
        cmd_list = []
        if isinstance(target, str):
            cmd_list.append(target)
        elif isinstance(target, list):
            cmd_list.extend(target)
        if isinstance(args, str):
            cmd_list.append(args)
        elif isinstance(args, list):
            cmd_list.extend(args)
        docker_config_contents_local["config"]["WorkingDir"] = currentdir
        docker_config_contents_local["config"]["Cmd"] = cmd_list
        from datetime import datetime
        docker_config_contents_local["created"] = datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S.%fZ')
        for layer_data in docker_sha_list:
            docker_config_contents_local["rootfs"]["diff_ids"].append("sha256:" + layer_data[1])
        with open(os.path.join(rootfs_dir, "config.json"), "w") as f:
            json.dump(docker_config_contents_local, f)
        with open(os.path.join(rootfs_dir, "config.json")) as f:
            sha256 = Utils.generate_sha_digest(f, "SHA256")
        os.rename(os.path.join(rootfs_dir, "config.json"), os.path.join(rootfs_dir, sha256 + ".json"))
        log.debug("Config file successfully got created. File: %s" % sha256 + ".json")
        return sha256 + ".json"

    def _compose_docker_manifest_data(self, docker_sha_list, app_id, config_json, rootfs_dir, image_name=None, image_tag=None):
        """
        Compose the manifest data for the docker archive
        [
              {
                "Config": "",
                "RepoTags": ["id:tag"],
                "Layers": ["3e956c1639d737c48d9ed8b59d1572b5baeee3f2e139bf495ac65bd802462f5e/layer.tar",
                            "ef577a10e7acd54c339f698ac587c92a14f3ebb9da1ac2b335123a3c7a9dfa00/layer.tar"]
              }
        ]
        """
        log.info("Creating docker manifest data!")
        docker_manifest_contents_local = copy.deepcopy(docker_manifest_contents)
        docker_manifest_contents_local[0]["Config"] = config_json
        if image_name:
            if image_tag:
                docker_manifest_contents_local[0]["RepoTags"].append(image_name+":"+image_tag)
            else:
                docker_manifest_contents_local[0]["RepoTags"].append(image_name)
        else:
            docker_manifest_contents_local[0]["RepoTags"].append(str(app_id).lower() + ":latest")
        for layer_data in docker_sha_list:
            docker_manifest_contents_local[0]["Layers"].append(os.path.join(layer_data[0], LAYER_ARCHIVE_FILE))
        with open(os.path.join(rootfs_dir, DOCKER_MANIFEST_FILE_NAME), "w") as f:
            json.dump(docker_manifest_contents_local, f)
        log.debug("Docker manifest data is sucessfully created. File: %s" % DOCKER_MANIFEST_FILE_NAME)

    def _create_copy_layer_archive(self, layer_contents_dir, dest_dir):
        """
        This will create the layer.tar if it is not available.
        If it is available then it will simply copy the layer .tar to dest_dir.
        It will return the SHA256 of the layer.tar
        """
        layer_archive_file = os.path.join(layer_contents_dir, LAYER_ARCHIVE_FILE)
        if os.path.isfile(layer_archive_file):
            shutil.copy2(layer_archive_file, dest_dir)
        else:
            log.debug(
                "Layer: %s ,layer archive: %s are not present in layer-repo. So creating one!" % (
                    dest_dir, LAYER_ARCHIVE_FILE))
            layer_archive = os.path.join(dest_dir, LAYER_ARCHIVE_FILE)
            with tarfile.open(layer_archive, "w") as tar:
                tar.add(layer_contents_dir, arcname="")
            # Delete all the data from layer contents and copy the layer.tar in to the dir
            for root, dirs, files in os.walk(layer_contents_dir):
                for f in files:
                    os.unlink(os.path.join(root, f))
                for d in dirs:
                    shutil.rmtree(os.path.join(root, d), ignore_errors=True)
            shutil.copy2(layer_archive, layer_contents_dir)
        with open(layer_archive_file) as f:
            return Utils.generate_sha_digest(f, "SHA256")

