__author__ = 'suressan'

import logging
from ..utils.utils import Utils
from ..utils.infraexceptions import *
from ctypes import *
import yaml
import os
from appfw.api.systeminfo import SystemInfo
from agent_internals import SmartAgentNotification as notif
from agent_internals import SmartAgentEntitlementEnforceMode as lic_mode
from agent_internals import SmartAgentDebugFlags as dbgflag
from agent_internals import CallHomeDebugFlags as ch_dbg
from agent_internals import SmartAgentReturnCodes as RetCode
log = logging.getLogger("sla_service")

'''
Helper class to initialize Smart licenseing agent and register platform with Cisco Smart Software Manager (CSSM).
'''


class SmartLicenseService(object):
    __singleton = None
    __singleton_init_done = False

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            #if not cls.__singleton:
            cls.__singleton = super(SmartLicenseService, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self):
        if not self.__singleton_init_done:
            try:
                self.sla_handle = None
                self.DEFAULT_PRODUCTION_URL = "https://tools.cisco.com/its/service/oddce/services/DDCEService"
                self.DEFAULT_ALHPA_URL = "http://10.22.183.117:8080/ddce/services/DDCEService"
                self.ALPHA_SOFTWARE_TAG = "regid.2013-08.com.cisco.CSR1000V,1.0_1562da96-9176-4f99-a6cb-14b4dd0fa135"
                self.IOX_SOFTWARE_TAG = "regid.2017-07.com.cisco.IOX-SOFTWARE,1.0_1325a1ae-87ad-4ba0-9602-efec524ad07c"
                self.IOX_BASE_ENTITLEMENT_TAG = "regid.2017-07.com.cisco.IOX_BASE,1.0_45bce458-349f-4b01-950e-c76b7c2fe18c"
                self.iox_base_handle = None
                self.IOX_BASE_VERSION = "1.0"
                self.secure_store_dir = Utils.getSystemConfigValue('secure_storage_server', 'encrypted_dir')
                self.smart_license_dir = os.path.join(self.secure_store_dir, "smartlicense")
                self.sla_init_config = {}
                self.sla_runtime_config = {}
                self.sla_runtime_config_file = ".sla"
                self.sla_runtime_fullpath = ""
                self.globalcb_ptr = None
                self.entitlementcb_ptr = None
                self.reg_status = 0
                self.entitlement_mode = 0
                self.load_config()
                self.sla_push_notif_list = [
                    notif.SmartAgentNotifyRegisterFailed,
                    notif.SmartAgentNotifyRegisterSuccess,
                    notif.SmartAgentNotifyIdCertExpired,
                    notif.SmartAgentNotifyIdCertRenewFail,
                    notif.SmartAgentNotifyAuthRenewSuccess,
                    notif.SmartAgentNotifyAuthRenewFailure,
                    notif.SmartAgentNotifyCommFailure,
                    notif.SmartAgentNotifyEvalExpiryWarning,
                    notif.SmartAgentNotifyEvalExpired,
                    notif.SmartAgentNotifyIdCertExpiryWarning,
                    notif.SmartAgentNotifyDeRegisterSuccess,
                    notif.SmartAgentNotifyDeRegisterFailed,
                    lic_mode.SmartAgentEnforceModeInCompliance,
                    lic_mode.SmartAgentEnforceModeOutOfCompliance,
                    lic_mode.SmartAgentEnforceModeOverage,
                    lic_mode.SmartAgentEnforceModeEval,
                    lic_mode.SmartAgentEnforceModeEvalExpired,
                    lic_mode.SmartAgentEnforceModeGracePeriod,
                    lic_mode.SmartAgentEnforceModeAuthorizationExpired,
                    lic_mode.SmartAgentEnforceModeGracePeriodExpired
                    ]
                self.load_sla_library()
                self.sla_init()
                __singleton_init_done = True
            except Exception as ex:
                log.exception("Failed to initialize smart license service")
                raise ex


    def load_sla_library(self):
        try:
            self.sla_handle = cdll.LoadLibrary("libioxsmartagent.so")
        except Exception as ex:
            log.exception("Failed to load smart agent library")
            raise ex

    def set_config(self, val):
        try:
            self.sla_runtime_config.update(val)
            with open(self.sla_runtime_fullpath, "w") as f:
                yaml.safe_dump(self.sla_runtime_config, f, default_flow_style=False)
            if "url" in val:
                self.sla_setup_custom_url(val["url"])
            log.debug("updated config file, configuration: %s", self.sla_runtime_config)
        except Exception as ex:
            log.error("Failed to set smart license agent configuration")
            raise ex



    def get_config(self):
        return self.sla_runtime_config

    def init_config(self):
        self.sla_init_config["url"] = self.DEFAULT_PRODUCTION_URL
        self.sla_init_config["enabled"] = Utils.getSystemConfigValue('smartlicense', 'enabled', True, "bool")
        self.sla_init_config["software-tag"] = self.IOX_SOFTWARE_TAG
        self.sla_init_config["loglevel"] = "error"


    def load_config(self):
        self.init_config()
        repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
        work_folder=None
        if Utils.hasSystemConfigOption("DEFAULT", "caf_work_dir"):
            work_folder = Utils.getSystemConfigValue("DEFAULT", "caf_work_dir", None)
        if work_folder is None:
            work_folder = os.path.dirname(repo_folder)

        running_config_repo = os.path.join(work_folder, "running_config")
        self.sla_runtime_fullpath = os.path.join(running_config_repo, self.sla_runtime_config_file)

        try:
            from ..runtime.runtime import RuntimeService
            self.runtime = RuntimeService.getInstance()
            self.sla_runtime_config = self.runtime.merge_configs(self.sla_init_config, self.sla_runtime_fullpath)
        except Exception as ex:
            log.error("Failed to load run time configuration. Switching back to default configuration")
            self.sla_runtime_config.update(self.sla_init_config)
        log.debug("Smart license service config: Enabled = %s, Backend URL = %s", self.sla_runtime_config["enabled"], self.sla_runtime_config["url"])

    def show_license_status(self):
        try:
            showfunc = self.sla_handle.IOxPluginShowStatus
            showfunc.restype = c_char_p
            showret = showfunc(None)
            log.debug("obtained show license status = %s", showret)
            freefunc = self.sla_handle.IOxPluginShowFree
            freefunc(None)
            return showret
        except Exception as ex:
            log.error("Failed to obtained license status.")
            raise ex

    def show_license_summary(self):
        try:
            showfunc = self.sla_handle.IOxPluginShowSummary
            showfunc.restype = c_char_p
            showret = showfunc(None)
            log.debug("obtained show license summary = %s", showret)
            freefunc = self.sla_handle.IOxPluginShowFree
            freefunc(None)
            return showret
        except Exception as ex:
            log.error("Failed to obtained license summary.")
            raise ex

    def show_license_udi(self):
        try:
            showfunc = self.sla_handle.IOxPluginShowUDI
            showfunc.restype = c_char_p
            showret = showfunc(None)
            log.debug("obtained show license UDI = %s", showret)
            freefunc = self.sla_handle.IOxPluginShowFree
            freefunc(None)
            return showret
        except Exception as ex:
            log.error("Failed to obtained license UDI.")
            raise ex

    def show_license_usage(self):
        try:
            showfunc = self.sla_handle.IOxPluginShowUsage
            showfunc.restype = c_char_p
            showret = showfunc(None)
            log.debug("obtained show license usage details = %s", showret)
            freefunc = self.sla_handle.IOxPluginShowFree
            freefunc(None)
            return showret
        except Exception as ex:
            log.error("Failed to obtained license usage details.")
            raise ex

    def show_license_all(self):
        try:
            showfunc = self.sla_handle.IOxPluginShowAll
            showfunc.restype = c_char_p
            showret = showfunc(None)
            log.debug("obtained show license all information = %s", showret)
            freefunc = self.sla_handle.IOxPluginShowFree
            freefunc(None)
            return showret
        except Exception as ex:
            log.error("Failed to obtained license all information.")
            raise ex

    def show_license_techsupport(self):
        try:
            showfunc = self.sla_handle.IOxPluginShowTechSupport
            showfunc.restype = c_char_p
            showret = showfunc(None)
            log.debug("obtained show license tech support = %s", showret)
            freefunc = self.sla_handle.IOxPluginShowFree
            freefunc(None)
            return showret
        except Exception as ex:
            log.error("Failed to obtained license tech support.")
            raise ex

    def sla_test_reset_eval_period(self):
        try:
            reset_evalfunc = self.sla_handle.SmartAgentTestResetEvaluationPeriod
            reset_evalfunc.restype = c_int
            ret_code = reset_evalfunc(None)
            log.debug("Test reset eval period returned = %s", ret_code)
            return ret_code
        except Exception as ex:
            log.error("Failed to reset eval period")
            raise ex

    def sla_debug_set(self, level):
        try:
            # set call home debug flag
            self.ch_debug_set(level)
            # set smart agent debug flag
            debug_set = self.sla_handle.IOxPluginDebugSet
            debug_set.restype = c_int
            debug_set.argtypes = [c_int]
            bitmask = 0
            if level == "all":
                bitmask = dbgflag.SMART_AGENT_DEBUG_FLAG_ALL
            elif level == "error":
                bitmask = dbgflag.SMART_AGENT_DEBUG_ERROR
            elif level == "trace":
                bitmask = dbgflag.SMART_AGENT_DEBUG_TRACE
            elif level == "debug":
                bitmask = dbgflag.SMART_AGENT_DEBUG_DEBUG
            else:
                return -1
            ret_code = debug_set(bitmask)
            log.debug("smart agent enabled debug level = %s", level)
            
            return ret_code
        except Exception as ex:
            log.error("Failed to enable smart license debug level = %s", level)
            raise ex

    def ch_debug_set(self, level):
        try:
            debug_set = self.sla_handle.ch_lib_set_debug_flag
            debug_set.restype = c_int
            debug_set.argtypes = [c_int]
            bitmask = 0

            if level == "trace":
                bitmask = ch_dbg.CH_DBG_FLAG_TRACE
            elif level == "debug" or level == "all":
                bitmask = ch_dbg.CH_DBG_FLAG_DETAIL
            else:
                bitmask = ch_dbg.CH_DBG_FLAG_ERROR
            ret_code = debug_set(bitmask)
            log.debug("Return code for setting call home debug flag %s - %s ", level, ret_code)
            return ret_code
        except Exception as ex:
            log.error("Failed to enable call home debug flag %s - %s", level, ret_code)
            raise ex

    def sla_debug_clear(self, level):
        try:
            # clear call home debug flag
            self.ch_debug_clear()
            # clear smart agent debug flag
            debug_clear = self.sla_handle.IOxPluginDebugClear
            debug_clear.restype = c_int
            debug_clear.argtypes = [c_int]
            bitmask = 0
            if level == "all":
                bitmask = dbgflag.SMART_AGENT_DEBUG_FLAG_ALL
                # set gch to none
            elif level == "error":
                bitmask = dbgflag.SMART_AGENT_DEBUG_ERROR
            elif level == "trace":
                bitmask = dbgflag.SMART_AGENT_DEBUG_TRACE
            elif level == "debug":
                bitmask = dbgflag.SMART_AGENT_DEBUG_DEBUG
            else:
                return -1
            ret_code = debug_clear(bitmask)
            log.debug("smart agent disabled debug level = %s", level)
            return ret_code
        except Exception as ex:
            log.error("Failed to disable smart license debug level = %s", level)
            raise ex

    def ch_debug_clear(self):
        try:
            debug_clear = self.sla_handle.ch_lib_set_debug_flag
            debug_clear.restype = c_int
            debug_clear.argtypes = [c_int]
            bitmask = ch_dbg.CH_DBG_FLAG_ERROR
            ret_code = debug_clear(bitmask)
            log.debug("Return code for clearing call home debug flag - %s ", ret_code)
            return ret_code
        except Exception as ex:
            log.error("Failed to clear call home debug flag - %s", ret_code)
            raise ex

    def sla_renew_idcert(self):
        try:
            renew_id = self.sla_handle.SmartAgentRenewIdCert
            renew_id.restype = c_int
            ret_code = renew_id(None)
            log.debug("smart agent ID cert renewal return code = %s", ret_code)
            return ret_code
        except Exception as ex:
            log.error("Failed to renew smart license ID certificate")
            raise ex

    def sla_renew_auth(self):
        try:
            renew_auth = self.sla_handle.SmartAgentRenewAuth
            renew_auth.restype = c_int
            ret_code = renew_auth(None)
            log.debug("smart agent entitlement authorization renewal return code = %s", ret_code)
            return ret_code
        except Exception as ex:
            log.error("Failed to renew smart license authorization")
            raise ex

    def sla_create_and_request_entitlement(self, tagname, version):
        #Invoke this fn in plugin and store the handle mapped to the tag:version
        # SmartAgentReturnCodes_t IOxPluginCreateAndRequestEntitlement(const char *tagName, const char *tagVersion, int *ret_handle)
        try:
            entitlement_req = self.sla_handle.IOxPluginCreateAndRequestEntitlement
            entitlement_req.restype = c_int
            entitlement_req.argtypes = [c_char_p, c_char_p, POINTER(c_int)]
            ret_handle = c_int()
            ret_code = entitlement_req(tagname, version, byref(ret_handle))
            if ret_code == RetCode.SmartAgentSuccess:
                log.info("Sucessfully requested entitlement for %s:%s", tagname, version)
            else:
                log.error("Failed to create and request entitlement for %s:%s - return code %s", tagname, version, ret_code)
            return ret_handle
        except Exception as ex:
            log.error("Failed to create and request entitlement")
            raise ex

    def sla_release_and_destroy_entitlement(self, handle):
        try:
            rel_entitlement_req = self.sla_handle.IOxReleaseAndDestroyEntitlement
            rel_entitlement_req.restype = c_int
            rel_entitlement_req.argtypes = [c_int]
            ret_code = rel_entitlement_req(self.iox_base_handle)
            if ret_code == RetCode.SmartAgentSuccess:
                log.info("Sucessfully released and destroyed entitlement for handle %s", handle)
            else:
                log.error("Failed to release and destroy entitlement for handle %s - return code %s", handle, ret_code)
            return ret_code
        except Exception as ex:
            log.error("Failed to create and request entitlement")
            raise ex

    def sla_setup_custom_url(self, url):
        try:
            # for alpha environments
            setcustomurl = self.sla_handle.IOxPluginCommPluginSetURL
            setcustomurl.restype = c_int
            setcustomurl.argtypes = [POINTER(c_char)]
            ret_code = setcustomurl(c_char_p(url))
            if ret_code != RetCode.SmartAgentSuccess:
                log.error("Failed to set custom development environment for smart agent")
            return ret_code
        except Exception as ex:
            log.error("Failed to set custom development environment for smart agent")
            raise ex


    # If val = 1; setup testroot CA
    # If val = 0; unset testroot CA
    def sla_setup_testrootca(self, val):
        try:
            settestdev = self.sla_handle.SmartAgentTestSetRootCA
            settestdev.restype = c_int
            settestdev.argtypes = [c_int]
            ret_code = settestdev(val)
            if ret_code == RetCode.SmartAgentSuccess:
                log.info("Sucessfully setup root CA with smart agent, enabled = %s, ret_code = %s", val, ret_code)
            else:
                log.error("Failed to setup root CA with smart agent with ret code = %s", ret_code)
            return ret_code
        except Exception as ex:
            log.error("Failed to setup root CA with smart agent")
            raise ex


    def sla_register(self, token, force):
        try:
            # register sla
            regfunc = self.sla_handle.SmartAgentIdTokenRegister
            regfunc.restype = c_int
            regfunc.argtypes = [POINTER(c_char), c_bool]
            reg_token_p = c_char_p(token)
            ret_code = regfunc(reg_token_p, force) # pass 0 in second param for not forcing re-registration with backend
            return ret_code
        except Exception as ex:
            log.error("Failed to register IOx with license management portal")
            raise ex


    def sla_deregister(self):
        try:
            dereg = self.sla_handle.SmartAgentDeRegister
            dereg.restype = c_int
            ret_code = dereg(None)
            return ret_code
        except Exception as ex:
            log.error("Failed to deregister IOx with license management portal")
            raise ex

    def notif_to_str(self, notif_type):
        try:
            notif_to_str_conv = self.sla_handle.SmartAgentNotificationToString
            notif_to_str_conv.argtypes = [c_int]
            notif_to_str_conv.restype = c_char_p
            return notif_to_str_conv(notif_type)
        except Exception as ex:
            log.error("Failed to convert notification type to description")
            raise ex

    def enforce_mode_to_str(self, notif_type):
        try:
            enforce_mode_to_str_conv = self.sla_handle.SmartAgentEnforceModeToStr
            enforce_mode_to_str_conv.argtypes = [c_int]
            enforce_mode_to_str_conv.restype = c_char_p
            return enforce_mode_to_str_conv(notif_type)
        except Exception as ex:
            log.error("Failed to convert enforce mode type to description")
            raise ex

    def sla_return_code_to_str(self, ret_code):
        try:
            ret_code_to_str_conv = self.sla_handle.SmartAgentGetErrorMessage
            ret_code_to_str_conv.argtypes = [c_int]
            ret_code_to_str_conv.restype = c_char_p
            return ret_code_to_str_conv(ret_code)
        except Exception as ex:
            log.error("Failed to convert return code type to description")
            raise ex

    def global_callback(self, notif_type):
        try:
            log.debug("Received global notifcation type = %s", notif_type)
            notif_str = self.notif_to_str(notif_type)
            if notif_str is not None:
                log.debug("Received global notification description = %s", notif_str)
            self.reg_status = notif_type
        except Exception as ex:
            log.exception("Failed to execute global callback for notification type = %s", notif_type)

    def entitlement_callback(self, notif_type, entitlement_tag, notif_data_type):
        try:
            log.debug("Received entitlement notifcation = %s, tag = %s, data type = %s", notif_type, entitlement_tag, notif_data_type)
            enforce_mode_str = self.enforce_mode_to_str(notif_data_type)
            if enforce_mode_str is not None:
                log.debug("Received entitlement enforce mode description = %s", enforce_mode_str)

            self.entitlement_mode = notif_data_type
        except Exception as ex:
            log.exception("Failed to execute entitlement callback for notification type = %s", notif_type)

    def api_set_global_callback(self):
        try:
            GLOBALCB = CFUNCTYPE(None, c_int)
            self.globalcb_ptr = GLOBALCB(self.global_callback)

            gcbfunc = self.sla_handle.IOxSetGlobalCB
            gcbfunc.restype = c_int
            gcbfunc.argtypes = [GLOBALCB]
            ret_code = gcbfunc(self.globalcb_ptr)
            if ret_code == 0:
                log.info("Smart licensing global notification callback successfully registered")
            else:
                log.error("Smart licensing global notification callback registration failed, return code = %s", ret_code)
                raise SLASetGlobalCBFailed("Smart licensing global notification callback registration failed, return code = %s", ret_code)
        except Exception as ex:
            log.error("Failed to register smart license agent global notification callback %s", str(ex))
            raise ex

    def api_set_entitlement_callback(self):
        try:
            ENTITLEMENTCB = CFUNCTYPE(None, c_int, c_char_p, c_int)
            self.entitlementcb_ptr = ENTITLEMENTCB(self.entitlement_callback)

            entitlefunc = self.sla_handle.IOxSetEntitlementCB
            entitlefunc.argtypes = [ENTITLEMENTCB]
            ret_code = entitlefunc(self.entitlementcb_ptr)
            if ret_code == 0:
                log.info("Smart licensing entitlement notification callback successfully registered")
            else:
                log.error("Smart licensing entitlement notification callback registration failed, return code = %s", ret_code)
                raise SLASetEntitlementCBFailed("Smart licensing entitlement callback registration failed, return code = %s", ret_code)
        except Exception as ex:
            log.error("Failed to register smart license agent entitlement notification callback %s", str(ex))
            raise ex

    def sla_init(self):
        try:
            # Setup env varibles needed for Smart License agent
            self.sla_setup_env()

            # setup global notifications callback - Invoke this fn in plugin:
            # >>> SmartAgentReturnCodes_t IOxSetGlobalCB(IOxSmartAgentGlobalCB_t globalcb)
            # Implement the callback "globalcb" here to handle different notifications
            self.api_set_global_callback()
            # setup entitlement notification callback
            # SmartAgentReturnCodes_t IOxSetEntitlementCB (IOxSmartAgentEntitlementCB_t entitlementcb)
            # Implement the callback "entitlementcb" here to handle different notifications
            self.api_set_entitlement_callback()
            initfunc = self.sla_handle.SmartAgentInit
            initfunc.restype = c_int
            ret_code = initfunc(None)
            if ret_code == 0:
                log.info("Smart licensing agent successfully initialized")
            else:
                log.error("Smart licensing agent initialization failed, return code = %s", ret_code)
                raise SLAAgentInitFailed("Smart licensing agent initialization failed, return code = %s", ret_code)

            read_begin = self.sla_handle.IOxPluginSendPlatEvtConfigReadBegin
            read_begin.restype = c_int
            ret_code = read_begin(None)
            if ret_code == 0:
                log.info("Sent read config BEGIN event to smart licensing agent")
            else:
                log.error("Failed to send read config BEGIN event to smart licensing agent, ret code = %s", ret_code)
                raise SLAPlatformBeginConfigReadFailed("Failed to send read config BEGIN event to smart licensing agent, ret code = %s", ret_code)

            read_done = self.sla_handle.IOxPluginSendPlatEvtConfigReadDone
            read_done.restype = c_int
            ret_code = read_done(None)
            if ret_code == 0:
                log.info("Sent read config DONE event to smart licensing agent")
            else:
                log.error("Failed to send read config DONE event to smart licensing agent, ret code = %s", ret_code)
                raise SLAPlatformEndConfigReadFailed("Failed to send read config DONE event to smart licensing agent, ret code = %s", ret_code)
                
            self.platform_ready_event()
        except Exception as ex:
            log.exception("Failed to initialize IOx license service")
            raise ex

    def sla_setup_env(self):
        try:
            os.environ["PLATFORM_PRODUCT_ID"] = SystemInfo.get_productid()
            os.environ["PLATFORM_SERIAL_ID"] = SystemInfo.get_systemid()
            os.environ["PLATFORM_UUID"] = SystemInfo.get_system_uuid()
            os.environ["PLATFORM_SOFTWARE_TAG"] = self.sla_runtime_config["software-tag"]
            if not os.path.isdir(self.smart_license_dir):
                os.makedirs(self.smart_license_dir)
            os.environ["PLATFORM_TRUSTED_STORE"] = self.smart_license_dir
        except Exception as ex:
            log.error("Failed to setup environment parameters for Smart license agent")
            raise ex


    def platform_ready_event(self):
        try:            
            # send plat sys is initialized event to smart agent
            sysinit_ev = self.sla_handle.IOxPluginSendPlatEvtSystemIsInit
            sysinit_ev.restype = c_int
            ret_code = sysinit_ev(None)
            if ret_code == 0:
                log.info("Sent platform initialized event to smart licensing agent")
            else:
                log.error("Failed to send platform initialized event to smart licensing agent, ret code = %s", ret_code)
                raise SLAPlatformInitEventFailed("Failed to send platform initialized event to smart licensing agent, ret code = %s", ret_code)

            # Persist smart agent config across reboots
            cfgpersist = self.sla_handle.IOxPluginSendPlatEvtConfigPersisted
            cfgpersist.restype = c_int
            ret_code = cfgpersist(None)
            if ret_code == 0:
                log.info("smart licensing agent config persist event sent")
            else:
                log.error("Failed to send smart licensing agent config persist event sent, ret code = %s", ret_code)
                raise SLAPersistConfigEventFailed("Failed to send smart licensing agent config persist event sent, ret code = %s", ret_code)
        except Exception as ex:
            log.error("Failed to initialize IOx license service")
            raise ex

    def start(self):
        try:
            self.iox_base_handle = self.sla_create_and_request_entitlement(self.IOX_BASE_ENTITLEMENT_TAG, self.IOX_BASE_VERSION)
            # setup configured call home url
            self.sla_setup_custom_url(self.sla_runtime_config["url"])
            # setup loglevel for SmartAgent
            self.sla_debug_set(self.sla_runtime_config["loglevel"])
        except Exception as ex:
            log.exception("Failed to start smart license subsystem")
            raise ex


    def enable_smart_license(self):
        # smart licensing is enabled by default
        pass

    def disable_smart_license(self):
        pass

    def stop(self):
        #  services will release entitlements that have been created before stopping smart license service
        try:
            if self.iox_base_handle is not None:
                self.sla_release_and_destroy_entitlement(self.iox_base_handle)

            stopfunc = self.sla_handle.SmartAgentShutdown
            stopfunc.restype = c_int
            ret_code = stopfunc(None)
            if ret_code == 0:
                log.info("Smart licensing agent successfully shutdown")
            else:
                log.error("Smart licensing agent shutdown failed, return code = %s", ret_code)
                raise SLAShutdownFailed("Smart licensing agent shutdown failed, return code = %s", ret_code)
        except Exception as ex:
            log.exception("Failed to stop smart license subsystem")
            raise ex


    def sla_test_expire_job(self, jobname, seconds):
        try:
            test_expire_job = self.sla_handle.SmartAgentTestExpireJobName
            test_expire_job.argtypes = [POINTER(c_char), c_int]
            jobstr = c_char_p(jobname)
            test_expire_job.restype = c_int
            ret_code = test_expire_job(jobstr, c_int(seconds))
            log.debug("test expiry job - %s in %s seconds, return code = %s", jobname, seconds, ret_code)
            return ret_code
        except Exception as ex:
            log.error("Failed to test expire job - %s", jobname)
            raise ex

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = SmartLicenseService(*args)
        return cls.__singleton
