'''
Created on Apr 02, 2015

@author: utandon

Copyright (c) 2015-2016 by Cisco Systems, Inc.
All rights reserved.
'''

import shutil
import os
from ..utils import utils
from ..utils.utils import Utils
from   ..utils.commandwrappers import *
import tempfile
import os
import jsonschema
import shutil
import hashlib
import struct
import re
import json
from ..utils.infraexceptions import *
import tarfile
import logging
from ..runtime.platformcapabilities import PlatformCapabilities
from ..runtime.descriptormetadata import descriptor_metadata_wrapper

log = logging.getLogger("runtime.hosting")

def _validate_layers_integrity(layers_dir, layer_list_json_file):
    """
    compare the sha of layers specified in app's rootfs/layer_list.json with those in $(caf_work_dir)/layers directory.
a    """
    if not os.path.exists(layer_list_json_file):
        log.exception("file %s doesn't exist" %(layer_list_json_file))
        raise IntegrityCheckFailedError("layer list json file doesn't exist")
    try:
        with open(layer_list_json_file, "r") as f:
            mani_contents = json.loads(f.read())
            for fs_layer in mani_contents.get("fsLayers"):
                log.debug("layer:%s" %(fs_layer))
                layer_path = os.path.join(layers_dir, fs_layer, "layer_contents/layer.tar")
                if os.path.exists(layer_path):
                    with open(layer_path) as fobj:
                        if fs_layer != Utils.generate_sha_digest(fobj, "SHA256"):
                            log.exception("SHA digest check is failed")
                            raise IntegrityCheckFailedError("SHA digest is not matching for layer %s " % fs_layer)
                        else:
                            log.debug("layer %s verification done" %(fs_layer))
                else:
                    log.exception("layer %s doesnt' exist" %(layer_path))
                    raise IntegrityCheckFailedError("layer doesn't exist")
    except Exception as e:
        log.exception("Error while opening layer list json file")
        raise e
    log.debug("layers sha verification passed")
    return

def _validate_package_integrity(manifest_contents, source_dir, exclude_complete_check = False):
    """
    Match the contents of manifest file with the generated values and throws in case of any mis-match occurs.
    """
    pattern = re.compile(
        '(?P<hashing_technique>SHA1|SHA256)[(](?P<filename>[a-zA-Z0-9-_.\/]+)[)]=\s(?P<shadigest>[A-Fa-f0-9]+)')
    temp = source_dir
    filenamelist = []
    # validating the entries in package manifest
    for line in str(manifest_contents).splitlines():
        # Ignoring the empty and commented lines
        line = line.strip()
        if line == "" or line.startswith("#"):
            continue
        match = pattern.match(line)
        if match:
            hashing_technique = match.group("hashing_technique")
            filename = match.group("filename")
            filenamelist.append(filename)
            shadigest = match.group("shadigest")
            try:
                with open(os.path.join(temp, filename)) as fobj:
                    if shadigest != Utils.generate_sha_digest(fobj, hashing_technique):
                        log.exception("SHA digest check is failed")
                        raise IntegrityCheckFailedError("SHA digest is not matching for file %s " % filename)
            except Exception as e:
                log.exception("Error while doing sha verification")
                raise e
        else:
            log.exception("Invalid digest value")
            raise IntegrityCheckFailedError("Invalid entry [ %s ] in package manifest file " % line)

    if exclude_complete_check:
        return
    # checking for all files in package has entry in package manifest
    for subdir, dirs, files in os.walk(temp):
        for file in files:
            real_path = os.path.relpath(os.path.join(subdir, file), temp)
            if real_path not in filenamelist and real_path != utils.PACKAGE_MANIFEST_NAME and real_path != utils.APP_CERTIFICATE_NAME and real_path != utils.APP_SIGNATURE_NAME:
                log.exception("Entry for a file is missing in %s "% utils.PACKAGE_MANIFEST_NAME)
                raise IntegrityCheckFailedError("For the %s file digest not existing in the package manifest ( %s )" % (
                real_path, utils.PACKAGE_MANIFEST_NAME))



class PackageInterface(object):    
    """
    Base class for all the package type 
    """
    def __init__(self, archiveFile = None, dest_dir=None, conIni=None, appmanifest_file=None,
                 appmanifest=None, metadatafile=None, appconfig_file=None,
                 configschema_file=None, check_integrity=False, check_signature=False,
                 check_mandatory_files=False, check_descriptor_schema=False,
                 check_app_compatibility=True, check_descriptor_semantics=True, 
                 sign_model="None", app_group=False):
        """
        Will extract the archiveFile only if dest_dir has no descriptor files
        e.g. package.yaml or cartridge.yaml
        If dest_dir already has any descriptor file, will use that as the 
        contents of the package and will not extract the sent archive
        """
        self.tempArchivePath = archiveFile
        self.tempConIni = conIni
        self.appmanifest_file = appmanifest_file
        self.appmanifest = appmanifest
        self.metadatafile = metadatafile
        self.appconfig_file = appconfig_file
        self.configschema_file = configschema_file 
        self.tempDir = None
        self.check_integrity = check_integrity
        self.check_signature = check_signature
        self.check_mandatory_files = check_mandatory_files
        self.check_descriptor_schema = check_descriptor_schema
        self.check_app_compatibility = check_app_compatibility
        self._dest_dir = dest_dir
        self.check_descriptor_semantics = check_descriptor_semantics
        self.cert_extension = 'cert'
        self.app_metadata = None
        self.verified_signature_model = "None"
        self.sign_model = sign_model
        self.app_group = app_group
        if not os.path.exists(dest_dir):
            os.makedirs(dest_dir)

    @property
    def dest_dir(self): 
        return self._dest_dir

    @dest_dir.setter
    def dest_dir(self, value): 
        self._dest_dir = value

    def move_extracted_contents(self, target):
        for f in os.listdir(self.dest_dir):
            if os.path.exists(os.path.join(target,f)):
                shutil.rmtree(os.path.join(target,f), ignore_errors=True)
            shutil.move(os.path.join(self.dest_dir,f), os.path.join(target,f))
        os.rmdir(self.dest_dir) 
        self._dest_dir = target

    def _load_manifest(self):
        """
        Loads the manisfest file in package
        """
        if self.dest_dir:
            self.appmanifest_file = utils.Utils.get_app_manifest_filename_in_path(self.dest_dir)
            if not self.appmanifest_file:
                log.error("App descriptor file not found in path: %s" % self.dest_dir)
                raise MandatoryFileMissingError("App descriptor file not found in extracted dir")
        elif self.tempArchivePath:
            self.appmanifest_file = utils.Utils.get_app_manifest_filename(self.tempArchivePath)
            if not self.appmanifest_file:
                log.error("App descriptor file not found in archive")
                raise MandatoryFileMissingError("App descriptor file not found in archive")
        else:
            log.error("App descriptor file not found archive")
            raise MandatoryFileMissingError("App descriptor file not found in extracted dir")

        try:
            self.metadatafile = os.path.join(self.dest_dir, self.appmanifest_file)
            self.appmanifest = descriptor_metadata_wrapper(None, self.metadatafile)
        except Exception as cause:
             log.exception("Unable to load %s  from  archive : %s" % (self.appmanifest_file, str(cause)))
             raise MalformedManifestError("Unable to extract data from Connector's archive : %s" % str(cause))

    def get_app_manifest(self):
        return self.get_file_contents(self.metadatafile)

    def get_appmanifest(self):
        if self.appmanifest is None:
            self._load_manifest()
        return self.appmanifest

    def get_app_config(self):
        """
        Returns the app config in the package
        """
        return self.get_file_contents(self.appconfig_file)

    def validate_package(self):
        """
        Validates the package
        """
        pass

    def get_metadatafile(self):
        """
        extracts the metadata file in archive
        """
        pass

    # Always do verification after extraction to confirm the file is not maliciious like extracted to some symlinked dir on host.
    # Refer bug - CSCvr02052
    # Recommend to use linux tar extract all utility which is safer.
    def extract_to_destination(self, fn, dest_path):
        """
        extract fn to dest_path
        """
        pass

    def extractall_to_destination(self, dest_path):
        """
        extracts the archive to specified destination
        """
        pass
    
    def exists_in_package(self, path):
        """
        Returns True or False based on the path exists in package or not
        """
        if os.path.exists(self.dest_dir):
            search_path = os.path.join(self.dest_dir, path)
            return os.path.exists(search_path)
        else:
            return False

    def get_file_contents(self, path):
        """
        Returns the contents of the specified file in archive
        """
        contents = None
        file_path =  os.path.join(self.dest_dir, path)
        try:
            with open(file_path, 'r') as f:
                contents = f.read()
                log.debug("File %s contents: %s", file_path, contents)
        except Exception as cause:
            log.exception("Unable to read %s  Error:%s" % (file_path, str(cause)))
            log.debug("File %s does not exists", file_path)
        return contents

    def get_app_config_schema(self):
        """
        returns  the app config schema
        """
        return self.get_file_contents(self.configschema_file)

    def get_startup_template(self):
        """
        Return app's startup template
        """
        if self.appmanifest:
            return self.appmanifest.uiStartupTemplate

        return None

    def get_html_template(self):
        """
        Return app's html template
        """
        if self.appmanifest:
            return self.appmanifest.uiHtmlTemplate

        return None

    def get_list_of_files(self):
        """
        List out the all archive files and returns the same
        """
        pass

    def close(self):
        self.tempArchivePath = None
        if self.tempDir:
           shutil.rmtree(self.tempDir, ignore_errors=True)

    def is_exists(self, name, path):
        """
        Check for existance of the file in package
        """
        if os.path.exists(os.path.join(path, name)):
            return True
        else:
            return False

    #Function should not be overridden
    def check_for_mandatory_files(self, root_path):
        """
        Check for the availability of app files in package.
        If the files are not present throws MandatoryMetadataError.
        """
        log.debug("Looking for the presence of all mandatory files in given path %s"% root_path)
        if not (self.is_exists(utils.APP_DESCRIPTOR_NAME, root_path) or self.is_exists(
                utils.INI_MANIFEST_NAME, root_path) or self.is_exists(utils.YAML_MANIFEST_NAME, root_path)
                or self.is_exists(utils.CARTRIDGE_MANIFEST_NAME, root_path) or self.is_exists(utils.SW_UPDATE_METADATA_FILE, root_path)):
            raise MandatoryFileMissingError("App/Service descriptor file is missing in package")
        if self.check_integrity:
            if not self.is_exists(utils.PACKAGE_MANIFEST_NAME, root_path):
                raise MandatoryFileMissingError(
                "Package-Manifest file %s is not found in package" % utils.PACKAGE_MANIFEST_NAME)
        if self.check_signature:
            if not self.is_exists(utils.APP_CERTIFICATE_NAME, root_path):
                if not self.is_exists(utils.APP_SIGNATURE_NAME, root_path):
                    raise MandatoryFileMissingError(
                    "App signature validation is enabled. App signature file %s or %s not found in package" %
                    (utils.APP_CERTIFICATE_NAME, utils.APP_SIGNATURE_NAME))
                else:
                    log.debug("Package signature file %s found" % utils.APP_SIGNATURE_NAME)
                    self.cert_extension = 'sign'
            else:
                log.debug("Package signature file %s found" % utils.APP_CERTIFICATE_NAME)
                self.cert_extension = 'cert'
        pc = PlatformCapabilities.getInstance()
        compression_types = pc.supported_compression_types

        if (not self.is_exists(utils.INNER_PACKAGE_NAME, root_path)) and \
            not ("xz" in compression_types and self.is_exists(utils.INNER_PACKAGE_NAME_XZ)):
            raise MandatoryFileMissingError("No artifact file is found in package %s or %s  " % (utils.INNER_PACKAGE_NAME, utils.INNER_PACKAGE_NAME_XZ))


    #Function should not be overridden
    def validate_package_integrity(self, root_path):
        """
        Parse the package manifest file and do the integrity check on package from the contents of manifest file.
        """
        log.debug("Validating package integrity")
        if self.is_exists(utils.PACKAGE_MANIFEST_NAME, root_path):
            manifest_contents = self.get_file_contents(utils.PACKAGE_MANIFEST_NAME)
            _validate_package_integrity(manifest_contents, root_path)
        else:
            log.exception("Error while validating package integrity")
            raise MandatoryFileMissingError("Package-Manifest file %s is not found in package" % utils.PACKAGE_MANIFEST_NAME)

    #Function should not be overridden
    def verify_sign_package_signature(self, root_path):
        import subprocess
        log.debug("Verifying Package Signature")
        output = 0
        if not self.is_exists(utils.PACKAGE_MANIFEST_NAME, root_path):
            log.error("Package Manifest file not found while verifying package signature")
            raise MandatoryFileMissingError("Package Manifest file %s "
                  "not found in the package" % utils.PACKAGE_MANIFEST_NAME)
        if not self.is_exists(utils.APP_SIGNATURE_NAME, root_path):
            log.error("Package Signature file not found while verifying package signature")
            raise MandatoryFileMissingError("Package Signature file %s "
                  "not found in the package" % utils.APP_SIGNATURE_NAME)
        verify_sign_script = os.path.join(Utils.getScriptsFolder(), "verify_sign.sh")
        manifest_path = os.path.join(root_path, utils.PACKAGE_MANIFEST_NAME)
        certificate_path = os.path.join(root_path, utils.APP_SIGNATURE_NAME)

        sign_model = self.sign_model
        if sign_model == "None":
            raise MandatoryDataMissingError("Package Verification Signature model is not present"
                                            "in the platform to verify")

        cmd = [verify_sign_script, sign_model, manifest_path, certificate_path]
        try:
            output = subprocess.check_output(cmd)
        except subprocess.CalledProcessError as c:
            log.error("Package Signature verification returned failure. "
                      "CalledProcessor Exception: %s, Command: %s"
                      % (str(c.output), cmd))
            log.debug("Stack traceback", exc_info=True)
            raise InvalidSignatureError("App package signature (%s) verification "
                  "failed for package manifest file %s. Re-sign the application and then deploy again."
                  % (utils.APP_SIGNATURE_NAME, utils.PACKAGE_MANIFEST_NAME))
        except Exception as e:
            log.error("Package Signature verification returned failure. "
                      "General Exception: %s, Command: %s"
                      % (str(e), cmd))
            log.debug("Stack traceback", exc_info=True)
            raise InvalidSignatureError("App package signature (%s) verification "
                  "failed for package manifest file %s. Re-sign the application and then deploy again."
                  % (utils.APP_SIGNATURE_NAME, utils.PACKAGE_MANIFEST_NAME))
        else:
            if output:
                log.error("Package Signature verification returned failure. "
                          "Command Parser Exception: %s, Command: %s"
                          % (str(output), cmd))
                log.debug("Stack traceback", exc_info=True)
                raise InvalidSignatureError("App package signature (%s) verification "
                      "failed for package manifest file %s. Re-sign the application and then deploy again."
                      % (utils.APP_SIGNATURE_NAME, utils.PACKAGE_MANIFEST_NAME))
            else:
                log.debug("Package Signature verified successfully")
                self.verified_signature_model = sign_model

    #Function should not be overridden
    def validate_package_signature(self):
        """
        Validate the package signature.
        """
        if (self.cert_extension == 'cert'):
            try:
                from pkgsign import PackageSigning
                pkgsign = PackageSigning.getInstance()
                pkgsign.verify_package_signature(self.dest_dir)
                log.debug("Package signature validation passed")
                self.verified_signature_model = "ThirdParty"
            except Exception as e:
                log.exception("Package signature validation failed: %s" % str(e))
                raise
        else:
            self.verify_sign_package_signature(self.dest_dir)

    def validate_descriptor_schema(self, root_path):
        """
        Validate the descriptor contents counter to the schema.
        """
        log.debug("Validating descriptor schema")
        if self.is_exists(utils.APP_DESCRIPTOR_NAME, root_path):
            descriptor_file = os.path.join(root_path,utils.APP_DESCRIPTOR_NAME)
        elif self.is_exists(utils.YAML_MANIFEST_NAME, root_path):
            descriptor_file = os.path.join(root_path,utils.YAML_MANIFEST_NAME)
        elif self.is_exists(utils.INI_MANIFEST_NAME, root_path):
            log.exception("Schema validation is not avail for %s descriptor file"% utils.INI_MANIFEST_NAME)
            return
        else:
            raise MandatoryFileMissingError("App/Service descriptor file is missing in package")
        descriptor_dict = utils.Utils.read_yaml_file(descriptor_file)
        version = None
        if "descriptor-schema-version" in descriptor_dict:
            version = descriptor_dict["descriptor-schema-version"]
        elif "manifest-version" in descriptor_dict:
            version = descriptor_dict["manifest-version"]
        if version:
            # Convert to string, since our key is a string
            version = str(version)
            schema_file = self._get_schemafile(version)
            if schema_file:
                log.debug("Schema file %s and version %s"% (schema_file, version))
                complete_schema_path = os.path.join(utils.Utils.getConfigFolder(), schema_file)
            else:
                return
        else:
            raise MandatoryDataMissingError("Descriptor schema version is a mandatory field for validating schema")
        if os.path.exists(complete_schema_path) and os.path.isfile(complete_schema_path):
            with open(complete_schema_path, "r") as f:
                schema_contents = f.read().strip()

            #Make sure that empty schema contents will validate any descriptor contents as true
            if schema_contents and schema_contents != "":
                schema_dict = json.loads(schema_contents.strip())
                if descriptor_dict:
                    try:
                        log.debug("Validating the descriptor contents with schema")
                        jsonschema.validate(descriptor_dict, schema_dict)
                    except jsonschema.ValidationError as ex:
                        # Validation is not fine. Lets get more details
                        v = jsonschema.Draft4Validator(schema_dict)
                        errors = sorted(v.iter_errors(descriptor_dict), key=lambda e: e.path)
                        raise SchemaValidationError(validation_errors=errors)
                else:
                    log.exception("No data available in %s yaml file to compare with schema"%descriptor_file)
                    raise MandatoryDataMissingError("No data available in %s yaml file to compare with schema"%descriptor_file)
            else:
                log.exception("No schema contents are found")
                raise MandatoryDataMissingError("No schema contents are found")
        else:
            log.exception("Schema file %s is not exist or is not a file."%schema_file)
            raise MandatoryFileMissingError("Schema file %s is not exist or is not a file."%schema_file)

    def _get_schemafile(self, version):
        pc = PlatformCapabilities.getInstance()
        schemas = pc.supported_app_schemas
        for schema in schemas:
            if schema["version"] == version:
                if "validator_schema" in schema:
                    log.debug("Schema %s found for version %s"% (schema["validator_schema"], version))
                    return schema["validator_schema"]
                else:
                    log.debug("No schema defined for version %s"% (version))
                    return None
        raise MandatoryFileMissingError("Given descriptor version %s is not supported." % version)

    def extract_artifact_xzfile(self, src_file, dest_path, checkabspath=True):

        import contextlib
        import lzma
        import tarfile

        with contextlib.closing(lzma.LZMAFile(src_file)) as xz:
            with tarfile.open(fileobj=xz)as f:
                if checkabspath is not None:
                    self._check_for_absolutepaths(f, clear_cache=True)
                rval, errcode = tar_extractall(src_file, dest_path)
                if errcode != 0:
                    log.error("Failed to extract artifact .xz file - error: %s", str(rval))
                    raise Exception("Failed to extract artifact .xz file - error: %s", str(rval))

    def extract_inner_envelope(self, package_extracted_path):
        """
        :param package_extracted_path: Path where parent archive is extracted (which will contain artifact archive in it)
        :return: None
        By assuming that app artifact archive will be present and extracting the all contents of
         inner app archive to the root of parent archive extracted path.
        """
        log.debug("Extracting inner envelope and deleting it once extracted!")
        pc = PlatformCapabilities.getInstance()
        compression_types = pc.supported_compression_types

        if utils.INNER_PACKAGE_NAME in os.listdir(package_extracted_path):
            with tarfile.open(os.path.join(package_extracted_path, utils.INNER_PACKAGE_NAME), errors='ignore') as tar:
                try:
                    self._check_for_absolutepaths(tar, clear_cache=True)
                    #Dont use python tarfile extractall(). It has vulnerability CSCvr02052
                    rval, errcode = tar_extractall(os.path.join(package_extracted_path, utils.INNER_PACKAGE_NAME), package_extracted_path)
                    if errcode != 0:
                        log.error("Failed to extract inner eve")
                except struct.error:
                    log.exception("Invalid package format : %s"% utils.INNER_PACKAGE_NAME)
                    raise PackageInvalidError("Invalid package format : %s"% utils.INNER_PACKAGE_NAME)
            os.remove(os.path.join(package_extracted_path, utils.INNER_PACKAGE_NAME))

        elif "xz" in compression_types and utils.INNER_PACKAGE_NAME_XZ in os.listdir(package_extracted_path):
            try:
                extract_srcfile = os.path.join(package_extracted_path, utils.INNER_PACKAGE_NAME_XZ)
                self.extract_artifact_xzfile(extract_srcfile, package_extracted_path, checkabspath=True)
                os.remove(os.path.join(package_extracted_path, utils.INNER_PACKAGE_NAME_XZ))
            except struct.error:
                msg = ("Invalid package format : %s" % utils.INNER_PACKAGE_NAME_XZ)
                log.exception(msg)
                raise PackageInvalidError(msg)

        else:
            log.exception("Inner envelope %s is not found in path %s"% (utils.INNER_PACKAGE_NAME, package_extracted_path))

    #Passing tar file as input and opening it again will be a costly operation for us.
    #so that we are expecting tar_obj as input.(Recommended to call this method inside with block)
    def _check_for_absolutepaths(self, tar_obj, clear_cache=False):
        """
        Check for the any absolute paths are present inside the archive.
        If any absolute paths present then it will raise the exception
        """
        log.debug("Checking for the absolute path in the archive cache %s" % clear_cache)
        pattern = re.compile("^[^(\/.)][a-zA-Z0-9-/_]+([.][a-zA-Z0-9-/_]+)*$")
        fobj = tar_obj.next()
        while fobj is not None:
            #log.debug("Filename:  %s "  % fobj.name)
            if pattern.match(fobj.name) and fobj.name[0] == "/" or fobj.name.find("../") != -1:
                log.error("Absolute file path %s is not allowed inside the archive"% fobj.name)
                raise PackageInvalidError("Absolute file path %s is not allowed inside the archive"% fobj.name)
            if clear_cache:
                #https://stackoverflow.com/questions/21039974/high-memory-usage-with-pythons-native-tarfile-lib
                tar_obj.members = []
            fobj = tar_obj.next()

    #Passing tar file as input and opening it again will be a costly operation for us.
    #so that we are expecting tar_obj as input.(Recommended to call this method inside with block)
    def exists_in_tar(self, tar_obj, filename):
        """
        Check for the any absolute paths are present inside the archive.
        If any absolute paths present then it will raise the exception
        """
        log.debug("Checking for the %s in the archive" % filename)
        fobj = tar_obj.next()
        while fobj is not None:
            if fobj.name == filename:    
                log.debug("Filename %s exists in archive"% fobj.name)
                return True
            #https://stackoverflow.com/questions/21039974/high-memory-usage-with-pythons-native-tarfile-lib
            tar_obj.members = []
            fobj = tar_obj.next()
        return False


    def _check_descriptor_semantics(self, metadata):
        """
        Check for the equality between network asked to port-mapping defined under provides.
        Eg:
        #Valid ask :
        networks:
            -
                interface-name: eth0
                ports:
                    tcp: [9000]
        provides:
            -
                id: <service_id>
                api-version: <version number>
                port-mapping: ["eth0:tcp:9000"]
        #Invalid ask :
        networks:
            -
                interface-name: eth0
                ports:
                    tcp: [9000]
        provides:
            -
                id: <service_id>
                api-version: <version number>
                port-mapping: ["eth0:tcp:8080"]
        """
        resources = metadata.resources
        provides = metadata.provides
        if 'network' not in resources:
            log.error("No network section found for the service")
            log.debug("Checking for none of the provided services should got the port-mapping info about them")
            for service in provides:
                if "port-mapping" in service:
                    log.error("There is no network asked by service bundle, "
                              "so expecting that no port-mapping for provided services")
                    raise MandatoryDataMissingError("There is no network asked by service bundle, "
                                    "so expecting that no port-mapping for provided services")
            return
        networks = resources["network"]
        ports_matched = [] #To check service asking for same to port to listen
        for service in provides:
            if "port-mapping" in service:
                network_info = utils.Utils.parse_service_provide_portmpping(service["port-mapping"])
                log.debug("Parsed port-mapping info is: %s"%network_info)
                for ser_interface, ser_port_dict in network_info.iteritems():
                    is_inf_present = False
                    for network in networks:
                        if ser_interface == network['interface-name']:
                            is_inf_present = True
                            nw_ports = network['ports']
                            for ser_port_type, ser_ports in ser_port_dict.iteritems():
                                if ser_port_type in nw_ports:
                                    nw_ports_list = nw_ports[ser_port_type]
                                    intersection = set(nw_ports_list).intersection(set(ser_ports))
                                    if set(intersection) != set(ser_ports):
                                        log.error("Asked ports are not listed in network section:ports in section: %s,ports in mapping given:%s"%(nw_ports_list, ser_ports))
                                        raise InvalidDescriptorError("Asked ports are not listed in network section:ports in section: %s,ports in mapping given:%s"%(nw_ports_list, ser_ports))
                                    else:
                                        if not set(ports_matched).intersection(set(ser_ports)):
                                            ports_matched.extend(ser_ports)
                                        else:
                                            log.error("Two services are trying to get map to the same port")
                                            raise InvalidDescriptorError("Two services are trying to get map to the same port")
                                else:
                                    log.error("Asked port type %s is not listed in networks"%ser_port_type)
                                    raise InvalidDescriptorError("Asked port type %s is not listed in networks"%ser_port_type)
                    if not is_inf_present:
                        log.error("Interface %s asked is not present in network section %s"%ser_interface)
                        raise InvalidDescriptorError("Interface %s asked is not present in network section %s"%ser_interface)

    def extract_docker_rootfs(self, src_dir, dest_path, metadata, is_docker_image):
        pass
