'''

@author: rgowrimo

Copyright (c) 2017-2018 by Cisco Systems, Inc.
All rights reserved.
'''


from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *
from ..utils.utils import Utils

import logging
import tempfile
import os
import falcon
import json
import re
import tarfile

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()


@ResourceRoute("/swupdate/images", endpoint="images")
class SwUpdateResource(AuthenticatedResource):

    def on_post(self, request, response):
        '''
        Accepts a image archive and deploys it
        '''

        update_manager = APIService.instance.update_manager
        swupdate = update_manager.getInstance()
        if not swupdate:
            make_error_response(response, "SwUpdate service is unavailable", "SwUpdate Service is unavailable", falcon.HTTP_503)
            return
        image = None
        imageLocation = request.get_header('X-Image-Location')
        if imageLocation is not None:
            imageLocation = imageLocation.strip()
            if len(imageLocation) == 0:
                imageLocation = None

        filePath = None
        tmpUploadDir = '/tmp'
        if APIService.instance.config.has_option("controller", "upload_dir"):
            tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")

        if not os.path.exists(tmpUploadDir):
            os.makedirs(tmpUploadDir)

        # Whether to cleanup the source filepath after install or in case of errors
        is_cleanup = True
        # Read upload location from system config, default to /tmp if not specified
        if imageLocation:
            log.debug("Header X-Image-Location is present: %s", imageLocation)
            log.debug("Skipping parsing the request body..")
            filePath = imageLocation
            log.debug("Will install the app from %s", filePath)
            is_cleanup = Utils.getSystemConfigValue('controller', 'delete_after_local_install', False, 'bool')
        else:
            # Do not assume file to be zip file, it can be .zip or .ova
            fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
            with os.fdopen(fd, "wb") as f:
                while True:
                    chunk = request.stream.read(4096)
                    if not chunk:
                        break
                    f.write(chunk)

            f.close()

            log.debug("Downloaded the file at:%s" % filePath)

        try:
            image_id = swupdate.add(filePath)
        except Exception as ex:
            log.exception("Error in image installation: (%s)", str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error during image installation",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during image installation: %s" % str(ex),
                                                "Error during image installation",
                                                falcon.HTTP_500)
            return
        finally:
            os.remove(filePath) if os.path.exists(filePath) else None

        imageURL = request.uri + "/" + image_id

        # Post a deployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'deployed' event on %s" % str(image_id))
            event_message = "Image: %s deployed using: %s" % (str(image_id), CAFEvent.SOURCE_RESTAPI)

            ns.post_event(CAFEvent(image_id,
                                   CAFEvent.TYPE_SWUPDATE_DEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message=event_message))

        response.status = falcon.HTTP_201
        response.body = "Successfully deployed image"
        response.set_header("Content-Location", imageURL)

    def on_get(self, request, response):
        update_manager = APIService.instance.update_manager
        swupdate = update_manager.getInstance()
        if not swupdate:
            make_error_response(response, "SwUpdate service is unavailable", "SwUpdate Service is unavailable", falcon.HTTP_503)
            return
        swupdate = update_manager.getInstance()
        out = jsonencoder.encode(swupdate.list())
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


    def on_delete(self, request, response):
        update_manager = APIService.instance.update_manager
        swupdate = update_manager.getInstance()
        if not swupdate:
            make_error_response(response, "SwUpdate service is unavailable", "SwUpdate Service is unavailable", falcon.HTTP_503)
            return
        try:
            swupdate.delete_all_images()
        except Exception as ex:
            log.exception("Error in image deletion: (%s)", str(ex))
            response = make_error_response(response,
                                           "Error removing image : %s" % str(ex),
                                           "Error removing image : %s" % str(ex),
                                           falcon.HTTP_500)
            return

        response.status = falcon.HTTP_200
        response.body = "All Images successfully deleted"
        return


@ResourceRoute("/swupdate/images/{image_id}", endpoint="image")
class SwUpdateResource(AuthenticatedResource):

    def on_get(self, request, response, image_id):
        update_manager = APIService.instance.update_manager
        swupdate = update_manager.getInstance()
        if not swupdate:
            make_error_response(response, "SwUpdate service is unavailable", "SwUpdate Service is unavailable", falcon.HTTP_503)
            return
        image = swupdate.get(image_id)
        if image is None:
            response = make_error_response(response,
                                            "The image: %s, does not exist" %
                                            image_id,
                                            "The image: %s, does not exist" %
                                            image_id,
                                            falcon.HTTP_404)
            return

        response.status = falcon.HTTP_200
        ser_obj = image.serialize()
        log.debug("Serialized Object: %s" % str(ser_obj))
        response.body = jsonencoder.encode(ser_obj)
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_post(self, request, response, image_id):
        update_manager = APIService.instance.update_manager
        swupdate = update_manager.getInstance()
        if not swupdate:
            make_error_response(response, "SwUpdate service is unavailable", "SwUpdate Service is unavailable", falcon.HTTP_503)
            return
        action = request.get_param("action")
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        event_type = None
        try:
            if action == "activate":
                swupdate.activate_image(image_id)
                event_type = CAFEvent.TYPE_SWUPDATE_ACTIVATED
                event_message = "Image: %s activated using: %s" % (image_id,  CAFEvent.SOURCE_RESTAPI)
                response = make_response(response, "Image Activated, initiate reload to update the image", falcon.HTTP_200)
                log.info("App: %s activated" % image_id)
            elif action == "deactivate":
                swupdate.deactivate_image(image_id)
                event_type = CAFEvent.TYPE_SWUPDATE_DEACTIVATED
                event_message = "App: %s deactivated using: %s" %(image_id,  CAFEvent.SOURCE_RESTAPI)
                response = make_response(response, "Image Deactivated\n", falcon.HTTP_200)
                log.info("Image: %s deactivated" % image_id)
            else:
                log.error("Invalid post request")
                make_error_response(response, "Invalid post request", falcon.HTTP_500)
                return
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error while changing image state",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                    "Error while changing image state: %s" % str(ex),
                                                    "Error while changing image state",
                                                    falcon.HTTP_500)
            return

        # If notification service is available, post the event now..
        if ns and event_type:
            log.debug("Posting '%s' event on %s" % (event_type, str(image_id)))
            ns.post_event(CAFEvent(image_id,
                                   event_type,
                                   CAFEvent.SOURCE_RESTAPI, event_message=event_message))


    def on_delete(self, request, response, image_id):
        '''
        delete the image
        '''
        update_manager = APIService.instance.update_manager
        swupdate = update_manager.getInstance()
        if not swupdate:
            make_error_response(response, "SwUpdate service is unavailable", "SwUpdate Service is unavailable", falcon.HTTP_503)
            return
        exists = swupdate.exists(image_id)
        if not exists:
            response = make_error_response(response,
                                           "The image : %s, does not exist" % image_id,
                                           "The image : %s, does not exist" % image_id,
                                           falcon.HTTP_404)
            return
        try:
            swupdate.remove(image_id)
        except Exception as ex:
            log.exception("Error in image deletion: (%s)", str(ex))
            response = make_error_response(response,
                                           "Error removing image : %s" % str(ex),
                                           "Error removing image : %s" % image_id,
                                           falcon.HTTP_500)
            return

        # Post undeployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")

        if ns:
            log.debug("Posting 'undeployed' event on %s" % str(image_id))
            event_message = "Image: %s deleted using: %s" %(image_id,  CAFEvent.SOURCE_RESTAPI)
            ns.post_event(CAFEvent(image_id,
                                   CAFEvent.TYPE_SWUPDATE_DELETED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message = event_message))

        response.status = falcon.HTTP_200
        response.body = "Image successfully deleted"
        return
