#!/bin/sh
# Will setup the network for Given container

create_dir(){
    directory=$1
    if [ -d "$directory" ]; then
	    echo "Given dir $directory is already there"
    else
	    echo "Given dir $directory is not found, so creating one"
	    mkdir -p $directory
    fi
}

dhclient_config(){
config_file=$1
client_id=$2
CLIENT_PARAM=""
CLIENT_PARAM="send dhcp-client-identifier \"$client_id\";"
printf "
option rfc3442-classless-static-routes code 121 = array of unsigned integer 8;
send host-name = gethostname();
%s
request subnet-mask, broadcast-address, time-offset, routers,
	domain-name, domain-name-servers, domain-search, host-name,
	dhcp6.name-servers, dhcp6.domain-search,
	netbios-name-servers, netbios-scope, interface-mtu,
	rfc3442-classless-static-routes, ntp-servers,
	dhcp6.fqdn, dhcp6.sntp-servers;" "$CLIENT_PARAM" > $config_file
}

HELP(){
    echo 'This script sets up networking for docker style applications'
    echo 'Will be executed at the time of app startup'
    echo 'Usage: ./docker_networksetup.sh -d $DOMID -p $PIDREPO -i $INTERFACE -r $RESOLV_CONF -s $SCRIPT_PATH'
    echo '-d: domain id of the conatiner'
    echo '-f: IP family for which you want to run the DHCP client'
    echo '-p: PID repo to store the DHCP client daemon pids'
    echo '-i: Interface to which ip needs to be assigned'
    echo '-r: DNS resolve file path. Should be the location w.r.t the apps rootfs'
    echo '-s: DHCP client - custom script path'
    echo '-c: Provide DHCP client identifier'
    echo 'Note: Before running this script, make sure you have network name space created in host for the container'
    echo 'To create network namespace of the container, run this command: ln -s /proc/{CON_INIT_PID}/ns/net /var/run/netns/{DOM_ID}'
    exit 1
}
echo "Running the docker network up script"
# Making sure process is running with root user
if [ "$(id -u)" != "0" ]; then
        echo "This script must be run as root" 1>&2
        exit 1
fi
pointer=1
while [ $pointer -le $# ]; do
   #param=$1
   if [ "`expr \"$1\" : \"- .*\"`" = "0" ];then
      case $1 in
         -d | --domid)
                shift
                DOMAIN_ID=$1
                ;;
         -p | --pidrepo)
                shift
                PID_REPO=$1
                ;;
         -i | --interface)
                shift
                INTERFACE=$1
                ;;
         -r | --resolvconf)
                shift
                R_CONF=$1
                ;;
         -s | --script)
                shift
                SCRIPT_PATH=$1
                ;;
         -c | --clientid)
                shift
                CLIENT_ID=$1
                ;;
         -f | --ipfamily)
                shift
                IP_FAMILY=$1
                ;;
         -m | --macaddress)
                shift
                MAC=$1
                ;;
         -pid | --processid)
                shift
                PID=$1
                ;;
         -l | --seclabel)
                shift
                SEC_LABEL=$1
                ;;
         -h|--help) HELP ;;
         *)
            echo "Invalid option: $1"
            exit 1 ;;
      esac
   else
      echo "Invalid parameters passed: check --help option"
      exit 1
   fi
   shift
done
if [ "x$DOMAIN_ID" = "x" ] || [ "x$PID_REPO" = "x" ] || [ "x$INTERFACE" = "x" ] || [ "x$R_CONF" = "x" ] || [ "x$IP_FAMILY" = "x" ] || [ "x$PID" = "x" ];then
	echo "Mandatory Arguments: DomainId:$DOMAIN_ID, PidRepo:$PID_REPO ,Intf:$INTERFACE, ResolveConf:$R_CONF, IP family:$IP_FAMILY , PID:$PID are missing"
	exit 1
fi

# Check for the udhcpc binary is exists
if which dhclient >/dev/null; then
    echo "dhclient binary exists"
else
    echo "dhclient binary not found!"
    exit 1
fi

# If security label is provided, use it to run the process
if [ "x$SEC_LABEL" != "x" ];then
    echo $SEC_LABEL > /proc/self/attr/current
fi

# Check and create pid repo
create_dir $PID_REPO

DHCLIENT_CONFIG_FILE="$PID_REPO/dhcp_$INTERFACE.conf"

# Releasing the existing lease on the interface
#ip netns exec $DOMAIN_ID dhclient -r $INTERFACE
if [ "x$CLIENT_ID" = "x" ]; then
        CLIENT_ID="1:0:$MAC"
fi

if [ "x$IP_FAMILY" = "xipv6" ];then
    DHCLIENT_CONFIG_FILE="$PID_REPO/dhcpipv6_$INTERFACE.conf"
fi
# Create dhclient config file with client id in it
dhclient_config $DHCLIENT_CONFIG_FILE $CLIENT_ID

# dhclient to run for ipv4 or ipv6
fam_param=-4
family=ipv4
if [ "x$IP_FAMILY" = "x" ];then
echo "No ipfamily parameter provided so, assuming defaults"
else
case $IP_FAMILY in
     ipv4)
            fam_param=-4
            family=ipv4
            ;;
     ipv6)
            fam_param=-6
            family=ipv6
            ;;
     *)
        echo "Invalid option for ip family: $IP_FAMILY"
        exit 1 ;;
esac
fi
LEASE=.lease
attempts=30
if [ "x$IP_FAMILY" = "xipv6" ];then
for attempt in $(seq 1 $attempts); do
	echo $attempt
	lladdress=$(ip netns exec $DOMAIN_ID ip -6 -o a s dev "$INTERFACE" scope link -tentative)
	if [ -n "$lladdress" ]; then
	    echo $lladdress
   		attempt=0
   		break
  	fi
	if [ ! -d "/proc/$PID" ]; then
	    echo "App got crashed, before CAF can setup the networking!"
   		exit 3
  	fi
	sleep 1
done
fi
# Entering the container network space and running dhclient on a interface and trying to get ip for it.
if [ "x$SCRIPT_PATH" = "x" ];then
    ip netns exec $DOMAIN_ID dhclient -v $fam_param -nw -e RESOLVCONF=$R_CONF -lf $PID_REPO/$INTERFACE$family$LEASE -pf $PID_REPO/$INTERFACE$family -cf $DHCLIENT_CONFIG_FILE $INTERFACE
    net_return=$?
else
    ip netns exec $DOMAIN_ID dhclient -v $fam_param -nw -e RESOLVCONF=$R_CONF -lf $PID_REPO/$INTERFACE$family$LEASE -pf $PID_REPO/$INTERFACE$family -sf $SCRIPT_PATH -cf $DHCLIENT_CONFIG_FILE $INTERFACE
    net_return=$?
fi
if [ $net_return != 0 ]; then
	echo "Issue while running dhclient. So exiting. Retuen code: $net_return"
	if [ ! -d "/proc/$PID" ]; then
        echo "App got crashed, before CAF can setup the networking!"
        exit 3
    fi
	exit $net_return
fi
exit 0
